// ---------------------------------------------------------------------------
// - Directory.cpp                                                           -
// - afnix:sio module - directory class implementation                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Interp.hpp"
#include "System.hpp"
#include "Vector.hpp"
#include "Directory.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "csio.hpp"
#include "cdir.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // open the default directory

  Directory::Directory (void) {
    // update directory handle
    char* dnam = c_getcwd ();
    d_name     = dnam;
    p_handle   = c_opendir (dnam);
    // clean result
    delete [] dnam;
    if (p_handle == nilp)
      throw Exception ("dir-error", "cannot open directory", d_name);
  }

  // open a directory by name

  Directory::Directory (const String& name) {
    // update directory handle
    d_name     = name;
    char* dnam = name.tochar ();
    p_handle   = c_opendir (dnam);
    // clean result
    delete [] dnam;
    if (p_handle == nilp) 
      throw Exception ("dir-error", "cannot open directory", name);
  }

  // destroy this directory 

  Directory::~Directory (void) {
    c_closedir (p_handle);
  }

  // return the class name

  String Directory::repr (void) const {
    return "Directory";
  }

  // return the directory name
    
  String Directory::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // return a list of this directory names

  List* Directory::getlist (void) {
    rdlock ();
    try {
      List* result = new List;
      while (true) {
	char* name = c_readdir (p_handle);
	if (name == nilp) break;
	result->insert (new String (name));
	delete [] name;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a list of this directory files

  List* Directory::getfiles (void) {
    rdlock ();
    try {
      List* result = new List;
      while (true) {
	char* name = c_readdir (p_handle);
	if (name == nilp) break;
	if (c_isfile (name) == true) result->insert (new String (name));
	delete [] name;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a list of the sub directories

  List* Directory::getdirs (void) {
    rdlock ();
    try {
      List* result = new List;
      while (true) {
	char* name = c_readdir (p_handle);
	if (name == nilp) break;
	if (c_isdir (name) == true) result->insert (new String (name));
	delete [] name;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // create a new directory by name

  void Directory::mkdir (const String& name) {
    wrlock ();
    try {
      // join the name
      String path = System::join (d_name, name);
      char*  buf  = path.tochar ();
      // try to create the directory
      if (c_mkdir (buf) == false) {
	delete [] buf;
	throw Exception ("directory-error", "cannot create directory", path);
      }
      delete [] buf;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // remove an empty directory by name

  void Directory::rmdir (const String& name) {
    wrlock ();
    try {
      String path = System::join (d_name, name);
      if (System::rmdir (path) == false) {
	throw Exception ("directory-error", "cannot remove directory", path);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // remove a file in a directory

  void Directory::rmfile (const String& name) {
    wrlock ();
    try {
      String path = System::join (d_name, name);
      if (System::rmfile (path) == false) {
	throw Exception ("directory-error", "cannot remove file", path);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 7;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_MKDIR      = zone.intern ("mkdir");
  static const long QUARK_RMDIR      = zone.intern ("rmdir");
  static const long QUARK_RMFILE     = zone.intern ("rmdir");
  static const long QUARK_GETNAME    = zone.intern ("get-name");
  static const long QUARK_GETLIST    = zone.intern ("get-list");
  static const long QUARK_GETFILES   = zone.intern ("get-files");
  static const long QUARK_GETSUBDIRS = zone.intern ("get-subdirs");

  // create a new object in a generic way

  Object* Directory::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 arguments
    if (argc == 0) return new Directory;
    // check for 1 argument
    if (argc == 1) {
      String val = argv->getstring (0);
      return new Directory (val);
    }
    throw Exception ("argument-error",
		     "too many argument with directory open");
  }

  // return true if the given quark is defined

  bool Directory::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Directory::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
 
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETNAME)    return new String (d_name);
      if (quark == QUARK_GETLIST)    return getlist  ();
      if (quark == QUARK_GETFILES)   return getfiles ();
      if (quark == QUARK_GETSUBDIRS) return getdirs  ();
    }

    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_MKDIR) {
	String val = argv->getstring (0);
	mkdir (val);
	return new Directory (System::join (d_name, val));
      }
      if (quark == QUARK_RMDIR) {
	String val = argv->getstring (0);
	rmdir (val);
	return nilp;
      }
      if (quark == QUARK_RMFILE) {
	String val = argv->getstring (0);
	rmdir (val);
	return nilp;
      }
    }
    // call the object method 
    return Object::apply (robj, nset, quark, argv);
  }
}
