// ---------------------------------------------------------------------------
// - Logtee.cpp                                                              -
// - afnix:sio module - message logging tee class implementation             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Logtee.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {
  
  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------
  
  // create a default logtee 

  Logtee::Logtee (void) {
    p_os   = nilp;
    d_teef = true;
  }

  // create a logtee by size

  Logtee::Logtee (const long size) : Logger (size) {
    p_os   = nilp;
    d_teef = true;
  }

  // create a logtee with an output stream

  Logtee::Logtee (Output*os) {
    Object::iref (p_os = os);
    d_teef = true;
  }

  // create a logtee by size with an output stream

  Logtee::Logtee (const long size, Output*os) : Logger (size) {
    Object::iref (p_os = os);
    d_teef = true;
  }

  // create a logtee by size, info and with an output stream

  Logtee::Logtee (const long size, const String& info, Output*os) : 
          Logger (size, info) {
    Object::iref (p_os = os);
    d_teef = true;
  }
  
  // destroy this logtee

  Logtee::~Logtee (void) {
    Object::dref (p_os);
  }

  // return the class name

  String Logtee::repr (void) const {
    return "Logtee";
  }

  // set the logtee output stream

  void Logtee::setos (Output* os) {
    wrlock ();
    Object::iref (os);
    Object::dref (p_os);
    p_os = os;
    unlock ();
  }

  // return the logtee output stream

  Output* Logtee::getos (void) const {
    rdlock ();
    Output* result = p_os;
    unlock ();
    return result;
  }

  // set the tee flag

  void Logtee::setteef (const bool teef) {
    wrlock ();
    d_teef = teef;
    unlock ();
  }

  // return the tee flag

  bool Logtee::getteef (void) const {
    rdlock ();
    bool result = d_teef;
    unlock ();
    return result;
  }

  // add a message in the logger by log level

  void Logtee::add (const String& mesg, const long mlvl) {
    wrlock ();
    try {
      // add the message
      Logger::add (mesg, mlvl);
      // format the message
      if ((p_os != nilp) && (d_teef == true) && (mlvl <= d_rlvl)) {
	if (d_info.isnil () == true) {
	  *p_os << mesg << eolc;
	} else { 
	  *p_os << d_info << " : " << mesg << eolc;
	}
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------
  
  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETOS = zone.intern ("set-output-stream");
  static const long QUARK_GETOS = zone.intern ("get-output-stream");
  static const long QUARK_SETTF = zone.intern ("set-tee");
  static const long QUARK_GETTF = zone.intern ("get-tee");

  // create a new object in a generic way

  Object* Logtee::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check 0 argument
    if (argc == 0) return new Logtee;
    // check 1 argument
    if (argc == 1) {
      Object* obj = argv->get (0);
      // check for an integer
      Integer* iobj = dynamic_cast <Integer*> (obj);
      if (iobj != nilp) {
	long size = iobj->tointeger ();
	return new Logtee (size);
      }
      // check for an output stream
      Output* os = dynamic_cast <Output*> (obj);
      if (os != nilp) {
	return new Logtee (os);
      }
      throw Exception ("type-error", "invalid object with logtee",
		       Object::repr (obj));
    }
    // check 2 arguments
    if (argc == 2) {
      long  size = argv->getint (0);
      Object* obj = argv->get (1);
      Output* os = dynamic_cast <Output*> (obj);
      if (os == nilp) {
      throw Exception ("type-error", "invalid object with logtee",
		       Object::repr (obj));
      }
      return new Logtee (size, os);
    }
    // check 3 arguments
    if (argc == 3) {
      long   size = argv->getint    (0);
      String info = argv->getstring (1);
      Object* obj = argv->get (2);
      Output* os = dynamic_cast <Output*> (obj);
      if (os == nilp) {
      throw Exception ("type-error", "invalid object with logtee",
		       Object::repr (obj));
      }
      return new Logtee (size, info, os);
    }
    throw Exception ("argument-error", "too many argument for logtee");
  }

  // return true if the given quark is defined
  
  bool Logtee::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Logger::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

    // apply this object class with a set of arguments and a quark
  
  Object* Logtee::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETTF) return new Boolean (getteef ());
      if (quark == QUARK_GETOS) {
	rdlock ();
	try {
	  Object* result = getos ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETTF) {
	bool teef = argv->getbool (0);
	setteef (teef);
	return nilp;
      }
      if (quark == QUARK_SETOS) {
	Object* obj = argv->get (0);
	Output* os = dynamic_cast <Output*> (obj);
	if (os == nilp) {
	  throw Exception ("type-error", "invalid object with setos",
			   Object::repr (obj));
	}
	setos (os);
	return nilp;
      }
    }
    // apply these arguments with the logger
    return Logger::apply (robj, nset, quark, argv);
  }
}
