// ---------------------------------------------------------------------------
// - Sort.cpp                                                                -
// - afnix:txt module - array sorting implementation                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Sort.hpp"
#include "Sorter.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace afnix {

  // this function compare an object with another one in ascent mode
  static bool qsort_cmplth (Object* ref, Object* slv) {
    if (ref == nilp) return true;
    Object*   obj = ref->oper (Object::LTH, slv);
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    bool result   = (bobj == nilp) ? false : bobj->toboolean ();
    Object::cref (obj);
    return result;
  }

  // this function compare an object with another one in descent mode
  static bool qsort_cmpgth (Object* ref, Object* slv) {
    if ((ref == nilp) && (slv == nilp)) return true;
    if ((ref == nilp) && (slv != nilp)) return false;
    Object*   obj = ref->oper (Object::GTH, slv);
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    bool result   = (bobj == nilp) ? false : bobj->toboolean ();
    Object::cref (obj);
    return result;
  }
 
 // this function compare an object with another one in lexicographic mode
  static bool qsort_cmplex (Object* ref, Object* slv) {
    if ((ref == nilp) && (slv == nilp)) return true;
    if ((ref == nilp) && (slv != nilp)) return false;
    // try to get a literal object
    Literal* rlit = dynamic_cast <Literal*> (ref);
    if (rlit == nilp) 
      throw Exception ("type-error", "cannot map object to literal");
    Literal* slit = dynamic_cast <Literal*> (slv);
    if (slit == nilp) 
      throw Exception ("type-error", "cannot map object to literal");
    // map it to string and compare
    String rstr = rlit->tostring ();
    String sstr = slit->tostring ();
    return rstr < sstr;
  }

  // quick sort ascending function

  Object* txt_qsort_lth (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) throw Exception ("argument-error", 
				    "invalid arguments with sort-ascent");
    // get the argument vector
    Vector* vobj = dynamic_cast <Vector*> (argv->get (0));
    if (vobj == nilp) throw Exception ("type-error", 
				       "invalid object with sort-ascent");
    // build the sorter object
    Sorter sorter (qsort_cmplth);
    // sort the vector
    try {
      sorter.qsort (vobj);
      delete argv;
      return nilp;
    } catch (...) {
      delete argv;
      throw;
    }
  }

  // quick sort ascending function

  Object* txt_qsort_gth (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) throw Exception ("argument-error", 
				    "invalid arguments with sort-descent");
    // get the argument vector
    Vector* vobj = dynamic_cast <Vector*> (argv->get (0));
    if (vobj == nilp) throw Exception ("type-error", 
				       "invalid object with sort-descent");
    // build the sorter object
    Sorter sorter (qsort_cmpgth);
    // sort the vector
    try {
      sorter.qsort (vobj);
      delete argv;
      return nilp;
    } catch (...) {
      delete argv;
      throw;
    }
  }

  // quick sort lexicographic function

  Object* txt_qsort_lex (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) throw Exception ("argument-error", 
				    "invalid arguments with sort-ascent");
    // get the argument vector
    Vector* vobj = dynamic_cast <Vector*> (argv->get (0));
    if (vobj == nilp) throw Exception ("type-error", 
				       "invalid object with sort-ascent");
    // build the sorter object
    Sorter sorter (qsort_cmplex);
    // sort the vector
    try {
      sorter.qsort (vobj);
      delete argv;
      return nilp;
    } catch (...) {
      delete argv;
      throw;
    }
  }
}
