// ---------------------------------------------------------------------------
// - Session.cpp                                                             -
// - afnix:wam module - http session class implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Sha1.hpp"
#include "Wamsid.hxx"
#include "Vector.hpp"
#include "System.hpp"
#include "Session.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // this procedure returns a new session for deserialization
  static Serial* mksob (void) {
    return new Session;
  }
  // register this session serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_SESS_ID, mksob);

  // this procedure compute the session id with the help of an hasher object
  // the SHA-1 hasher object is used by operating on the session name
  static String get_session_id (const String& name) {
    // create the SHA-1 object
    Sha1 hasher;
    // get the result
    return hasher.compute (name);
  }

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty session

  Session::Session (void) {
    d_path = System::tempnam ();
  }

  // create a session by name

  Session::Session (const String& name) {
    d_name = name;
    d_id   = get_session_id (d_name);
    d_path = System::tempnam ();
  }

  // create a session  by name and path

  Session::Session (const String& name, const String& path) {
    d_name = name;
    d_id   = get_session_id (d_name);
    d_path = path;
  }

  // return the class name

  String Session::repr (void) const {
    return "Session";
  }

  // return the session serial code

  t_byte Session::serialid (void) const {
    return SERIAL_SESS_ID;
  }

  // serialize this session

  void Session::wrstream (Output& os) const {
    rdlock ();
    try {
      d_name.wrstream (os);
      d_id.wrstream   (os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this cell

  void Session::rdstream (Input& is) {
    wrlock ();
    d_name.rdstream (is);
    d_id.rdstream   (is);
    unlock ();
  }

  // get the session name

  String Session::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // get the session id

  String Session::getid (void) const {
    rdlock ();
    String result = d_id;
    unlock ();
    return result;
  }

  // set the session path
  
  void Session::setpath (const String& path) {
    wrlock ();
    d_path = path;
    unlock ();
  }

  // get the session path

  String Session::getpath (void) const {
    rdlock ();
    String result = d_path;
    unlock ();
    return result;
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GETID   = zone.intern ("get-id");
  static const long QUARK_GETNAME = zone.intern ("get-name");
  static const long QUARK_GETPATH = zone.intern ("get-path");
  static const long QUARK_SETPATH = zone.intern ("set-path");

  // create a new object in a generic way

  Object* Session::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 1 argument
    if (argc == 1) {
      String name  = argv->getstring (0);
      return new Session (name);
    }
    // check for 2 arguments
    if (argc == 2) {
      String name = argv->getstring (0);
      String path = argv->getstring (1);
      return new Session (name, path);
    }
    // invalid arguments
    throw Exception ("argument-error", "invalid arguments with session");
  }

  // return true if the given quark is defined

  bool Session::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* Session::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETID)   return new String (getid   ());
      if (quark == QUARK_GETNAME) return new String (getname ());
      if (quark == QUARK_GETPATH) return new String (getpath ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETPATH) {
	String path = argv->getstring (0);
	setpath (path);
	return nilp;
      }      
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
