// ---------------------------------------------------------------------------
// - InputMapped.hpp                                                         -
// - standard object library - input mapped file class definition            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_INPUTMAPPED_HPP
#define  AFNIX_INPUTMAPPED_HPP

#ifndef  AFNIX_INPUT_HPP
#include "Input.hpp"
#endif

#ifndef  AFNIX_NAMEABLE_HPP
#include "Nameable.hpp"
#endif

namespace afnix {

  /// The InputMapped class is derived from the Input base base class and 
  /// provide a facility for reading mapped file. The file is mapped at 
  /// construction given a size and a file offset.
  /// @author amaury darsch

  class InputMapped : public Input, public Nameable {
  private:
    /// the file name
    String d_name;
    /// the mapped buffer
    char* p_mbuf;
    /// the buffer marker
    t_long d_mark;
    /// the buffer size
    t_long d_size;
    /// the file offset
    t_long d_foff;

  public:
    /// create a new mapped file by name
    /// @param name the file name
    InputMapped (const String& name);

    /// create a new mapped file by name, size and offset
    /// @param name the file name
    /// @param size the size to map
    /// @param offt the file offset
    InputMapped (const String& name, const t_long size, const t_long offt);

    /// destroy this mapped file
    ~InputMapped (void);

    /// @return the class name
    String repr (void) const;

    /// @return the next available character
    char read (void);

    /// @return true if we are at the eof
    bool iseof (void) const;

    /// check if we can read a character
    /// @param tout the timeout value
    bool valid (const long tout) const;

    /// place the marker at a certain position
    /// @param pos the position to go
    void lseek (const t_long pos);

    /// @return the file name
    String getname (void) const;

    /// @return the input map size
    t_long length (void) const;

    /// @return the file offset
    t_long foffset (void) const;

  private:
    // make the copy constructor private
    InputMapped (const InputMapped&);
    // make the assignment operator private
    InputMapped& operator = (const InputMapped&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
    
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
