// ---------------------------------------------------------------------------
// - HashTable.hpp                                                           -
// - standard object library - hash table object class definition            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_HASHTABLE_HPP
#define  AFNIX_HASHTABLE_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The HashTable class is another container class which maps an object
  /// with a key. The hash table is dynamic and get resized automatically
  /// when needed. When an object is added, the object reference count is
  /// increased. When the object is retreived, the reference count is not
  /// touched. The lookup method throw an exception if the key is not found.
  /// The get method returns nilp if the object is not found.
  /// @author amaury darsch

  class HashTable : public virtual Object {
  private:
    /// the hash table size
    long d_size;
    /// the hash table length
    long d_hlen;
    /// the hash table threshold
    long d_thrs;
    /// array of buckets
    struct s_bucket** p_htbl;

  public:
    /// create a hash table with a default size
    HashTable (void);
  
    /// create a hash table with an initial size
    HashTable (const long size);
  
    /// destroy the hash table. 
    ~HashTable (void);

    /// return the class name
    String repr (void) const;

    /// reset the hash table
    void reset (void);

    /// @return the number of elements
    long length (void) const;

    /// @return true if the table is empty
    bool empty (void) const;

    /// @return the element key by index
    String getkey (const long index) const;

    /// @return the element object by index
    Object* getobj (const long index) const;
  
    /// @return true if the key exists in this table
    bool exists (const String& key) const;

    /// set or create a new object in the table.
    /// @param key the object key to set
    /// @param obj the object to set or add
    void add (const String& key, Object* obj);

    /// get an object by key. If the object is not found nil is returned.
    /// this means that this function always succeds. You can use lookup 
    /// instead to get an exception if the key is not found. You can also use
    /// the exists method to check for a key but exists and get is very slow.
    /// @param key the object key to search
    Object* get (const String& key) const;

    /// find an object by key or throw an exception
    /// @param key the object key to find
    Object* lookup (const String& key) const;

    /// remove an object in the hash table. 
    /// @param key the object key to remove
    void remove (const String& key);

    /// @return a vector of objects
    Vector* getvobj (void) const;

    /// resize this hash table
    void resize (const long size);

  private:
    // make the copy constructor private
    HashTable (const HashTable&);
    // make the assignment operator private
    HashTable& operator = (const HashTable&);    

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
