/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifndef __GCS_TYPES_H
#define __GCS_TYPES_H

#include <glibmm/ustring.h>
#include <boost/shared_ptr.hpp>
#include "gcs-checked-int.h"

/** The gcs namespace includes both the colorscheme library and the
 * GNOME colorscheme application */
namespace gcs
{

    /** A Type to hold levels for individual color components such as red or blue */
    typedef float tColorByte;

    const gushort maxGdkColorValue = 65535;
    /** The highest value that can be assigned to an RGB colorByte */
    const gint maxRgbValue = 255;
    /** The highest value that can be assigned to a Hue colorByte */
    const gint maxHueValue = 360;
    /** The highest value that can be assigned to a Saturation or Value
     * colorByte */
    const gint maxSvValue = 100;
    /** The lowest value that can be assigned to a colorByte */
    const gint minColorValue = 0;


    /** A type to hold a hue specification.
     *
     * The value of this type can range from 0 to 360, and will wrap around
     * when the value would exceed either limit since it is meant to represent
     * the degrees on a color wheel
     * */
    typedef CheckedInt<minColorValue, maxHueValue, ModularArithmetic> hue_t;
    /** A type to hold a Saturation or Value specification
     *
     * This value can range from 0 to 100 and will throw an out-of-range
     * exception if it exceeds these limits
     * */
    typedef CheckedInt<minColorValue, maxSvValue, SaturatedArithmetic> sv_t;
    /** A type to hold a Red, Green, or Blue specification
     *
     * This value can range from 0  to 255 and will throw an out-of-range
     * exception if it exceeds these limits
     * */
    typedef CheckedInt<minColorValue, maxRgbValue, SaturatedArithmetic> rgb_t;

    /** A type to hold HSV color data
     *
     * Contains data members to hold hue, saturation, and value data separately
     */
    class tColorHsv
    {
        public:
            tColorHsv(const int h=0, const int s=0, const int v=0);
            tColorHsv(const tColorHsv& hsv);
            hue_t hue;
            sv_t saturation;
            sv_t value;
    };

    /** A type to hold RGB color data
     * Contains data members to hold red, green, and blue data separately */
    class tColorRgb
    {
        public:
            tColorRgb(const int r=0, const int g=0, const int b=0);
            tColorRgb(const tColorRgb& rgb);
            rgb_t red;
            rgb_t green;
            rgb_t blue;
    };

    /** Prints out RGB colors in a nice format for debugging */
    std::ostream& operator<<(std::ostream& out, tColorRgb color);

    /** Prints out HSV colors in a nice format for debugging */
    std::ostream& operator<<(std::ostream& out, tColorHsv color);

    /** Compare tColorRgb values */
    bool operator==(tColorRgb first, tColorRgb last);
    /** Compare tColorHsv values */
    bool operator==(tColorHsv first, tColorHsv last);


    /** A type to hold the Hex string representation of a color.
     * The normalized form contains 6 characters consisting of the numbers 0-9
     * and the letters A-F. 
     *
     * The first two characters define the levels for the red component, the
     * second two characters define the Green level, and the last two define
     * the blue level.
     */
    typedef Glib::ustring tHexString;

    /** smart pointer for holding Color objects
    */
    class Color;
    typedef boost::shared_ptr<Color> ColorPtr;

} // namespace gcs

#endif // __GCS_TYPES_H
