/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include "config.h"
#include <cstdlib>  // for srand(), rand()
#include <ctime>    // for time()
#include <cassert>  // for assert() macro

#include <boost/shared_ptr.hpp>

#include <gtkmm/icontheme.h>
#include <gdkmm/pixbuf.h>
#include <gdkmm/color.h>
#include <gtkmm/toolbar.h>
#include <gtk/gtkwindow.h>

#ifdef HAVE_GCONFMM
#include <gconfmm/client.h>
#endif // HAVE_GCONFMM

#include "gcs-mainwindow.h"
#include "gcs-debug.h"
#include "gcs-util.h"   // for get_dropped_color
#include "gcs-i18n.h"
#include "gcs-history.h"
#include "gcs-conf.h"
#include "core/gcs-scheme.h"
#include "widgets/gcs-schemebox.h"
#include "widgets/gcs-schemeselector.h"
#include "widgets/gcs-bookmarklist.h"
#include "widgets/gcs-colorswatch.h"
#include "widgets/gcs-paletteview.h"
#include "dialogs/gcs-about-window.h"

#define NO_SIZE_REQUEST (-1)

namespace gcs
{
    MainWindow* MainWindow::m_instance = NULL;
    const gint MainWindow::WINDOW_HEIGHT = 400;
    const gint MainWindow::WINDOW_WIDTH = 650;
    const gint MainWindow::FAVORITES_WIDTH = 130;

    MainWindow& MainWindow::Instance(void)
    {
        if (m_instance)
        {
            return *m_instance;
        }
        else
        {
            m_instance = new MainWindow();
            return *m_instance;
        }
    }


    MainWindow::MainWindow(void)
        : m_pLayout(Gtk::manage(new Gtk::VBox())),
        m_pInnerLayout(Gtk::manage(new Gtk::HBox())),
        m_pScrollBox(Gtk::manage(new Gtk::ScrolledWindow())),
        m_pSchemeColumn(Gtk::manage(new Gtk::VBox())),
        m_pBookmarkColumn(Gtk::manage(new Gtk::VBox())),
        m_pPane(Gtk::manage(new Gtk::HPaned())),
        m_pControls(Gtk::manage(new Gtk::HBox())),
        m_pSchemeBox(Gtk::manage(new Widgets::SchemeBox())),
        // FIXME: build this path in a more portable way
        m_PaletteView(Gtk::manage(new Widgets::PaletteView(
                        AGAVE_PALETTEDIR "/Web.gpl"))),
        m_pStatusbar(Gtk::manage(new Gtk::Statusbar())),
        m_pHistory(new HistoryNavigation<tHexString>()),
        m_pMainMenu(0),  // will be assigned in the constructor
        // FIXME: build this path in a more portable way
        m_bookmarks_file(Glib::get_home_dir() + "/" + Conf::FAVORITES_FILE)
    {
        m_pBookmarkList = Gtk::manage(new Widgets::BookmarkList(m_bookmarks_file));
        init_actions();
        init_ui();

        // seed the random number generator for generating random color schemes
        srand(time(NULL));


        /* The menu bar across the top of the window */
        m_pMainMenu = static_cast<Gtk::Menu *>(
                m_refUIManager->get_widget("/MainMenu"));
        assert(m_pLayout);
        assert(m_pMainMenu);
        m_pLayout->pack_start(*m_pMainMenu, Gtk::PACK_SHRINK);
        debug("Added main menu");

        m_pTweakBar = static_cast<Gtk::Toolbar *>(
                m_refUIManager->get_widget("/TweakBar"));
        assert(m_pTweakBar);
        m_pTweakBar->set_toolbar_style(Gtk::TOOLBAR_BOTH_HORIZ);
        m_pLayout->pack_start(*m_pTweakBar, Gtk::PACK_SHRINK, 0);

        assert(m_pPane);
        m_pLayout->pack_start(*m_pPane);

        assert(m_pInnerLayout);
        assert(m_pSchemeColumn);
        m_pInnerLayout->pack_start(*m_pSchemeColumn);

        assert(m_pPane);
        m_pPane->pack1(*m_pInnerLayout);

        assert(m_pBookmarkColumn);
        m_pPane->pack2(*m_pBookmarkColumn, Gtk::SHRINK);

        // Set up the SchemeBox
        assert(m_pSchemeBox);
        Gtk::Menu * pSwatchPopup = static_cast<Gtk::Menu *>(
                m_refUIManager->get_widget("/ColorSwatchPopup"));
        debug("popup widget: ", pSwatchPopup);
        m_pSchemeColumn->pack_start(*m_pSchemeBox, Gtk::PACK_EXPAND_WIDGET, 0);
        m_pSchemeBox->signal_color_selected().connect(
                sigc::mem_fun(*this, &MainWindow::on_schemebox_color_selected));

        // set up the main controls
        assert(m_pControls);
        m_pSchemeColumn->pack_start(*m_pControls, Gtk::PACK_SHRINK);
        debug("Added Controls box");

        m_pColorButton = Gtk::manage(new Gtk::ColorButton);
        assert(m_pColorButton);
        m_pColorButton->signal_color_set().connect(sigc::mem_fun(*this,
                    &MainWindow::on_color_changed));
        debug("created Color Button");
        m_pSchemeSelector = Gtk::manage(new Widgets::SchemeSelector);
        assert(m_pSchemeSelector);
        m_pSchemeSelector->signal_changed().connect(sigc::mem_fun(*this,
                    &MainWindow::on_schemetype_changed));
        debug("Created Scheme Selector");
        m_pControls->pack_start(*m_pColorButton, Gtk::PACK_SHRINK);
        m_pControls->pack_start(*m_pSchemeSelector, Gtk::PACK_EXPAND_WIDGET);
        m_pControls->set_spacing(Conf::UI_SPACING_SMALL);
        m_pControls->set_border_width(Conf::UI_SPACING_SMALL);

        assert(m_PaletteView);
        m_pSchemeColumn->pack_start(*m_PaletteView, Gtk::PACK_SHRINK, 0);
        m_PaletteView->set_border_width(Conf::UI_SPACING_SMALL);
        m_PaletteView->signal_color_selected().connect(
            sigc::mem_fun(*this, &MainWindow::on_schemebox_color_selected));
        m_PaletteView->set_expanded();


        assert(m_pBookmarkList);
        m_pBookmarkList->set_size_request(FAVORITES_WIDTH, NO_SIZE_REQUEST);
        m_pScrollBox->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
        m_pBookmarkColumn->pack_start(*m_pScrollBox);
        m_pScrollBox->add(*m_pBookmarkList);
        m_pBookmarkList->get_selection()->signal_changed().connect(
                sigc::mem_fun(*this, &MainWindow::on_bookmarks_selection_changed));
        
        m_pBookmarkBar = dynamic_cast<Gtk::Toolbar *>(m_refUIManager->get_widget("/BookmarkBar"));
        assert(m_pBookmarkBar);
        m_pBookmarkColumn->pack_start(*m_pBookmarkBar, Gtk::PACK_SHRINK);
        m_pBookmarkBar->set_toolbar_style(Gtk::TOOLBAR_BOTH_HORIZ);

        assert(m_pStatusbar);
        m_pStatusbar->push(_("Choose a Color and a Scheme Type"), 1);
        m_pLayout->pack_start(*m_pStatusbar, Gtk::PACK_SHRINK);

        m_pAbout = new Dialogs::AboutWindow();
        assert(m_pAbout);

        add(*m_pLayout);
        m_pLayout->show_all();
        set_position(Gtk::WIN_POS_CENTER);

        //Glib::RefPtr<Gtk::IconTheme> theme = Gtk::IconTheme::get_default();
        //theme->append_search_path(DATADIR "/icons");

        // need to use a GTK+ function since it's not available in gtkmm yet
        gtk_window_set_default_icon_name("agave-icon");

        set_default_size(Conf::get_window_width(), Conf::get_window_height());
        set_title(PACKAGE_NAME);

        std::list<Gtk::TargetEntry> listTargets;
        listTargets.push_back(Gtk::TargetEntry("application/x-color"));
        m_pSchemeBox->drag_dest_set(listTargets);
        m_pSchemeBox->signal_drag_data_received().connect(sigc::mem_fun(*this,
                    &MainWindow::on_drop_drag_data_received));
    }

    MainWindow::~MainWindow(void)
    {
        delete m_pAbout;
        //delete m_pHistory;
        debug("MainWindow DELETED!");
    }


    // Actions must be initialized first
    void MainWindow::init_ui(void)
    {
        m_refUIManager = Gtk::UIManager::create();
        m_refUIManager->insert_action_group(m_refActionGroup);
        add_accel_group(m_refUIManager->get_accel_group());

        try
        {
            m_refUIManager->add_ui_from_file(AGAVE_UIDIR "/agave.ui");
            debug("added UI");
        }
        catch(const Glib::Error& ex)
        {
            // we can't do anything without the UI / toolbar definition.
            std::cerr << __FILE__ << ": " << ex.what() << std::endl;
            throw ex;
        }

    }


    void MainWindow::set_color(ColorPtr c)
    {
        m_pColorButton->set_color(c->gdk());
        on_color_changed();
    }


    void MainWindow::on_show(void)
    {
        Gtk::Window::on_show();
        m_pSchemeSelector->set_scheme_type(Conf::get_last_scheme_type());
        // hack to get the schemetype selector to select the right starting
        // scheme
        on_schemetype_changed();
        update_bookmark_actions();

        set_color(Conf::get_last_color());
    }


    bool MainWindow::on_delete_event(GdkEventAny* event)
    {
        quit();
    }

    void MainWindow::quit(void)
    {
        hide();
        ColorPtr clr = m_pSchemeBox->get_color();
        Conf::set_last_color(clr);
        Conf::set_last_scheme_type(m_pSchemeBox->get_scheme_type());
        // only save the window size if the window isn't maximized
        if (!(get_window()->get_state() & Gdk::WINDOW_STATE_MAXIMIZED))
        {
            Conf::set_window_width(get_width());
            Conf::set_window_height(get_height());
        }
    }


    void MainWindow::on_color_changed(void)
    {
        ColorPtr clr = Color::create(m_pColorButton->get_color());
        m_pSchemeBox->set_color(clr);
        m_pHistory->add(clr->get_hexstring());
        //debug(*m_pHistory);
        m_refActionGroup->get_action("HistoryBack")->set_sensitive(m_pHistory->has_back());
        m_refActionGroup->get_action("HistoryFwd")->set_sensitive(m_pHistory->has_forward());

        // check if we're at limits
        Glib::RefPtr<Gtk::Action> action =
            m_refActionGroup->get_action("LightenScheme");
        if (clr->get_value() == maxSvValue)
        {
            // disable lighten button
            action->set_sensitive(false);
        }
        else
        {
            action->set_sensitive();
        }

        action = m_refActionGroup->get_action("DarkenScheme");
        if (clr->get_value() <= minColorValue + 5)
        {
            // disable darken button
            action->set_sensitive(false);
        }
        else
        {
            action->set_sensitive();
        }

        action = m_refActionGroup->get_action("SaturateScheme");
        if (clr->get_saturation() == maxSvValue)
        {
            // disable saturate button
            action->set_sensitive(false);
        }
        else
        {
            action->set_sensitive();
        }

        action = m_refActionGroup->get_action("DesaturateScheme");
        if (clr->get_saturation() <= minColorValue + 5)
        {
            // disable desaturate button
            action->set_sensitive(false);
        }
        else
        {
            action->set_sensitive();
        }
        debug("Color was changed!");
        ColorPtr ptr = m_pBookmarkList->get_color();
        if (ptr && (*ptr != *clr))
        {
            m_pBookmarkList->get_selection()->unselect_all();
        }
    }


    void MainWindow::on_schemetype_changed(void)
    {
        tSchemeType type = m_pSchemeSelector->get_scheme_type();
        m_pSchemeBox->set_scheme_type(type);

        debug("Scheme type is ", type);
    }


    void MainWindow::on_menu_file_popup_generic(void)
    {
        std::cout << "mainwindow popup handler" << std::endl;
    }


    void MainWindow::on_bookmarks_selection_changed(void)
    {
        debug("Bookmarks changed");
        ColorPtr pClr = m_pBookmarkList->get_color();
        if (pClr)
        {
            set_color(pClr);
        }
        update_bookmark_actions();
    }


    void MainWindow::on_schemebox_color_selected(ColorPtr pColor)
    {
        // Need to make a copy of the color that we're passed so that if we
        // change the color of the swatch it doesn't change the color in the
        // palette / favorite list as well
        ColorPtr c = Color::create(*pColor);
        set_color(c);
    }


    void MainWindow::on_drop_drag_data_received(const
            Glib::RefPtr<Gdk::DragContext>& context, int x, int y,
            const Gtk::SelectionData& selection_data, guint info,
            guint time)
    {
        debug("== Drop received ==");
        bool drag_success = false;
        boost::shared_ptr<Gdk::Color> c = get_dropped_color(selection_data);
        if(c)
        {
            // create a gcs::Color from the Gdk::Color
            ColorPtr pClr = Color::create(*c);

            // set the application's current color
            set_color(pClr);
            drag_success = true;
        }

        context->drag_finish(drag_success, false, time);
    }

} // namespace gcs
