#include <Foundation/Foundation.h>
#include "CalendarManager.h"
#include "CalendarUser.h"
#include "CalendarEvent.h"

static CalendarManager *server;
static CalendarDomain *_domain;
static CalendarUser *_user;
static NSString * userName;

void print_help(char *cmd);

void show_user_info(NSArray *cmd);
void show_user_event(NSArray *cmd);
void set_user(NSArray *cmd);
void set_info(NSArray *cmd);
void set_group(NSArray *cmd);
void set_event(NSArray *cmd);
void remove_group(NSArray *cmd);

int main(int argc, char **argv, char **env)
{
	NSAutoreleasePool *pool;
	id proxy;
	pool = [NSAutoreleasePool new];
	NSString *domainName;

	domainName = @"localhost";

	[NSProcessInfo initializeWithArguments:argv count:argc environment:env];


	proxy = [NSConnection rootProxyForConnectionWithRegisteredName:@"Agendum"
															  host:nil];
	if (proxy == nil)
	{
		fprintf(stderr, "Cannot connect to server\n");
		exit(1);
	}

	NSMutableArray *args = [NSMutableArray arrayWithArray:[[NSProcessInfo processInfo] arguments]];
	NSString *password = nil;
	int i;

	userName = NSUserName();

	[args removeObjectAtIndex:0];

	for (i = 0; i < [args count]; i++)
	{
		if ([[args objectAtIndex:i] isEqualToString:@"--help"])
		{
			print_help(argv[0]);
			exit(0);
		}

		if ([[args objectAtIndex:i] isEqualToString:@"-a"])
		{
			ASSIGN(userName, @"");
			[args removeObjectAtIndex:i];
			i--;
		}
		else if ([[args objectAtIndex:i] isEqualToString:@"-d"])
		{
			ASSIGN(domainName, [args objectAtIndex:i+1]);
			[args removeObjectAtIndex:i];
			[args removeObjectAtIndex:i];
			i--;
		}
		else if ([[args objectAtIndex:i] isEqualToString:@"-u"])
		{
			ASSIGN(userName, [args objectAtIndex:i+1]);
			[args removeObjectAtIndex:i];
			[args removeObjectAtIndex:i];
			i--;
		}
		else if ([[args objectAtIndex:i] isEqualToString:@"-p"])
		{
			ASSIGN(password, [args objectAtIndex:i+1]);
			[args removeObjectAtIndex:i];
			[args removeObjectAtIndex:i];
			i--;
		}
	}

	if (password == nil)
	{
		password = @"";
	}

	if ([args count] == 0)
	{
		fprintf(stderr, "%s: missing command\n",argv[0]);
		fprintf(stderr, "Try `%s --help' for more information.\n",argv[0]);
		exit(0);
	}

	server = [proxy calendarManagerForDomainName:domainName
						authenticateWithUserName:userName
										password:password];

	if (server == nil || [server respondsToSelector:@selector(count)])
	{
		fprintf(stderr, "Cannot connect to server (%s)\n",[[server objectAtIndex:1] lossyCString]);
		exit(0);
	}

	/* setup local user */
	_user = [CalendarUser calendarUserWithName:userName
									  password:password
									  timeZone:nil
									  userInfo:nil];

	_domain = [[CalendarDomain alloc] initWithDomainName:domainName];

	[_domain setCalendarUser: _user];


	if ([[args objectAtIndex:0] isEqualToString:@"userinfo"])
	{
		[args removeObjectAtIndex:0];
		show_user_info(args);
	}
	else if ([[args objectAtIndex:0] isEqualToString:@"eventinfo"])
	{
		[args removeObjectAtIndex:0];
		show_user_event(args);
	}
	else if ([[args objectAtIndex:0] isEqualToString:@"setuser"])
	{
		[args removeObjectAtIndex:0];
		set_user(args);
	}
	else if ([[args objectAtIndex:0] isEqualToString:@"setinfo"])
	{
		[args removeObjectAtIndex:0];
		set_info(args);
	}
	else if ([[args objectAtIndex:0] isEqualToString:@"setgroup"])
	{
		[args removeObjectAtIndex:0];
		set_group(args);
	}
	else if ([[args objectAtIndex:0] isEqualToString:@"setevent"])
	{
		[args removeObjectAtIndex:0];
		set_event(args);
	}
	else if ([[args objectAtIndex:0] isEqualToString:@"removegroup"])
	{
		[args removeObjectAtIndex:0];
		remove_group(args);
	}
	else
	{
		fprintf(stderr, "command `%s' is not recognized\n", [[args objectAtIndex:0] lossyCString]);
		exit(1);
	}

	printf("\n");

	exit(0);
}

void remove_group(NSArray *cmd)
{
	NSEnumerator *en;
	NSString *gname;
	id <CalendarUser> user = [server calendarUserWithName:userName];

	en = [cmd objectEnumerator];
	while (gname = [en nextObject])
	{
		[user removeCalendarGroupWithName:gname];
	}
}

void set_group(NSArray *cmd)
{
	NSEnumerator *en;
	NSString *gname;
	id <CalendarUser> user = [server calendarUserWithName:userName];

	en = [cmd objectEnumerator];
	while (gname = [en nextObject])
	{
		[user calendarGroupWithName:gname];
	}
}

void set_event(NSArray *cmd)
{
	CalendarUser *newEvent;
	NSMutableDictionary *dict = [NSMutableDictionary dictionary];

	NSEnumerator *en;
	NSString *str;
	NSRange range;

	en = [cmd objectEnumerator];

	while (str = [en nextObject])
	{
		range = [str rangeOfString:@":"];
		if (range.length > 0)
		{
			[dict setObject:[str substringFromIndex:range.location+1]
					 forKey:[[str substringToIndex:range.location] lowercaseString]];
		}
	}

	NSString *name;

	name = [dict objectForKey:@"name"];

	if (name == nil)
	{
		name = [NSString
			stringWithFormat:@"agendum-client:%@",[[NSCalendarDate date] description]];
	}

	newEvent = [CalendarEvent calendarEventWithName:name
									 calendarDomain:_domain
									   calendarUser:_user
									  calendarGroup:[_user calendarGroupWithName:[dict objectForKey:@"group"]]
										  startDate:[NSDate dateWithString:[dict objectForKey:@"startdate"]]
										   stopDate:[NSDate dateWithString:[dict objectForKey:@"stopdate"]]
									  summaryString:[dict objectForKey:@"summary"]];

	[server setCalendarEvent:newEvent];

}

void set_user(NSArray *cmd)
{
	CalendarUser *newUser;
	NSMutableDictionary *dict = [NSMutableDictionary dictionary];

	NSEnumerator *en;
	NSString *str;
	NSRange range;
	en = [cmd objectEnumerator];

	while (str = [en nextObject])
	{
		range = [str rangeOfString:@":"];
		if (range.length > 0)
		{
			[dict setObject:[str substringFromIndex:range.location+1]
					 forKey:[[str substringToIndex:range.location] lowercaseString]];
		}
	}

	NSString *name,*password,*timezone;
	NSTimeZone *timeZone;

	name = [dict objectForKey:@"login"];
	password = [dict objectForKey:@"password"];
	if ([dict objectForKey:@"timezone"] != nil)
	{
		timeZone = [NSTimeZone
			timeZoneWithName:[dict objectForKey:@"timezone"]];
	}
	else
	{
		timeZone = nil;
	}

	if (name == nil)
	{
		name = userName;
	}

	[dict removeObjectForKey:@"login"];
	[dict removeObjectForKey:@"password"];
	[dict removeObjectForKey:@"timezone"];

	newUser = [CalendarUser calendarUserWithName:name
										password:password
										timeZone:timeZone
										userInfo:dict];

	[server setCalendarUser:newUser];

}

void set_info(NSArray *cmd)
{
	CalendarUser *newUser;

	NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithDictionary:[[server calendarUserWithName:userName] userInfo]];

	NSEnumerator *en;
	NSString *str;
	NSRange range;
	en = [cmd objectEnumerator];

	while (str = [en nextObject])
	{
		range = [str rangeOfString:@":"];
		if (range.length > 0)
		{
			[dict setObject:[str substringFromIndex:range.location+1]
					 forKey:[[str substringToIndex:range.location] lowercaseString]];
		}
	}

	NSString *name,*password,*timezone;
	NSTimeZone *timeZone;

	name = [dict objectForKey:@"login"];
	password = [dict objectForKey:@"password"];
	if ([dict objectForKey:@"timezone"] != nil)
	{
		timeZone = [NSTimeZone
			timeZoneWithName:[dict objectForKey:@"timezone"]];
	}
	else
	{
		timeZone = nil;
	}

	if (name == nil)
	{
		name = userName;
	}

	[dict removeObjectForKey:@"login"];
	[dict removeObjectForKey:@"password"];
	[dict removeObjectForKey:@"timezone"];

	newUser = [CalendarUser calendarUserWithName:name
										password:password
										timeZone:timeZone
										userInfo:dict];

	[server setCalendarUser:newUser];

}

void show_user_info(NSArray *cmd)
{
	int i;
	id <CalendarUser> user;
	id <CalendarGroup> group;

	if ([cmd count] == 0)
	{
		user = [server calendarUserWithName:userName];
	}
	else
	{
		user = [server calendarUserWithName:[cmd objectAtIndex:0]];
	}

	if (user == nil)
	{
		fprintf(stderr, "Cannot get information for user '%s'\n", [[cmd objectAtIndex:0] lossyCString]);
		exit(1);
	}

	printf("Information:\n");
	NSDictionary *dict;
	NSEnumerator *en;
	NSString *key;
	printf("\tLogin = '%s'\n", [[user name] lossyCString]);

	key = [user password];
	if (key == nil)
	{
		printf("\tPassword = N/A\n");
	}
	else
	{
		printf("\tPassword = '%s'\n", key == nil?"nil":[key lossyCString]);
	}

	dict = [user userInfo];

	printf("\tTimeZone = '%s'\n", [[[user timeZone] description] lossyCString]);
	dict = [user userInfo];

	printf("\n");

	en = [dict keyEnumerator];

	while (key = [en nextObject])
	{
		printf("\t%s = '%s'\n",[key lossyCString],[[dict objectForKey:key] lossyCString]);
	}

	printf("\n");

	printf("Group:\n");
	en = [[user calendarGroups] objectEnumerator];
	for (i = 1;group = [en nextObject];i++)
	{
		printf("\t%d) '%s'\n", i, [[group name] lossyCString]);
	}

}

void show_user_event(NSArray *cmd)
{
	int i;
	id <CalendarUser> user;
	id <CalendarGroup> group;
	id <CalendarEvent> ev;

	if ([cmd count] == 0)
	{
		user = [server calendarUserWithName:userName];
	}
	else
	{
		user = [server calendarUserWithName:[cmd objectAtIndex:0]];
	}

	if (user == nil)
	{
		fprintf(stderr, "Cannot get information for user '%s'\n", [[user name] lossyCString]);
		exit(1);
	}

	NSArray *array;
	NSEnumerator *en;
	NSTimeZone *tz;
	printf("Events for '%s' in domain '%s':\n", [[user name] lossyCString],
			/* FIXME description -> name*/
			[[_domain description] lossyCString]);

	array = [user calendarEvents];

	tz = AUTORELEASE([[NSTimeZone alloc] initWithName:[[user timeZone] timeZoneName]]);

	en = [array objectEnumerator];

	for (i=1;ev = [en nextObject];i++)
	{
		printf("%d) '%s'\nCalendar:%s\nStart:%s\nStop:%s\nSummary:%s\n\n",i,
				[[ev name] lossyCString],
				[[[ev calendarGroup] name] lossyCString],
				[[[ev startDate] description] lossyCString],
				[[[ev stopDate] description] lossyCString],
				[[ev summaryString] lossyCString]
				);
	}

	printf("\n");

}

void print_help(char *cmd)
{
	printf("Usage: %s [OPTION]... COMMAND [COMMAND OPTION]...\n",cmd);
	printf("Agendum Server Manager\n");
	printf("\nOption:\n");
	printf("-a        login as admin\n");
	printf("-d        use domain name\n");
	printf("-u        use login name\n");
	printf("-p        use password\n");
	printf("\nCommand:\n");
	printf("userinfo [login]\n\n");
	printf("eventinfo <login>\n\n");
	printf("setuser [login:<login>] [password:<password>] [timezone:<timezone>] [<info key>:<info value>]\n\n");
	printf("setevent [name:<event name>] [startdate:<date>] [stopdate:<date>] [summary:<summary string>]\n\n");
	printf("setgroup [group1] [group2] ...\n\n");
	printf("removegroup [group1] [group2] ...\n\n");
	printf("Example:\n\n");
	printf("%s -a -p admin123 setuser login:kate password:abc123 mail:kate@example.com address:\"111 Bangkok Thailand\" www:http://www.gnu.org/ timezone:GMT+7\n",cmd);
	printf("%s -u john -p h4xm3 setevent startdate:2004-5-3 stopdate:\"2004-5-4 10:10:10\" summary:\"Take a bath\" \n",cmd);
	printf("\n");
}
