/**********************************************************************
 ** Syslog class: Handles the syslog, for storing regular messages like
 **               player logins, logoffs, the more suspect commands
 **               being run, etc
 **
 ** Last reviewed: 
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef SYSLOG_C
#define SYSLOG_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "logs.h"
#include "syslog.h"
#include "utils.h"
#include "timespan.h"

/***********************************************************************
 ** SysLog (constructor) - Opens the logfile by a specified name
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

SysLog::SysLog()
{
	valid = 0;
   if (open_log(SYSLOGNAME) <= 0)
   {
      printf("Error - could not open the syslog\n");
      return;
   }
   fprintf(the_log, "&+W<%s> &+b** Mud Starting up **&*\n", get_time_str());
   fflush(the_log);
   valid = 1;
}


/***********************************************************************
 ** ~SysLog (destructor)  - empty (closing actions located in Logs)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

SysLog::~SysLog()
{
      /* If the logfile is not there, return */ 
   if (the_log == NULL)
      return;

   fprintf(the_log, "&+W<%s> &+b** Mud Closing down **&*\n", get_time_str());
   fflush(the_log);

}


/***********************************************************************
 ** log_event - logs an event in the syslog log
 **
 ** Parameters: event_str - the event string to print
 **
 ** Returns:  1 if success
 **          -1 if error file is not open
 **
 ***********************************************************************/

int SysLog::log_event(char *event_str)
{
   Timespan the_time;

   /* If the logfile is not there, return */ 
   if (the_log == NULL)
      return -1;

   fprintf(the_log, "&+W<%s>&* %s \n", get_time_str(), event_str);
   fflush(the_log);
   return 1;
}


/***********************************************************************
 ** archive_syslog - archives the system log and opens a new one
 **
 ** Parameters: None
 **
 ** Returns:  1 if success
 **          -1 if error file is not open
 **
 ***********************************************************************/

int SysLog::archive_syslog()
{
   int results; 

   fprintf(the_log, "&+W<%s>&+Y ** Archiving SysLog **&*\n", get_time_str());
   fflush(the_log);
   
   if ((results = archive_log(SYSLOGNAME)) != -1)
   {
      if (open_log(SYSLOGNAME) <= 0)
      {
         printf("Error - could not open the syslog\n");
	 valid = 0;
         return -1;
      }
      fprintf(the_log, "&+W<%s>&+Y ** Syslog Reopened after Archive **&*\n", 
                                                          get_time_str());
      fflush(the_log);
      valid = 1;
   }
   return results;
}


#endif

