;; AIscm - Guile extension for numerical arrays and tensors.
;; Copyright (C) 2013, 2014, 2015, 2016, 2017 Jan Wedekind <jan@wedesoft.de>
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;
(use-modules (system foreign)
             (oop goops)
             (srfi srfi-26)
             (aiscm obj)
             (aiscm element)
             (aiscm int)
             (aiscm float)
             (aiscm bool)
             (aiscm sequence)
             (aiscm jit)
             (aiscm asm)
             (aiscm util)
             (guile-tap))
(define obj (make <obj> #:value 'sym))
(define address (scm->address 'sym))
(define ctx (make <context>))

(ok (eqv? 8 (size-of <obj>))
    "size of SCM reference is 64 bits")
(ok (equal? #vu8(#xaa #x00 #x00 #x00 #x00 #x00 #x00 #x00) (pack (make <obj> #:value 42)))
    "pack Scheme object")
(ok (equal? (make <obj> #:value 42) (unpack <obj> #vu8(#xaa #x00 #x00 #x00 #x00 #x00 #x00 #x00)))
    "unpack Scheme object")
(ok (eq? <obj> (coerce <obj> <obj>))
    "objects coerce to objects")
(ok (eq? <obj> (coerce <obj> <int>))
    "object and integer coerce to object")
(ok (eq? <obj> (coerce <int> <obj>))
    "integer and object coerce to object")
(ok (equal? "#<<obj> abc>"
            (call-with-output-string (lambda (port) (write (make <obj> #:value 'abc) port))))
    "write wrapped object")
(ok (eq? <obj> (native-type 'a))
    "native type for a symbol is <obj>")
(ok (equal? 'sym (build <obj> (list 'sym)))
    "build SCM value")
(ok (equal? (list address) (unbuild <obj> 'sym))
    "content of symbol returns internal 64 bit representation")
(ok (eq? -300 ((jit ctx (list <obj>) (cut make-native-function obj-negate <>)) 300))
    "compile and run call to scm_difference with one argument")
(ok (eq? -124 ((jit ctx (list <obj>) (cut make-native-function scm-lognot <>)) 123))
    "compile and run bitwise not")
(ok (equal? '(#f #t) (map (jit ctx (list <obj>) (cut make-native-function obj-zero-p <>)) '(3 0)))
    "compile and run comparison with zero")
(ok (equal? '(#t #f) (map (jit ctx (list <obj>) (cut make-native-function obj-nonzero-p <>)) '(3 0)))
    "compile and run not-equal-to zero")
(ok (equal? '(#t #f #f #f #f) (map (jit ctx (list <obj>) (cut make-native-function obj-not <>)) '(#f #t () 0 1)))
    "compile logical not for Scheme objects")
(ok (eq? 300 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-sum <...>)) 100 200))
    "compile and run call to scm_sum")
(ok (eq? 100 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-difference <...>)) 300 200))
    "compile and run call to scm_difference")
(ok (eq? 600 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-product <...>)) 20 30))
    "compile and run call to scm_product")
(ok (eq? 30 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-divide <...>)) 600 20))
    "compile and run call to scm_divide")
(ok (eq? 33 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-remainder <...>)) 123 45))
    "compile and run call to scm_remainder")
(ok (eq? 72 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-logand <...>)) 123 456))
    "compile and run call to scm_logand")
(ok (eq? 507 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-logior <...>)) 123 456))
    "compile and run call to scm_logior")
(ok (eq? 435 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-logxor <...>)) 123 456))
    "compile and run call to scm_logxor")
(ok (equal? '(#f b) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-and <...>)) '(#f a) '(b b)))
    "compile logical and for Scheme objects")
(ok (equal? '(b a) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-or <...>)) '(#f a) '(b b)))
    "compile logical or for Scheme objects")
(ok (eq? 123 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-min <...>)) 123 456))
    "compile and run call to scm_min")
(ok (eq? 456 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-max <...>)) 123 456))
    "compile and run call to scm_max")
(ok (eq? 1968 ((jit ctx (list <obj> <obj>) (cut make-native-function scm-ash <...>)) 123 4))
    "compile and run call to scm_ash")
(ok (eq? 123 ((jit ctx (list <obj> <obj>) (cut make-native-function obj-shr <...>)) 1968 4))
    "compile and run call to obj_shr")
(ok (equal? '(#f #t) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-equal-p <...>)) '(21 42) '(42 42)))
    "compile and run equality of objects")
(ok (equal? '(#t #f) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-nequal-p <...>)) '(21 42) '(42 42)))
    "compile and run inequality of objects")
(ok (equal? '(#t #f #f) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-less-p <...>)) '(3 5 7) '(5 5 5)))
    "compile and run lower-than comparison for objects")
(ok (equal? '(#t #t #f) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-leq-p <...>)) '(3 5 7) '(5 5 5)))
    "compile and run lower-equal comparison for objects")
(ok (equal? '(#f #f #t) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-gr-p <...>)) '(3 5 7) '(5 5 5)))
    "compile and run greater-than comparison for objects")
(ok (equal? '(#f #t #t) (map (jit ctx (list <obj> <obj>) (cut make-native-function obj-geq-p <...>)) '(3 5 7) '(5 5 5)))
    "compile and run greater-equal comparison for objects")
(ok (not (pointerless? <obj>))
    "object meory is not pointerless")
(ok (not (signed? <obj>))
    "object references don't need sign-extension")
(ok (equal? '(#f #t) (map (jit ctx (list <bool>) (cut make-native-function obj-from-bool <>)) '(#f #t)))
    "compile and run call to obj_from_bool")
(ok (equal? '(#f #t) (map (jit ctx (list <obj>) (cut make-native-function scm-to-bool <>)) '(#f #t)))
    "compile and run call to scm_to_bool")
(ok (eqv? 123 ((jit ctx (list <obj>) (cut make-native-function scm-to-uint8 <>)) 123))
    "compile and run call to scm_to_int8")
(ok (eqv? 123 ((jit ctx (list <ubyte>) (cut make-native-function scm-from-uint8 <>)) 123))
    "compile and run call to scm_from_int8")
(ok (eqv? -123 ((jit ctx (list <obj>) (cut make-native-function scm-to-int8 <>)) -123))
    "compile and run call to scm_to_int8")
(ok (eqv? -123 ((jit ctx (list <byte>) (cut make-native-function scm-from-int8 <>)) -123))
    "compile and run call to scm_from_int8")
(ok (eqv? 123 ((jit ctx (list <obj>) (cut make-native-function scm-to-uint16 <>)) 123))
    "compile and run call to scm_to_int16")
(ok (eqv? 123 ((jit ctx (list <usint>) (cut make-native-function scm-from-uint16 <>)) 123))
    "compile and run call to scm_from_int16")
(ok (eqv? -123 ((jit ctx (list <obj>) (cut make-native-function scm-to-int16 <>)) -123))
    "compile and run call to scm_to_int16")
(ok (eqv? -123 ((jit ctx (list <sint>) (cut make-native-function scm-from-int16 <>)) -123))
    "compile and run call to scm_from_int16")
(ok (eqv? 123 ((jit ctx (list <obj>) (cut make-native-function scm-to-uint32 <>)) 123))
    "compile and run call to scm_to_int32")
(ok (eqv? 123 ((jit ctx (list <uint>) (cut make-native-function scm-from-uint32 <>)) 123))
    "compile and run call to scm_from_int32")
(ok (eqv? -123 ((jit ctx (list <obj>) (cut make-native-function scm-to-int32 <>)) -123))
    "compile and run call to scm_to_int32")
(ok (eqv? -123 ((jit ctx (list <int>) (cut make-native-function scm-from-int32 <>)) -123))
    "compile and run call to scm_from_int32")
(ok (eqv? 123 ((jit ctx (list <obj>) (cut make-native-function scm-to-uint64 <>)) 123))
    "compile and run call to scm_to_int64")
(ok (eqv? 123 ((jit ctx (list <ulong>) (cut make-native-function scm-from-uint64 <>)) 123))
    "compile and run call to scm_from_int64")
(ok (eqv? -123 ((jit ctx (list <obj>) (cut make-native-function scm-to-int64 <>)) -123))
    "compile and run call to scm_to_int64")
(ok (eqv? -123 ((jit ctx (list <long>) (cut make-native-function scm-from-int64 <>)) -123))
    "compile and run call to scm_from_int64")
(ok (equal? -42 ((jit ctx (list <long>) (cut make-native-function obj-negate <>)) 42))
    "implicit conversion from long integer to object")
(run-tests)
