# Binding for ICQ protocol library

package require icq 0.8.8

namespace eval meta {
	set name "ICQ"
	set author "Ihar Viarheichyk <iverg@mail.ru>"
	set description "ICQ transport"
	set icon img:online
	array set uin {
		type digit description "ICQ UIN" weight .10
		property Global:ICQ:Info|Uin save change}
	array set alias {property Global:ICQ:Info|Alias save change
		weight .11}
	array set status {
		property Global:ICQ:Network|StartupStatus save change
		description "Status after startup" default offline
		type variant weight .12
		values {offline online occ dnd ffc away na invisible}
	}
	array set password { type password property Global:ICQ:Info|Password
		description "ICQ password" save change weight .101}
	array set encoding [list type variant save change\
		values [map x [encoding names] {set x [string index $x 0]:$x}]\
		default [encoding system] desciption "Default encoding"\
		property Global:ICQ:Info|encoding weight .13]
	unset x	
	array set ChangePassword {type action menu {ICQ "Change Password"}}
}

namespace eval network::meta {
	array set server {
		type fqdn default login.icq.com save change
		property Global:ICQ:Network|server weight .21
	}
	array set port {
		type port default 5190 save change weight .22
		property Global:ICQ:Network|port
	}
	array set proxy {
		type variant save change default "" weight .23
		description "Proxy to use" menu "ICQ Proxy"
		valuescript {concat {""} [map x [select Proxy] {
				lindex [split $x :] end}]}
	}
	array set reconnect {
		type boolean save change property Global:ICQ:Network|reconnect
		description "Automatically re-connect after disconnect"
		default 1 weight .24
	}
	array set ping {
		type integer save change property Global:ICQ:Network|ping
		description "Ping interval, seconds" default 0 weight .25
	}
	array set keepalive {
		type integer save change property Global:ICQ:Network|keepalive
		description "Keepalive interval, seconds" default 0 weight .26
	}
}

namespace eval roster::meta {
	array set use { 
		type boolean default 1 property Global:ICQ:Roster|use
		save change weight .31
		description "Use server-side roster"
	}
	array set modify {
		type boolean default 1 property Global:ICQ:Roster|modify
		save change weight .32
		description "Save Changes in contact list in server-side roster"
	}
	array set updated {
		type cache property Global:ICQ:Roster|updated
		save exit default 0
	}
	array set count {
		type cache property Global:ICQ:Roster|count
		save change default 0
	}
}

# ICQ-specific properties for ICQ contacts
namespace eval [ref Contact:ICQ]::meta {
	array set Status  {
		type {cache variant} default offline 
		values {offline online occ dnd ffc away na invisible}
	}
	array set mobile 	{save change}
	array set auth		{type {cache boolean} default 0 save change}
	array set rid		{type integer save change}
	array set visible	{type boolean default 0 save change
		menu {Lists Visible} weight .31}
	array set invisible	{type boolean default 0 save change 
		menu {Lists Invisible} weight .32}
	array set Send:text	{type action menu {Send Message} weight .10}
	array set Send:contacts	{type action menu {Send Contacts} weight .11}
array set authorization:grant	{type action 
		menu {Authorization Grant} weight .20}
	array set authorization:deny	{type action
		menu {Authorization Deny} weight .21}
	array set encoding	{ menu Encoding }
}

set ref [ref Contact:ICQ]
array set ${ref}::meta::encoding [array get meta::encoding]

# Sync contact default encoding with global encoding
upvar #0  ${ref}::meta::encoding(default) [namespace current]::encoding


namespace eval [ref Group:common]::meta {
	array set rid	{type integer save change}
}

namespace eval ::validate {
	proc port {val} {
		expr { [string is integer $val] && $val>=0 && $val<=65535 }
	}
}

variable icq
variable recreate 0
variable itemCount 0

proc Uid2Uin {list} { map x $list { lindex [split $x :] end } }

# Proceed ICQ events
proc ICQEvent {event args} {
	if {[llength [info commands icq:$event]]} {
		eval icq:$event $args
	} else { eval Event [list $event] $args}
}

# Process contact statuses
proc icq:Status {uin newstatus {ip ""}} {
	set [ref Contact:ICQ:$uin](Status) $newstatus
	if {$ip!=""} { set [ref Contact:ICQ:$uin](IP) $ip }
}

# Process personal status
proc icq:MyStatus {newstatus} {
	if {$newstatus=="offline"} {
		foreach x [select Contact:ICQ] { set [ref $x](Status) offline }
	} 
	set [ref Me](Status) $newstatus
}

# Translate ICQ message to alicq messages
proc icq:Incoming {type sender time message} {
	set id [clock seconds]:[clock clicks] 
	if {$type!="web"} {
		Event Incoming $type Contact:ICQ:$sender $time $message $id
	} else {
		Event Incoming text Contact:ICQ:0 $time "$sender:\n$message" $id
	}
}

proc icq:Registered {uin password} {
	variable recreate 1
	set [namespace current]::uin $uin
	set [namespace current]::password $password
	Event SetStatus online
}

proc icq:RegistrationRefused {} {
	Event Error registration "ICQ server refused registration of new UIN"
}

proc icq:Outgoing {type sender time message msgid} {
	Event Outgoing $type Contact:ICQ:$sender $time $message $msgid
}

proc icq:Client {uin args} {
	set [ref Contact:ICQ:$uin](Client) $args
}

proc icq:ACK {type uin msgid} {
	Event Log 1 "Ack $type from $uin on $msgid"
	Event Contact:ICQ:$uin|Acknowledgement $type Contact:ICQ:$uin $msgid
}

proc icq:Capabilities {uin caps} {
	Event Log 5 "Capabilities $uin: $caps"
}

proc icq:Roster:Items {items time} {
	variable itemCount
	foreach x $items {
		foreach {type id name aux} $x break
		if {!$id && $type=="group"} {
			incr itemCount
			continue
		}
		if {[llength [info commands roster:$type]]} {
			Event Log 8 "roster $type: $id $name $aux"
			if {[catch { roster:$type $id $name $aux } v]} {
				Event Error roster "In roster $type $name: $v"
			}
		} else { incr itemCount }
	}
	if {$time} { 
		set roster::updated $time
		set roster::count $itemCount
		set itemCount 0
		MonitorRoster 
	}
}

proc icq:Roster:OK {args} { MonitorRoster }

proc icq:Roster:Update {status} {
	Event Log 6 "Contact updated with status $status"
	ProcessResult $status
}

# Install server-side roster related monitoring of objects
proc MonitorRoster {args} {
	variable groups
	foreach x [select Group:common] { CommonRosterMonitor $x }
	foreach x [select Contact:ICQ] {
		if {![info exists groups($x)]} { CalculateGroup $x }
		CommonRosterMonitor $x 
		ContactRosterMonitor $x
	}
	hook {New:Contact:ICQ:* New:Group:common:*} [nc CommonRosterMonitor]
	hook {New:Contact:ICQ:*} [nc ContactRosterMonitor]
	# Prevent re-entering
	proc MonitorRoster args return 
}

# Calculate group ID for contact
proc CalculateGroup {uid} {
	variable groups
	if {[info exists groups($uid)]} { unset groups($uid) }
	foreach x [get $uid Groups {}] {
		if {$x=="other"} continue
		set var [ref Group:common:$x](rid)
		if {[info exists $var]} { 
			set groups($uid) [set $var] 
			return
		}
	}
}

# Sent common handlers for contact and group: alias change and delete
proc CommonRosterMonitor {uid} {
	set ref [ref $uid]
	if {![info exists ${ref}(rid)]} { Enqueue add $uid }
	trace variable ${ref}(Alias) w [nc Enqueue update $uid]
	trace variable ${ref}(Groups) u [nc Enqueue delete $uid]
}

# Handle movement from one group to another
proc ContactRosterMonitor {uid} {
	trace variable [ref $uid](Groups) w [nc Move $uid]
}
proc Move {uid args} { foreach x {delete add} {Enqueue $x $uid} }

# Enqueue roster operation
proc Enqueue {action uid args} { 
	if {![string is true $roster::modify]} return
	lappend roster::queue [list $action $uid] 
	Event Log 6 "enqueue roster item $action for $uid"
	ProcessQueue
}

# Process head of roster action queue if it exists and not locked by previous
# operation
proc ProcessQueue {} {
	variable rosterLock
	if {[info exists rosterLock] || [set [ref Me](Status)]=="offline" ||
	    ![llength $roster::queue]} return
	set rosterLock 1
	foreach {action uid} [lindex $roster::queue 0] break
	if {![RosterSync $action $uid]} { ProcessResult -1 }
}

# Handle result of SSI operation, and shedule processing of new item in
# roster actions queue if needed
proc ProcessResult {result} {
	variable rosterLock
	if {![info exists rosterLock] || ![llength $roster::queue]} return
	unset rosterLock
	foreach {action uid} [lindex $roster::queue 0] break
	# Contact is not authorized, try to add it with auth TLV
	if {$result==14} {
		set [ref $uid](auth) 1
	} else {
		if {$result==0} {
			# Successfully added info to SSI, save rid,
			if {$action=="add"} { 
				variable last_rid
				set [ref $uid](rid) $last_rid 
			} elseif {$action=="delete"} {
				unset [ref $uid](rid)
			}
		} elseif {$result>0} {
			Event Error roster "Roster update error $result"
			if {$result==10} {
			}
		}
		set roster::queue [lrange $roster::queue 1 end]
	}
	after idle [nc ProcessQueue]
}

# Construct command to make proper sync of local and remote contact list item 
proc RosterSync {action uid} {
	variable icq
	set rid [Roster:$action $uid]
	foreach {type name aux} [RosterInfo $uid] break
	set list [list $action [list $type $rid $name $aux]]
	Event Log 6 [list $icq roster $action [list $type $rid $name $aux]]
	$icq roster $action [list $type $rid $name $aux]
	return 1
}

proc Roster:add {uid} {
	variable groups
	variable last_rid
	# Whenever we add new contact, group id should be recalculated
	if {[string match Contact:* $uid]} { CalculateGroup $uid }

	# Disable adding contacts which have no group mapping, and
	# "other" group
	if {[string match Contact:* $uid] && ![info exists groups($uid)] ||
	    [string match Group:common:other $uid]} { 
	    	return -code return 0
	}
	# Calculate new unique roster id
	set ridlist [map x [select {Contact:ICQ Group:common}\
		{[info exists rid]}] { set [ref $x](rid)}]
	while 1 {
		set last_rid [expr {int(rand()*32000.0+500)}]
		if {[lsearch $ridlist $last_rid]==-1} { return $last_rid }
	}
}

proc Roster:delete {uid} { 
	set var [ref $uid](rid)
	if {[info exists $var]} { set $var } else { return -code return 0} 
}

proc Roster:update {uid} { 
	set var [ref $uid](rid)
	if {[info exists $var]} { set $var } else { return -code return 0} 
}

# gater information common for all roster update methods
proc RosterInfo {uid} {
	upvar #0 [ref $uid] info
	foreach {type _ name} [split $uid :] break
	set aux [list]
	if {[string match Contact:ICQ:* $uid]} {
		variable groups
		lappend aux group $groups($uid) Alias $info(Alias)
		if {[info exists info(auth)] && $info(auth)} {
			lappend aux auth {}
		}
	} else { set name $info(Alias) }
	list [string tolower $type] $name $aux
}

# Set variable only if it's value will change, to prevent unneeded traces 
# invokation
proc different {var val} {
	if {![info exists $var] || [set $var]!=$val} { set $var $val }
}

proc roster:contact {id name aux} {
	variable groups
	array set Aux $aux
	set uid Contact:ICQ:$name
	set ref [ref $uid]
	set groups($uid) $Aux(group)
	# If groups are not set for the contact yet, try to find proper mapping
	if {![info exists ${ref}(Groups)] || [set ${ref}(Groups)]=="other"} {
	     set g [select Group:common "\[info exists rid\] && \$rid=={$Aux(group)}"]
	     lappend list Groups [expr {($g=="")?"other":[lindex [split $g :] end]}]
	}
	if {[info exists Aux(Alias)]} { lappend list Alias $Aux(Alias) }
	lappend list auth [info exists Aux(auth)] rid $id
	unset Aux
	# If contact already exists, set only fields which differ
	if {[info exists $ref]} { 
		foreach {key val} $list { different ${ref}($key) $val }
	} else { new $uid $list }
}

proc roster:group {id name aux} {
	# Try to map by rid first
	set uid [select Group:common "\[info exists rid\] && \$rid=={$id}"]
	if {$uid!=""} { 
		different [ref $uid](Alias) $name 
		return
	}
	# If no group rid mapping found, try to map by alias
	set uid [select Group "\[info exists Alias\] && \$Alias=={$name}"]
	if {$uid!=""} {
		set [ref $uid](rid) $id
		return
	}
	# If no existsing group found, create new one
	new Group:common:$id [list rid $id Groups [list] Alias $name]
}

# Fake icq command is used when no UIN is specified.
# It always returns empty string as result and sends Error:authorization
# when trying to change status
proc fake_icq {sub args} {
	if {$sub=="status" && [lindex $args 0]!="offline"} {
		Event Error:auth "UIN is not specified or non-numeric"
	}
	list
}

proc EstimatedRoster {} {
	expr { [llength [select {Contact:ICQ Group} {[info exists rid]}]]+
	       ${roster::count} }
}


proc CreateConnection {{new 0} {local_password ""}} {
	foreach x {recreate alias uin password} { variable $x }
	set recreate 0
	if {!$new} {
		if {![info exists uin] || ![info exists password] ||
		    ![string is digit -strict $uin]} {
			return [namespace current]::fake_icq
		} elseif {![info exists alias]} {
			trace variable [ref Me](Status) w [nc GetAlias]
		}
		set local_uin $uin
		set local_password $password
	} else { set local_uin new }
	set icq [icq::icq $local_uin $local_password\
		-capabilities {ICQ RELAY UTF8}\
		-event [namespace current]::ICQEvent]
	foreach x [info vars network::*] {
		$icq configure -[namespace tail $x] [set $x]
	}
	if {$local_uin!="new" && [string is true $roster::use]} {
		if {$roster::updated>0} {
			Event Log 6 "Estimated roster: [EstimatedRoster], $roster::updated"
			set roster [list $roster::updated [EstimatedRoster]]
		} else {set roster 1}
	} else {set roster 0}
	$icq configure -roster $roster
	# Create contact lists
	$icq contacts all add [Uid2Uin [select Contact:ICQ]]
	foreach id {visible invisible} {
		set cmd "\[info exists $id\]&&\[string is true \$$id\]"
		$icq contacts $id add [Uid2Uin [select Contact:ICQ $cmd]]
	}
	# Set contact encodings
	variable encoding
	$icq encoding $encoding
	foreach x [select Contact:ICQ {[info exists encoding]}] {
		$icq encoding [set [ref $x](encoding)] [Uid2Uin $x]
	}
	set icq
}

# Register new ICQ UIN. New icq command created for this purpose
handler ICQ:Register registrator {password} {
	variable reg
	Event ICQ:CancelRegistration
	set reg [CreateConnection 1 $password]
}
# Cancel ICQ UIN registration. Only one regustration command at the time
# allowed, thus no task identifiers required
handler ICQ:CancelRegistration registration_canceler {args} {
	variable reg
	if {[info exists reg]} { 
		$reg delete 
		unset reg
	}
}

handler search Search {info} {
	variable icq
	$icq search $info
}

handler UpdateInfo UserInfo {info} {
	variable icq
	$icq personal $info
}

# Add monitoring of contact properties
proc MonitorContact {uid} {
	set ref [ref $uid]
	foreach x {visible invisible} { 
		trace variable ${ref}($x) w [nc tracklist $x [Uid2Uin $uid]]
	}
	trace variable ${ref}(encoding) w [nc trackencoding [Uid2Uin $uid]]
	trace variable ${ref} u [nc trackall delete $uid]
}

# Track changes in contact list and sync it with icq command
proc trackall {action uid args} {
	if {[lindex $args 1]!=""} return
	variable icq
	$icq contacts all $action [Uid2Uin $uid]
}

# Track changes in visible/invisible lists and sync them with icq command
proc tracklist {list uin ref field args} {
	variable icq
	upvar ${ref}($field) val
	set action [expr {[string is true $val]?"add":"delete"}]
	$icq contacts $list $action $uin
}

# Track changes of contact's encoding
proc trackencoding {uin ref field args} {
	variable icq
	upvar 1 ${ref}($field) val
	$icq encoding $val $uin
}

# Track changes of global encoding
proc GlobalEncoding {ref args} {
	variable icq
	upvar 1 $ref val
	$icq encoding $val
}

proc Reconfigure {name args} {
	variable icq
	$icq configure -$name [set network::$name]
}

proc CredChanged {what name args} {
	upvar 1 $name val
	Event Log 7 "$what changed to $val"
	variable recreate 1
	# Cancel registration of new UIN
	Event ICQ:CancelRegistration
}

handler ConfigLoaded ConfigLoaded {args} {
	foreach x {icq alias status} { variable $x }

	if {[info exists alias]} { set [ref Me](Alias) $alias } else {
		trace variable [ref Me](Alias) w [nc SetAlias]
	}

	foreach x [select Contact:ICQ] { MonitorContact $x }
	hook New:Contact:ICQ:* [nc MonitorContact]
	hook New:Contact:ICQ:* [nc trackall add]

	trace variable [namespace current]::encoding w [nc GlobalEncoding]

	foreach x {uin password} {
		trace variable [namespace current]::$x w [nc CredChanged $x]
	}
	foreach x [info vars network::*] {
		trace variable $x w [nc Reconfigure [namespace tail $x]]
	}

	set icq [CreateConnection]
	Event SetStatus $status 
}

handler Send Send {type uid message} {
	if {![string match Contact:ICQ:* $uid]} { return -code continue }
	variable icq
	if {[string equal $type "SMS"]} {
		set var [ref $uid](mobile)
		if {[info exists $var]} {
			Event Send text Contact:SMS:[set $var] $message
		}
		return -code continue
	}
	$icq send $type [lindex [split $uid :] 2] $message
} 0.90

handler Send SendSMS {type uid message} {
	if {![string match Contact:SMS:* $uid]} {return -code continue}
	variable icq
	$icq send SMS [lindex [split $uid :] 2] $message
} 0.95

handler MessageTypes messagetypes {uid} {
	if {![string match Contact:ICQ:* $uid]} { return -code continue }
	set lst {text URL authrequest}
	if {[info exists [ref $uid](mobile)]} { lappend lst SMS }
	set lst
}

handler SetStatus SetStatus {setstatus} {
	foreach x {icq recreate} { variable $x }
	if {[set [ref Me](Status)]=="offline" && $recreate} {
		if [info exists icq] { $icq delete }
		set icq [CreateConnection]
	}
	$icq status $setstatus
}

handler ChangeICQPassword ChangeICQPassword {password ref} {
	variable icq
	$icq password $password $ref
}

handler InfoRequest InfoRequest {uid ref} {
	variable icq
	if {![string match Contact:ICQ:* $uid]} return
	$icq info [lindex [split $uid :] 2] $ref
}

handler Incoming AddContact {type sender time message args} {
	NewContact $sender
} 0.01

handler AddICQContact AddICQContact {uin groups alias} {
	NewContact Contact:ICQ:$uin
}

proc NewContact {uid} {
	if {[info exists [ref $uid](Groups)] ||
	    ![string match Contact:ICQ:* $uid]} return
	puts "adding $uid"
	set uin [lindex [split $uid :] end]
	new $uid [list Groups other Alias $uin]
	variable icq
	$icq contacts all add $uin
	set ref [expr [clock seconds] & 0xff]
	hook Info [nc NewContactInfo Contact:ICQ:$uin $ref]
	$icq info $uin $ref
}

# Sync icq alias when global alias changes
proc SetAlias {name idx args} {
	upvar 1 ${name}($idx) val
	variable alias $val
}

proc GetAlias {args} {
	foreach x {icq uin} { variable $x }
	if {[set [ref Me](Status)]=="offline"} return
	trace vdelete [ref Me](Status) w [nc GetAlias]
	set ref [expr {[clock seconds] & 0xFF}]
	hook Info [nc NewContactInfo Me $ref]
	$icq info $uin $ref
}

proc NewContactInfo {uid myref ref info} {
	if {$myref!=$ref} return
	array set Info $info
	if {[info exists Info(Nick)]} { set [ref $uid](Alias) $Info(Nick) }
	unhook Info [nc NewContactInfo $uid $ref]
}

