package provide alicq 0.8.8

package require hooks 3.1
package require helper

set compatible 1

# Core modules, provided by alicq itself.
namespace eval modules {
	# Global actions and properties
	namespace eval my::meta {
		array set exit {type action menu Exit weight 0.9}
	}
}
# Proxy servers management
namespace eval modules::Proxy {
	namespace eval meta {
		set description "Alicq core - proxies"
		set author "Ihar Viarheichyk <iverg@mail.ru>"
		set configuration-objects Proxy
	}
	package require proxy 0.3
	handler New:Proxy:* ProxyAdded {uid} {
		set ref [ref $uid]
		set name [lindex [split $uid :] end]
		foreach {key val} [array get $ref] { 
			proxy::proxy $name -$key $val
		}
		trace variable $ref w [nc ProxyChanged $name]
		trace variable $ref u [nc ProxyDeleted $name]
	}
	proc ProxyChanged {name ref field args} {
		upvar 1 ${ref}($field) value
		proxy::proxy $name -$field $value
	}
	proc ProxyDeleted {name ref field args} {
		if {$field!=""} return
		proxy::delete $name
	}
}

# Namespace 'objects' contains all objects which should be managed with 
# alicq helper procedures.
namespace eval objects {

	# Contacts of any type
	namespace eval Contact::meta {
		array set Rename { type action menu Rename weight .90}
		array set Delete {type action menu Delete weight .99}
		array set Alias {default "" save change}
		array set Groups {type groups default {} save change}
	}
	# Groups of any type
	namespace eval Group {
		# Common groups - permanent non-conditional membership
		namespace eval common::meta {
			array set Send:text {type action menu {Send Multicast} weight .10}
			array set Rename {type action menu Rename weight .98}
			array set Delete {type action menu Delete weight .99}
			array set Alias {default "" save change}
			#array set Groups {type groups default {} save change menu Groups}
		}
	}
	# Proxy servers
	namespace eval Proxy::meta {
		array set type {type variant weight .1 default http
			valuescript {proxy::types} save change}
		array set server {type fqdn save change weight .2}
		array set port {type {port digit} save change weight .3}
		array set user {save change weight .4}
		array set password {type password save change weight .5}
	}

	# Pseudo-object 'Me' to keep properties of current users similar to 
	# properties of ordinary users. These properties does not require 
	# metainformation, as user have no access to them.
	array set Me { Alias Me Status offline }
}

# Validation commands
namespace eval validate {
	foreach x {alnum alpha ascii boolean digit double integer 
			lower print upper wordchar xdigit} {
		interp alias {} [namespace current]::$x {} string is $x
	}
}

proc valid {metavar val} {
	upvar #0 $metavar meta
	if {![info exists meta(type)]} { return 1 }
	foreach x $meta(type) {
		set cmd [info commands validate::$x]
		if {$cmd!="" && ![$cmd $val]} {return 0}
	}
	return 1
}

set LOGLEVEL 6
set logfd stderr

proc ref {obj} { return ::objects::[string map {: ::} $obj] }
proc obj {ref} {join [lrange [split [string map {:: "\0"} $ref] "\0"] 2 end] :}

proc get {obj key {default ""}} {
	set var [ref $obj]($key)
	expr {[info exists $var]?[set $var]:[set default]}
}

proc pref {name} { return ::modules::[join $name ::] }

proc meta2var {x} {
	return [namespace qualifiers [namespace qualifiers $x]]::[namespace tail $x]
}

proc var2meta {x} {
	return [namespace qualifiers $x]::meta::[namespace tail $x]
}


proc select {{roots {}} {expr {}}} { 
	if {$roots==""} {
		set roots [namespace children ::objects]
	} else { set roots [map x $roots {ref $x}] }
	_select $roots $expr 
}

proc _select {roots expr} {
	set result [list]
	foreach root $roots {
		foreach x [info vars ${root}::*] {
			if {![array exists $x]} continue
			if {$expr!=""} {
				namespace eval test {}
				foreach {key val} [array get $x] { 
					set test::$key $val
				}
				set res [namespace eval test [list expr $expr]]
				namespace delete test
				if {!$res} continue
			}
			lappend result [obj $x]
		}
		foreach x [namespace children $root] {
			switch [namespace tail $x] meta continue
			set result [concat $result [_select $x $expr]]
		}
	}
	set result
}

proc changes? {var meta val} {
	if {[info exists $var]} {return  [expr {[set $var]!=$val} ]}
	if {[info exists ${meta}(default)]} {
		upvar #0 ${meta}(default) default
	} else { set default "" }
	expr {$default!=$val} 
}


proc DefaultProperties {} {
	property *|Alias {-type string -default unknown -save change}
	property *|Rename {-type action -name Rename -weight 97 }
	property *|Delete {-type action -name Delete -weight 98 }
	property Global|Exit {-type action -weight 99 -name Exit }
	property Contact:*|Groups [list -type list -default [nil] -save change]
	property Group:*|Groups	[list -type list -default [nil] -save change]
}

proc DefaultLogging {} {
	set ::LOGNAME [file join [file dirname $::RCFILE] alicq.log]
	if {[catch {
		if {[file exists $::LOGNAME]} {file delete $::LOGNAME}
		set ::logfd [open $::LOGNAME a+]
		fconfigure $::logfd -buffering none
	} reason]} {
		set ::logfd stderr
		Event Log 0 "Can not open $::LOGNAME, using stderr: $reason"
	}
	Event Log 0 "Alicq started"
}

# Default resources and keybindings common for all modules
proc DefaultTheme {} {
	option add *Entry.background gray90 widgetDefault
	option add *Text.background gray90 widgetDefault
	option add *borderWidth 1 widgetDefault
	option add *Button*overRelief raised widgetDefault
	option add *highlightThickness 0 startupFile
	option add *compound left widgetDefault
	
	switch -- $::tcl_platform(platform) {
		unix {
			option add *font -*-fixed-medium-r-normal--13-*\
				widgetDefault
		}
	}
	
	event add <<Close>> <Escape>
	event add <<Accept>> <Return>
}

proc AddObject {type name} {
	switch $type {
		group {Group [clock seconds] $name}
		contact {Event AddICQContact $name {other} $name}
	}
}

proc Contact {uin alias groups {properties {}}} {
	Setter Contact:ICQ:$uin $properties [list Alias $alias Groups $groups]
}
proc Group {gid alias {groups {}} {properties {}}} { 
	Setter Group:common:$gid $properties [list Alias $alias Groups $groups]
}

proc Setter {id plist list} {
	foreach item $plist {
		foreach {key val} [split $item =] break
		lappend list $key $val
	}
	new $id $list
}

proc new {id list} {
	variable compatible
	if {$compatible>1} {
	   trace variable [ref $id] w [list [namespace current]::onChange $id]
	}
	array set [ref $id] $list
	Event New:$id $id
}

# DEBUG: map contact groperty to variables
if $compatible {
	package require properties
	proc meta:property {name val} {
		hook $val [nc monitorProperty $name]
		trace variable $name w [list\
			[namespace current]::setProperty $val $name]
	}

	proc monitorProperty {name id old new} {
		upvar #0 $name var
		if {![info exists var] || $var!=$new} {set var $new}
	}

	proc setProperty {prop name args} {
		upvar #0 $name x
		pset $prop $x
	}

	if {$compatible>1} {
		handler {Contact:*|* Group:*|*} debug_assign {args} {
			if {[llength $args]!=3 ||
			    ![string match *|* [lindex $args 0]]} return
			foreach {uid prop} [split [lindex $args 0] |] break
			set ref [ref $uid]
			set ${ref}($prop) [lindex $args end]
		} 0.1

		proc onChange {id name1 name2 op} {
			upvar 1 ${name1}($name2) x
			pset $id|$name2 $x
		}
	}
} else {
	proc property {args} {}
	proc prop2var {key} {
		foreach x [metainfo:module] {
			if {[info exists ${x}(property)] && 
			    [set ${x}(property)]==$key} { 
			    	return [meta2var $x]
			}
		}
		return -code return ""
	}
	proc pset {key val} { set [prop2var $key] $val }
	proc pget {key} { set [prop2var $key] }

	proc nil {} { format c 0 }
	proc nil? {obj} { string equal $obj [format c 0] }
}

# This is quite independant routine used for unified saving changes to file
proc Save {file key newval} {
	set changed 0
	set attr 0600
	set fd2 [open ${file}.new w]
	if {![catch {set fd1 [open $file r]}]} {
		if {$::tcl_platform(platform)=={unix}} {
			set attr [file attributes $file -permissions]
		}	
		seek $fd1 0
		while {[gets $fd1 ln]!=-1} {
			if {[regexp $key $ln]} {
			 	if {![nil? $newval]} {puts $fd2 $newval}
			 	set changed 1
			} else { puts $fd2 $ln }
		}
		close $fd1
	}	
	if {!$changed && ![nil? $newval]} {puts $fd2 $newval }
	close $fd2
	file rename -force ${file}.new $file
	if {$::tcl_platform(platform)=={unix}} {
		file attributes $file -permissions $attr
	}	
}


proc rc_unix {} { list ~/.alicq/alicqrc }

proc rc_windows {} {
	set rclist [list]
	set mydir [file join [pwd] [file dirname [info script]]]
	if {[info exists ::env(USERPROFILE)]} {
		lappend rclist [file join $::env(USERPROFILE) alicq alicq.cfg]
	}
	lappend rclist [file join $mydir alicq.cfg]
	set rclist
}

proc rc_fallback {} { 
	list [file join [file dirname [info script]] alilcq.cfg]
}

# Generic handler of command-line options
proc cmdline {} {
	if {![info exists ::argv]} return
	array set opts [list]
	set key _
	foreach x $::argv {
		if {[string match -* $x]} { 
			set key [string range $x 1 end]
			set opts($key) [list]
		} else { lappend opts($key) $x }
	}
	foreach {opt val} [array get opts] {
		set cmd option_$opt
		if {[llength [info commands $cmd]]} {
			if {[catch {eval $cmd $val} v]} {
				Log 0 "Error while processing option $opt: $v"
			}
		}
	}
}

proc option_version {args} {
	puts "Alicq [package present alicq]"
	Finis
}

proc option_help {args} {
	puts "alicq options:
	-version		display alicq version
	-info \[<modules>\]	list metainformation for modules
	-xmlinfo \[<modules>\]	list metainformation for modules in xml format
	-config <file>		specify startup file to use
	-help			this help"
	Finis
}

proc option_info {args} {
	hook ConfigLoaded [list showmetainfo $args] .1
	hook ConfigLoaded Finis .11
}

proc option_xmlinfo {args} {
	hook ConfigLoaded [list xmlmetainfo $args] .1
	hook ConfigLoaded Finis .11
}

proc option_properties {mask args} {
	proc DumpProperties {mask} {
		puts [format {%-30s %-8s %s} Property Type Description]
		puts [string repeat - 80]
		foreach y [lsort [grep x [lsmeta] {[string match $mask $x]}]] {
			if {[property::exists $y -descr]} {
				set descr [property::meta $y -descr]
			} else { set descr "(no description)" }
			puts [format {%-30s %-8s %s} $y\
				[property::meta $y -type] $descr]
		}
		Finis
	}
	hook ConfigLoaded "DumpProperties $mask"
}

proc option_config {name args} { set ::rc [list $name] }

proc outname {name} { lrange [string map {:: " "} [meta2var $name]] 2 end }
proc showmetainfo {{names {}}} {
	if {![llength $names]} { 
		set names [namespace children ::modules]
	} else { set names [map x $names { set _ ::modules::$x }] }
	foreach ns $names {
		set meta [lsort [metainfo:module $ns]]
		if {![llength $meta]} continue
		puts "Module [namespace tail $ns]"
		set prop [list]
		foreach x $meta {
			if {![array exists $x]} {
				puts [format {%15s: %s} [outname $x] [set $x]]
			} else {lappend prop $x}
		}
		if {[llength $prop]} { puts "     Parameters:"}
		foreach x $prop {
			if {[info exists ${x}(description)]} { 
				set d [set ${x}(description)]
			} else { set d "(no description)" }
			puts [format {%20s - %s} [outname $x] $d]
			foreach field {type default} {
				if {[info exists ${x}($field)]} {
					puts [format {%30s: %s} $field [set ${x}($field)]]
				}
			}
			set value (undefined)
			catch {set value [set [meta2var $x]]} 
			puts [format {%30s: %s} current $value]
		}
		puts ""
	}
}

proc safexml {str} {
	string map [list "<" "&lt;" ">" "&gt;"] $str
}

proc xmlmetainfo {{names {}}} {
	if {![llength $names]} { 
		set names [namespace children ::modules]
	} else { set names [map x $names { set _ ::modules::$x }] }
	puts "<?xml version=\"1.0\"?>\n<metainformation>"
	foreach ns $names {
		set meta [lsort [metainfo:module $ns]]
		if {![llength $meta]} continue
		puts "\t<module name='[namespace tail $ns]'>"
		set prop [list]
		foreach x $meta {
			set name [outname $x]
			if {![array exists $x]} {
				puts "\t\t<$name>[safexml [set $x]]</$name>"
			} else {
				puts "\t\t<parameter name='$name'>"
				foreach {field val} [array get $x] {
					puts "\t\t\t<$field>[safexml $val]</$field>"
				}
				puts "\t\t</parameter>"
			}
		}
		puts "\t</module>"
	}
	puts "</metainformation>"
}

# Weird way to chech existance of namespace in pre-8.4 tcl
if {[package vsatisfies [package present Tcl] 8.4]} {
	proc nsexists {ns} { namespace exists $ns }
} else {
	proc nsexists {ns} { expr [catch { namespace parent $ns }]^1 }
}

proc LoadConfig {name} {
	set base [file dirname $name]
	if {[catch [list cd $base] v]} {
		puts "Can not cd into $base: $v"
		return 0
	}
	rename unknown unknown.config
	proc unknown {args} {
		set res ""
		if {[catch { set res [eval unknown.config $args] } v]} {
			# Check object namespace
			set uid [join [lrange $args 0 end-1] :]
			set ns [namespace qualifiers [ref $uid]]
			if {[nsexists $ns]} {
				new $uid [lindex $args end]
			} else { Log 0 $v }
		} 
		set res
	}
	set res [catch {uplevel #0 [list source $name]} v]
	rename unknown {}
	rename unknown.config unknown
	if {$res} {
		Log 0 "Error while loading startup file $name: $v"
		return 0
	} else {
		Event ConfigLoaded
		return 1
	}
}

proc moduleconfig {ns args} {
	set var [join [concat [list {} modules $ns] $args] ::]
	set name [var2meta $var]
	if {[llength [info vars $name]]} { return $var }
	set var [namespace qualifiers $var]
	set name [var2meta $var]
	if {![llength [info vars $name]]} { 
		Event Log 2 "No configuration parameter [lrange $args 0 end-1] in $ns"
		return ""
	}
	set new [lindex $args end]
	if {![valid $name $new]} {
		Event Log 2 "$ns $parm: $new is not valid $meta(type)"
		return ""
	}
	set $var $new
}

proc configuration-object {class name list} {
	puts "new $class:$name $list"
	new $class:$name $list
}

proc ::check_icons {name} { expr [lsearch [image names] img:$name]!=-1 }
proc ::load_icons {name file} { image create photo img:$name -file $file }

proc empty_check {args} {return 0}

proc check_modules {name} { 
	set name ::modules::[file tail [file rootname $name]]
	expr [lsearch [namespace children ::modules] $name]!=-1
}
proc load_modules {name file} { 
	set ns [file tail [file rootname $name]]
	namespace eval ::modules::$ns [list source $file]
	metainfo:update $ns
	if {[info commands $ns]=="" && 
	    [llength [grep x [metainfo:module ::modules::$ns] {
	    			[array exists $x]}]]} {
		interp alias {} $ns {} moduleconfig $ns
	}
	Event ModuleLoaded $name
}

# Procedures for work with metainformation

# Generic metainfo wrapper - interface for external modules 
proc metainfo {sub args} {
	set cmd metainfo:$sub
	if {[llength [info commands $cmd]]} {
		eval [list $cmd] $args
	} else { return -code error "Invalid metainfo subcommand $sub" }
}

# Get metainformation about all modules or given module's namespace
# configurable parameters
proc metainfo:module {{root ::modules}} {
	set list [list]
	foreach x [info vars ${root}::meta::*] { lappend list $x }
	foreach x [namespace children $root] { 
		switch -- [namespace tail $x] meta continue
		set list [concat $list [metainfo:module $x]]
	}
	set list
}

# Get list of variables containing metainformation on object, object namespace,
# or single object field
proc metainfo:object {obj {field ""}} {
	set result [list]
	for {set ns [ref $obj]} {$ns!=""} {set ns [namespace qualifiers $ns]} {
		if {$field!=""} {
			if {[info exists ${ns}::meta::${field}]} {
				return ${ns}::meta::${field}
			} else continue
		}
		foreach x [info vars ${ns}::meta::*] {
			if {[lsearch $result *::[namespace tail $x]]==-1} {
				lappend result $x
			}
		}
	}
	set result
}

# Update module metainformation - handle known metaproperties
proc metainfo:update {ns} {
	foreach x [metainfo:module ::modules::$ns] {
		upvar #0 $x meta
		if {[array exists $x]} {
			set var [meta2var $x]
			foreach {key val} [array get $x] {
				if {[llength [info commands meta:$key]]} {
					meta:$key $var $val
				}
			}
		}
	}
}

# Handle 'default' metaproperty of module
proc meta:default {name val} {
	trace variable $name r [list [namespace current]::setDefault $val $name]
}

proc setDefault {default var name1 name2 op} {
	upvar #0 $var x
	if {![info exists x]} { set x $default }
	trace vdelete $var r [list [namespace current]::setDefault $default $var]
}

proc resource {class names} {
	set checker check_$class
	if {![llength [info commands check_$class]]} {set checker empty_check}
	set toload [list]
	foreach x $names { if {![$checker $x]} {lappend toload $x} }
	if {![llength $toload]} return
	if {[llength [info commands option]]} {
		set preferred [option get . ${class}Set ResourceSet]
	} else { set preferred "" }
	set fdirs [resource_path $class $preferred]
	foreach name $toload {
		set found 0
		foreach dir $fdirs {
			foreach x [glob -nocomplain -directory $dir\
							$name $name.*] {
				if {[catch {load_$class $name $x} v]} {
					Event Log 1 "$class: Error loading $name from $x: $v"
				} else {
					Event Log 8 "$class: $name loaded from $x"
					set found 1
					break
				}
			}
			if {$found} break
		}
	}
}
interp alias {} Module {} resource modules
interp alias {} module {} resource modules

proc resource_path {class preferred} {
	lappend const $preferred
	if {$preferred!=""} { lappend const "" }
	set path [list]
	foreach x $::DATA {
		foreach y $const { lappend path [file join $x $class $y] }
		set path [concat $path [glob -nocomplain\
			-type d -directory [file join $x $class] *]]
	}
	set path
}

# Logging
handler Log Log {level str} {
	if {$level>$::LOGLEVEL} return
	set ln "\[[clock format [clock seconds] -format {%x %X}]\]: $str"
	if {[catch { puts $::logfd $ln } reason]} {
		Event Error output $reason
	}
	Event ViewLog $ln
} 

handler Error Error {id {description "Unknown error"}} {
	Event Log 0 "ERROR $id \($description\)" 
	list $id $description
}

handler exit Finis {args} { 
	exit
} 0.999

proc CopyConfig {src dest} {
	if {[file exists $src]} {
		set base [file dirname $dest]	
		if {![file isdirectory $base]} {
			if {[catch { file mkdir $base } v]} {
				Error internal "Can't create base directory: $v"
				return 0
			}
		}
		if {[catch { file copy $src $dest} v]} {
			Error internal "Can't copy default config: $v"
			return 0
		}
		if {$::tcl_platform(platform)=="unix"} {
			if {[catch { file attributes $dest\
					-permissions 0600 } v]} {
				Error internal "Can't change permissions: $v"
			}
		}
	} else {
		Error internal "Can't find default config file"
		return 0
	}
	return 1
}

proc InitRC {} {
	# If no rc-candidates yet, take platform-dependent list of them
	if {![info exists ::rc]} {
		set ::rc [list]
		set cmd rc_$::tcl_platform(platform)
		if {[llength [info commands $cmd]]} {
			set ::rc [$cmd]
		} else { set ::rc [rc_fallback] }
	}

	foreach name $::rc { 
		if {[file exists $name]} { set ::RCFILE $name; break } 
	}

	if {![info exists ::RCFILE]} {
		if {![llength $::rc]} {
			Error internal "Do not know where config file should be"
			Finis
		}
		Log 0 "No startup file found, creating default"
		set ::RCFILE [lindex $::rc 0]
		set default [file join [file dirname [info script]] default.rc]
		CopyConfig $default $::RCFILE
	}
}

proc alicq_init {} {
	set mydir [file join [pwd] [file dirname [info script]]]
	# Detect available data paths
	set ::DATA [list]
	foreach x [list [file dirname $::RCFILE] [file join $mydir data]\
			[file join [file dirname $mydir] share alicq]] {
		if {[file isdirectory $x]} { lappend ::DATA $x }
	}
	# Internatinalization
	package require msgcat
	namespace import msgcat::*
	foreach x $::DATA {
		if {[msgcat::mcload [file join $x messages]]} break 
	}
	LoadConfig $::RCFILE
}

if {![catch {package present Tk}]} DefaultTheme
# Parse command-line options
cmdline
# Check for default rc file
InitRC
# Default loggind
DefaultLogging

DefaultProperties

