/*****************************************************************************
*
* ALPS Project: Algorithms and Libraries for Physics Simulations
*
* ALPS Light Libraries
*
* Copyright (C) 1994-2002 by Matthias Troyer <troyer@itp.phys.ethz.ch>,
*                            Synge Todo <wistaria@comp-phys.org>
*
* This software is part of the "ALPS Light" Libraries, public-domain
* part of the ALPS Libraries. If you need the full functionality of
* the ALPS Libraries, such as Lattice, Model, Scheduler, etc, please
* use the full version of ALPS Libraries, which is available from
* http://alps.comp-phys.org/.
*
* Permission is hereby granted, free of charge, to any person or organization
* obtaining a copy of the software and accompanying documentation covered by
* this license (the "Software") to use, reproduce, display, distribute,
* execute, and transmit the Software, and to prepare derivative works of the
* Software, and to permit third-parties to whom the Software is furnished to
* do so, all subject to the following:
*
* The copyright notices in the Software and this entire statement, including
* the above license grant, this restriction and the following disclaimer,
* must be included in all copies of the Software, in whole or in part, and
* all derivative works of the Software, unless such copies or derivative
* works are solely in the form of machine-executable object code generated by
* a source language processor.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT 
* SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE 
* FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, 
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
* DEALINGS IN THE SOFTWARE.
*
*****************************************************************************/

/* $Id: vectormath.h,v 1.4 2004/03/16 15:12:26 wistaria Exp $ */

#ifndef ALPS_VECTORMATH_H
#define ALPS_VECTORMATH_H

#include <alps/config.h>
#include <alps/functional.h>
#include <alps/typetraits.h>
#include <vector>

namespace alps {

template <class T, class OP>
std::vector<T> vector_vector_apply(OP op, const std::vector<T>& x, const std::vector<T>& y)
{
  typedef typename std::vector<T>::size_type size_type;
  size_type end=std::min(x.size(),y.size());
  std::vector<T> res(std::max(x.size(),y.size()));
  for (size_type i=0;i<end;++i)
    res[i]=op(x[i],y[i]);
  for (size_type i=end;i<x.size();++i)
    res[i]=op(x[i],T());
  for (size_type i=end;i<y.size();++i)
    res[i]=op(T(),y[i]);
  return res;
}

template <class T, class S, class OP>
std::vector<T> scalar_vector_apply(OP op, S x, const std::vector<T>& y)
{
  typedef typename std::vector<T>::size_type size_type;
  std::vector<T> res(y.size());
  std::transform(y.begin(),y.end(),res.begin(),std::bind1st(op,x));
  return res;
}

} // namespace alps

#ifndef BOOST_NO_OPERATORS_IN_NAMESPACE
namespace std {
#endif

template <class T>
std::vector<T> operator+(const std::vector<T>& x, const std::vector<T>& y)
{
  return alps::vector_vector_apply(std::plus<T>(),x,y);
}

template <class T>
std::vector<T> operator-(const std::vector<T>& x, const std::vector<T>& y)
{
  return alps::vector_vector_apply(std::minus<T>(),x,y);
}

template <class T>
std::vector<T> operator-(const std::vector<T>& x)
{
  std::vector<T> res(x.size());
  std::transform(x.begin().x.end(),res.begin(),std::negate<T>());
  return res;
}

template <class T, class S>
std::vector<T> operator*(S x, const std::vector<T>& y)
{
  return alps::scalar_vector_apply(alps::multiplies<S,T,T>(),x,y);
}

template <class T, class S>
std::vector<T> operator*(const std::vector<T>& y, S x)
{
  return alps::scalar_vector_apply(alps::multiplies<S,T,T>(),x,y);
}

#ifndef BOOST_NO_OPERATORS_IN_NAMESPACE
} // end namespace std
#endif

#endif // ALPS_VECTORMATH_H
