# $Id: Qmail.pm,v 1.8 2003/03/17 23:04:21 bengen Exp $

#
# Support module for the Qmail MTA
#

package AMAVIS::MTA::Qmail;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;

use Fcntl;
use IO::Pipe;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

use vars qw(
	    $cfg_qmail_queue_binary
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line
	   );

sub init {
  my $self = shift;
  my $args = shift;

  $cfg_qmail_queue_binary=$AMAVIS::cfg->val('Qmail', 'qmail-queue');
  if ($cfg_qmail_queue_binary eq '') {
    writelog($args,LOG_CRIT, "Location of qmail-queue binary not specified");
    return 0;
  }
  if (! -x $cfg_qmail_queue_binary) {
    writelog($args,LOG_CRIT, "$cfg_qmail_queue_binary not executable");
    return 0;
  }
  if ($AMAVIS::cfg->val('global', 'x-header') eq 'true') {
    $cfg_x_header = 1
  };
  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  return 1;
}

# Create/obtain temp dir. Write mail if necessary.
sub get_directory( $ ) {
  my $self = shift;
  my $args = shift;

  my $prefix = "$AMAVIS::cfg_unpack_dir/amavis-unpack-";
  my $i = 0;
  my $message_id;
  while (1) {
    $message_id = sprintf("%.8x-%.4x",time,$$);
    unless (defined mkpath ($prefix.$message_id, 0, 0770)) {
      if (++$i > 10) {
	return 0;
      }
      else {
	next;
      }
    }
    last;
  }
  $$args{'message_id'}=$message_id;
  my $directory = $prefix.$message_id;
  mkdir "$directory/parts", 0777;

  my $message_handle = IO::Handle->new();
  # open(ENVELOPE,"<&STDOUT");
  my $envelope_handle;
  # my $envelope_handle=IO::File->new();

  # Mimic behaviour of qmail-queue(8)
  # This means taking the message from descriptor 0 and then the
  # envelope information from descriptor 1.
  unless ($message_handle->fdopen(0,'r')) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Could not read filedescriptor 0 for message");
    return 0;
  }

  # Open message file that is later going to be disected and scanned
  my $output = IO::File->new("+>$directory/email.txt");

  my $headers=1;
  $$args{'headers'}='';
  while (<$message_handle>) {
    if (/^ *$/) {
      $headers=0;
    }
    if ($headers==1) {
      next if (/^From /);
      $$args{'headers'}.=$_;
    }
    print $output $_;
  }
  $message_handle->close;

#  unless ($envelope_handle->open("<&STDOUT")) {
  open ENVELOPE, '<&1';
  unless ($envelope_handle = IO::Handle->new_from_fd(fileno(ENVELOPE),'r')){
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Could not open filedescriptor 1 for envelope information");
    return 0;
  }

  my $recipline=<$envelope_handle>;
  $envelope_handle->close;

  writelog($args,LOG_DEBUG, __PACKAGE__.": RLINE: $recipline\n");


  if ($recipline=~/^F(.*?)\0T(.*)\0$/) {
    @{$$args{'recipients'}}= split(/\0T/,($2.'T'));
    if ((defined $1) && $1 ne '') {
      $$args{'sender'}=$1;
    }
    else {
      writelog($args,LOG_DEBUG, __PACKAGE__.
	       ": Empty sender");
      $$args{'sender'}='<>';
    }
  }
  else {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Wrong recipient line format");
    return 0;
  }


  writelog($args,LOG_DEBUG,__PACKAGE__.": Sender is $$args{'sender'}");
  writelog($args,LOG_DEBUG,__PACKAGE__.": Recipients are "
	   .join(', ',@{$$args{'recipients'}}));

  $output->seek(0,0);
  $$args{'filehandle'} = $output;
  $$args{'directory'} = $directory;

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to continue message delivery
sub accept_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");

  my $envelope_pipe=new IO::Pipe;
  my $message_pipe=new IO::Pipe;
  my $pid = fork();

  unless(defined $pid) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Couldn't fork()");
  }
  if ($pid) {
    # We are the parent. So we write to the pipes.
    $envelope_pipe->writer();
    $message_pipe->writer();

    my @recipients = grep(!/<>/, @{$$args{'recipients'}});
    
    if ($#recipients == -1) {
      writelog($args,LOG_DEBUG, __PACKAGE__.
	       ": No recipients: Not sending mail.");
      return 1;
    }


    # Write envelope information
    $envelope_pipe->print('F'.$$args{'sender'}."\000T".
 			  join("\000T",@recipients).
 			  "\000\000");

    # Write message data
    if ($cfg_x_header) {
      $message_pipe->print("$cfg_x_header_tag: $cfg_x_header_line\n");
    }
    while (my $line=$$args{'filehandle'}->getline()) {
      $message_pipe->print($line);
    }

    $envelope_pipe->close;
    $message_pipe->close;

    if(waitpid($pid, 0) == -1) {
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": $cfg_qmail_queue_binary did not exist?!");
    }
    else {
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": $cfg_qmail_queue_binary exited: ".($?>>8));
    }
  }
  else {
    # We are the child. So we exec the real qmail-queue
    exec_qmail_queue($args, $message_pipe, $envelope_pipe);
  }

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to throw message away
sub drop_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message");

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to freeze message delivery
sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  if (AMAVIS->quarantine_problem_message($args)) {
    # Return successfully
    return 1;
  }
  else {
    writelog($args,LOG_ERR, "Could not freeze message");
    return 0;
  }
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message( $$$ ) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = grep(!/<>/, @_);

  if ($#recipients == -1) {
    writelog($args,LOG_DEBUG, __PACKAGE__.
	     ": No recipients: Not sending mail.");
    return 1;
  }

  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  if ($sender eq '<>') {
    $sender = '';
  }

  my $envelope_pipe=new IO::Pipe;
  my $message_pipe=new IO::Pipe;
  my $pid = fork();

  unless(defined $pid) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Couldn't fork()");
  }
  if ($pid) {
    # We are the parent. So we write to the pipes.
    $envelope_pipe->writer;
    $message_pipe->writer;

    # Write envelope information
    $envelope_pipe->print('F'.$sender."\000T".
			  join("\000T",@recipients).
			  "\000\000");

    # Write message data
    # $message_pipe->print($message);
    foreach my $line (split /$^/m, $message) {
      $message_pipe->print($line);
    }

    $envelope_pipe->close;
    $message_pipe->close;

    if(waitpid($pid, 0) == -1) {
      writelog($args,LOG_ERR, __PACKAGE__.
	       "$cfg_qmail_queue_binary did not exist?!");
    }
    else {
      writelog($args,LOG_ERR, __PACKAGE__.
	       "$cfg_qmail_queue_binary exited: ".($?>>8));
    }
  }
  else {
    # We are the child. So we exec the real qmail-queue
    exec_qmail_queue($args, $message_pipe, $envelope_pipe);
  }

  # Return successfully
  return 1;
}

sub exec_qmail_queue {
  my $args = shift;
  my $message_pipe=shift;
  my $envelope_pipe=shift;
  # We are the child. So qmail-queue will read from the pipes.
  $message_pipe->reader();
  $envelope_pipe->reader();

  # Duplicate the file descriptors
  close(STDERR);
  close(STDOUT);
  my $result;
  unless($result=fcntl($envelope_pipe, F_DUPFD, 1)) {
    writelog($args,LOG_ERR, "Error $! duping to 1");
  };
  $envelope_pipe->close;
  close(STDIN);
  unless($result=fcntl($message_pipe, F_DUPFD, 0)) {
    writelog($args,LOG_ERR, "Error $! duping to 0");
  };
  $message_pipe->close;
  
  $ENV{PATH}  = '/bin:/usr/bin:/sbin:/usr/sbin';
  undef $ENV{QMAILQUEUE};
  exec($cfg_qmail_queue_binary) || do {
    # If exec() succeeds, the following statements are not reached.
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Could not exec $cfg_qmail_queue_binary");
    die();
  };
}

sub deadhandler {
  die @_ if $^S;
}

1;
