# $Id: GZIP.pm,v 1.12 2002/07/17 14:02:41 bengen Exp $

#
# Module for unpacking gzipped files, including .tar.gz
#

# We should not just let gzipped tar files be handled by Tar.pm since
# strict checking whether a file is a tarfile is unreliable. We want
# to minimize Tar.pm's barfing on our feet which whould stop mail
# delivery alltogether.

# The downside is that we get yet another file.

package AMAVIS::Extract::GZIP;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;

use Compress::Zlib;

sub init {
  my $self = shift;
  my $args = shift;
  my $types = shift;
  $$types{'application/x-gzip'}=$self;
  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  return 1;
}

sub extract {
  my $self = shift;
  my $args = shift;
  my $filename = shift;
  my $securename=get_secure_filename($args);
  my $unpacked_size = 0;

  writelog($args,LOG_DEBUG, __PACKAGE__.
	   ": Attempting to unpack $filename as gzipped file");

  if ($$args{'unpacked_files'}++ > $cfg_maxfiles) {
    writelog($args,LOG_ERR, __PACKAGE__.": Unpacking uses too many files");
    return 0;
  }

  my $gz = gzopen("$$args{'directory'}/parts/$filename", "rb") or do {
    writelog($args,LOG_ERR, __PACKAGE__.
	     "Unable to open .gz file. GZERRNO=$gzerrno");
    return 0;
  };

  my $out_handle=IO::File->new(">$$args{'directory'}/parts/"
			       .$securename);

  my $len;

  while (($len = $gz->gzread(my $buffer, 4096)) > 0) {
    $out_handle->write($buffer, $len);
    $unpacked_size += $len;
    if ($$args{'unpacked_size'} + $unpacked_size >= $cfg_maxspace) {
      $out_handle->close();
      writelog($args,LOG_ERR, __PACKAGE__.": Unpacking takes too much space");
      return 0;
    }
  }

  $out_handle->close();
  if ($len < 0) {
    if ($gzerrno != Z_STREAM_END) {
      my $errstring=$gz->gzerror();
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": Error reading from $filename: $errstring. ".($gzerrno+0));
      $gz->gzclose();
      return 0;
    }
  }
  $gz->gzclose();

  $ {$$args{'contents'}}{$securename} = {};
  $ {$ {$$args{'contents'}}{$securename}}{insecure_type}='';
  $$args{'unpacked_size'} += $unpacked_size;
  return 1;
}

1;
