# $Id: Exim.pm,v 1.23 2005/01/14 18:30:03 braeucup Exp $

#
# MTA module for "regular" Exim setup
#

package AMAVIS::MTA::Exim;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;


use vars qw(
	    $cfg_exim_binary
	    $cfg_exim_args
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line
	   );

sub init {
  my $self = shift;
  my $args = shift;
  $cfg_exim_binary = ($AMAVIS::cfg->val('Exim', 'exim') || '/usr/sbin/exim');
  $cfg_exim_args = ($AMAVIS::cfg->val('Exim', 'args') || '-oMr no-scan -i -f');

  if (! -x $cfg_exim_binary) {
    writelog($args,LOG_CRIT, "$cfg_exim_binary not executable");
    return 0;
  }

  if (($AMAVIS::cfg->val('global', 'x-header') || '') eq 'true') {
    $cfg_x_header = 1
  };
  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  return 1;
}

# Create temp dir and write mail
sub get_directory {
  my $self = shift;
  my $args = shift;

  # Create temp directory. Try creating $prefix[date]-[pid] 
  # If unsuccessful after 10 tries, abort
  my $prefix = "$AMAVIS::cfg_unpack_dir/amavis-unpack-";
  my $i = 0;
  my $message_id;
  while (1) {
    $message_id = sprintf("%.8x-%.4x",time,$$);
    unless (defined mkpath ($prefix.$message_id, 0, 0770)) {
      if (++$i > 10) {
	return 0;
      }
      else {
	next;
      }
    }
    last;
  }
  $$args{'message_id'}=$message_id;
  my $directory = $prefix.$message_id;
  mkdir "$directory/parts", 0777;
  $$args{'directory'} = $directory;

  # Open message file that is later going to be disected and scanned
  my $output = IO::File->new("+>$directory/email.txt");

  # Get message from STDIN
  my $headers=1;
  $$args{'headers'}='';
  while (<STDIN>) {
    if (/^ *$/) {
      $headers=0;
    }
    if ($headers==1) {
      $$args{'headers'}.=$_;
    }
    print $output $_;
  }

  if ($#ARGV < 1) {
    writelog($args,LOG_ERR, __PACKAGE__.": Missing arguments to exim");
    return 0;
  }

  writelog($args,LOG_DEBUG, "Called as amavis ".join(' ',@ARGV));

  foreach (@ARGV) {
    /^-f$/ && next; # ignore "-f"
    /^-d$/ && next; # ignore "-d"
    /^(.*)$/; # untaint sender or recipient
    unless (defined $$args{'sender'}) {
      my $sender = $1;
      if ($sender =~ /^$/) {
        $sender= "<>";
      }
      writelog($args,LOG_DEBUG, "Sender: $sender");
      $$args{'sender'} = $sender;
    }
    else {
      my $recipient = $1;
      writelog($args,LOG_DEBUG, "Recipient: $recipient"); 
      push @{$$args{'recipients'}}, $recipient;
    }
  }

  # Message file has been written, reset file pointer and put it into
  # the record.
  $output->seek(0,0);
  $$args{'filehandle'} = $output;

  # Return successfully
  return 1;
}

# Generate a copy of the scanned message and pipe it to mailer.
sub accept_message {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");
  my @cfg_exim_args;

  push @cfg_exim_args, split(/\s+/,$cfg_exim_args);
  push @cfg_exim_args, $$args{'sender'};
  push @cfg_exim_args, @{$$args{'recipients'}};

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_exim_binary ".
	   join(' ',@cfg_exim_args));

  open(MAIL, "|-") || exec($cfg_exim_binary, @cfg_exim_args);
  # Copy the original message headers.
  while (my $line=$$args{'filehandle'}->getline()) {
    last if ($line =~ /^ *$/);
    print MAIL $line;
  }
  # Insert our headers.
  if ($cfg_x_header) {
    print MAIL "$cfg_x_header_tag: $cfg_x_header_line\n";
  }
  # Print an empty line.
  print MAIL "\n";
  # Copy the original message body.
  while (my $line=$$args{'filehandle'}->getline()) {
    print MAIL $line;
  }
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR, __PACKAGE__.": $cfg_exim_binary exited with ".($?>>8));
  }

  $$args{'status'} = 'accept';

  # Return successfully
  return 1;
}

# Just "forget" the mail.
sub drop_message {
  my $self = shift;
  my $args = shift;
  # Doing nothing should suffice
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message");

  # Return successfully
  return 1;
}

sub freeze_message {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  # Instead of freezing, drop the message, but store a copy in
  # $cfg_problem_dir -- analogous to AMAVIS::quarantine_message()
  if (AMAVIS->quarantine_problem_message($args)) {
    # Return successfully
    return 1;
  }
  else {
    writelog($args,LOG_ERR,__PACKAGE__.": Could not freeze message");
    return 0;
  }
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my @cfg_exim_args;

  push @cfg_exim_args, split(/\s+/,$cfg_exim_args);
  push @cfg_exim_args, $sender;
  push @cfg_exim_args, @recipients;

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_exim_binary ".
	   join(' ',@cfg_exim_args));

  open(MAIL, "|-") || exec($cfg_exim_binary, @cfg_exim_args);
  print MAIL $message;
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR,
	     __PACKAGE__.": $cfg_exim_binary @cfg_exim_args exited with ".($?>>8));
  }

  # Return successfully
  return 1;
}

1;
