// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#ifndef ANALYSE_H
#define ANALYSE_H
#ifdef _MSC_VER
#include "win32/config_win32.h"
#endif
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <list>
#include <ctime>
#include "report.h"
#include "parameter.h"

/** This is the abstract base class of every analyse object. After filtering the
  * input stream with a filter object, the this object make the analyse (...).
  * that's the main reason for the name analyse ;-). A plugin (at the moment the
  * analysePlugin object call first the function run(). After this one or more
  * times print(). Finally it calling free() to cleanup unused memory.
  *
  * Other plugins (current the reportPlugin Object) calling setReport to change
  * the output report type before, after or between any of this tree function
  * calls.
  *
  * The filtered data in the unity_map must not change between the run() and
  * the print() function call. You can also store this pointer to request details
  * while print(). But it's possible that the funtions of this object are called
  * many times in the order descibed above with different unity_map's.
  *@short Abstract base class of every analyse object.
  *@author Tobias Erbsland
  *@see reportPlugin
  *@see syslogFilter
  */
class Analyse : public Parameter
{

protected:
	/** This is a pointer to the defaultreport used in print().
	  *@see print
	  */
	Report * report; // Defaultreport.

public:
	/** This is the default structure to store a mta logline. 
	  */
	struct unity_t
	{
		long size;
		std::string message_id;
		std::string from;
		std::string client;
		time_t date;
		std::list<std::string> to_intern;
		std::list<std::string> to_extern;
	};

	/** Constructor of the Object. This constuctor here guarantee, that the report
	  * Pointer is initially 0.
	  */
	Analyse() : report( 0 ) { };

	/** This function should do all calculations for this analyse, and store the
	  * result for multible print() calls until someone call free() or destroy
	  * the object.
	  */
	virtual void run( std::map<std::string, unity_t> * unity_map ) = 0;

	/** This function printout the result of this analyse to the selected report
	  * object. Is no report object selected (report is a null pointer) - nothing
	  * should happens. (check also report first!).
	  */
	virtual void print( void ) = 0;

	/** This funcion should free unused memory (from map's etc.).
	  */
	virtual void free( void ) = 0;

	/** Set the output to a report object.
	  *@param r A pointer to an existing report Object.
	  */
	virtual void setReport( Report * r ) { report = r; };

	/** Get current selected report for this Object
	  * @return A pointer to the selected report object.
	  */
	virtual Report * getReport( void ) { return report; };
	
	/** Virtual deconstuctor.
	  */
	virtual ~Analyse();
};

#endif
