// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#include "ltdanalyse.h"

ltdAnalyse::ltdAnalyse() :
		strPrefix( "ltd" ),
		strName( "Local Traffic Distribution" ),
		strVersion( "V0.1" ),
		strAuthor( "Tobias Erbsland <te@profzone.ch>" ),
		strHelp( "Create statistics how the local traffic is distributed to external and local recipients." )
{
	addParam( "limit", "Limit Output to n Lines.", "Integer from 1 to x.", G_STRING, "25" );
	addParam( "no-delivery", "Statistic with not delivered mail",
	          "If set, not delivered mails are included.", G_FLAG, "0" );
	addParam( "only-no-delivery", "Statistic ONLY with not delivered mail",
	          "If set, ONLY not delivered mails are included.", G_FLAG, "0" );
}

void ltdAnalyse::run( std::map< std::string, unity_t > * unity_map )
{
	bool not_del = false;
	bool del = true;

	std::map< std::string, long > from_traffic_c;

	std::map< std::string, unity_t >::iterator i;
	std::map< std::string, long >::iterator j;

	if ( getFlagParam( "no-delivery" ) ) not_del = true; // Auch falsche;
	if ( getFlagParam( "only-no-delivery" ) )  // Hat auch vorrang.
	{
		not_del = true;
		del = false;
	};

	for ( i = unity_map->begin(); i != unity_map->end(); i++ )
	{
		// Nichtzugestellte oder zugestellte Mails aus statistik entfernen.
		if ( !not_del && ( i->second.to_intern.size() + i->second.to_extern.size() == 0 ) ) continue;
		if ( !del && ( i->second.to_intern.size() + i->second.to_extern.size() != 0 ) ) continue;

		// Externe mails ausklammern
		if ( i->second.from.find( "@" ) != std::string::npos ) continue;
		if ( i->second.from.length() < 2 ) continue; // zu kleine lokale namen ausklammern.

		from_traffic_c[ i->second.from ] +=
		    i->second.size * ( i->second.to_extern.size() + i->second.to_intern.size() );
		i2e_traffic_c[ i->second.from ] += i->second.size * i->second.to_extern.size();
		mailc_2e_c[ i->second.from ] += i->second.to_extern.size();
		mailc_2i_c[ i->second.from ] += i->second.to_intern.size();
	}
	// Temp map umrechnen.
	for ( j = from_traffic_c.begin(); j != from_traffic_c.end(); j++ )
		from_traffic.insert( std::make_pair( j->second, j->first ) );
}

void ltdAnalyse::print( void )
{
	if ( !report ) return ; // be sure :-)

	int limit = getIntParam( "limit" );

	std::multimap< long, std::string >::reverse_iterator k;

	report->startNewReport( "Local Traffic Distribution - ratio local to external", Report::TABLE );
	report->addCol( 4, Report::COUNTER, Report::RIGHT, "nr", Report::NONE );
	// -1 = soviel platz verwenden, wie noch vorhanden.
	report->addCol( -1, Report::STRING, Report::LEFT, "from local", Report::NONE );
	report->addCol( 10, Report::INTEGER, Report::RIGHT, "to local", Report::NONE, Report::SUM );
	report->addCol( 7, Report::INTEGER, Report::RIGHT, "count", Report::ROUND, Report::SUM );
	report->addCol( 10, Report::INTEGER, Report::RIGHT, "to extern.", Report::NONE, Report::SUM );
	report->addCol( 7, Report::INTEGER, Report::RIGHT, "count", Report::ROUND, Report::SUM );
	report->addCol( 10, Report::INTEGER, Report::RIGHT, "total", Report::NONE, Report::SUM );
	report->addCol( 8, Report::PERCENT, Report::RIGHT, "% ext.", Report::RECT );
	report->printHeader();
	for ( k = from_traffic.rbegin(); k != from_traffic.rend(); k++ )
	{
		report->printField( k->second );
		report->printField( k->first - i2e_traffic_c[ k->second ] );
		report->printField( mailc_2i_c[ k->second ] );
		report->printField( i2e_traffic_c[ k->second ] );
		report->printField( mailc_2e_c[ k->second ] );
		report->printField( k->first );
		report->printField( ( float ) i2e_traffic_c[ k->second ] * 100.0 / ( float ) ( k->first ) );

		if ( --limit < 1 ) break;
	}
	report->printFooter();
}

void ltdAnalyse::free( void )
{
	from_traffic.clear();
	i2e_traffic_c.clear();
	mailc_2e_c.clear();

	mailc_2i_c.clear();
}
