/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include "include.hpp"
#include <stack>
#include "database.hpp"
#include "utils.hpp"
#include "xsu.hpp"

using namespace std;

void XSU::translate( std::istream &inputStream,
                     std::ostream &outputStream ) const
  throw (Error)
{

  outputStream << "<?xml version='1.0' encoding='"SQLENCODING"'?>" << endl
               << "<data>" << endl;

  stack< string > tagName;

  while ( inputStream ) {

    string line;
    getline( inputStream, line );

    if ( line[0] == '#' ) {

      // Encountering comment. Check for possible "# OPEN tagname;" hook.
      if ( line.compare( 1, 5, " OPEN" ) == 0 ) {
        tagName.push( line.substr( 7, line.size() - 8 ) );
        // The error-handling for proper tag-names has to be done by the
        // Xalan-library at the moment.
        outputStream << "<" << tagName.top() << '>' << endl;
      } else if ( line.compare( 1, 6, " CLOSE" ) == 0 ) {
        string closingTagName = line.substr( 8, line.size() - 9 );
        ERRORMACRO( !tagName.empty(), Error, ,
                    "Cannot close tag \"" << closingTagName << "\", because "
                    "all tags have been closed already." );
        ERRORMACRO( tagName.top() == closingTagName, Error, ,
                    "Encountered closing of tag \"" << closingTagName
                    << "\", but was expecting closing of tag with name \""
                    << tagName.top() << "\"." );
        outputStream << "</" << closingTagName << '>' << endl;
        tagName.pop();
      };

    } else {

      // Pass on content to unixODBC/ODBC interface.
      StatementPtr statement = database->execQuery( line );

      if ( !tagName.empty() && statement->getNumCols() > 0 ) {

        // Extract meta-data.
        outputStream << "    <table>" << endl
                     << "      <meta>" << endl;
        for ( int i=0; i<statement->getNumCols(); i++ )
          outputStream << "        <column>"
                       << xmlText( statement->getColAttr( i ) )
                       << "</column>" << endl;
        outputStream << "      </meta>" << endl;

        // Extract data.
        ResultRowPtr row;
        while ( row = statement->fetchRow() ) {
          outputStream << "      <row>" << endl;
          for ( int i=0; i<statement->getNumCols(); i++ )
            outputStream << "        <column>" << xmlText( (*row)[i] )
                         << "</column>" << endl;
          outputStream << "      </row>" << endl;
          // Release object accessing results.
          row.reset();
        };
        outputStream << "    </table>" << endl;

      };
    };
  };

  ERRORMACRO( tagName.empty(), Error, ,
              "Expected closing of tag \"" << tagName.top() << "\"." );

  outputStream << "</data>" << endl;
}
