//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/un.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <unistd.h>
#include <netdb.h> 

//#define USE_BT

#ifdef USE_BT
#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#endif

#include "btio.h"
#include "common.h"
#include "parse.h"
#include "utils.h"
#include "conf.h"

extern char tmp[MAXMAXLEN];

static int portfd       = 0;
static int sportfd      = 0;
static int portflags    = 0;
static int oldflags     = 0;
static int rtscts       = 0;

static struct termios oldterm;
static struct termios portterm;

char * uSocket = NULL;

int bt_open_port(char* port) 
{
	
	/* get a file descriptor */
	if ((portfd = open(port, O_RDWR | O_NOCTTY | O_NDELAY)) < 0) {
		sprintf(tmp, "can't open %s", port);
		logger("ERROR", tmp);
		printf("ERROR: can't open %s\n", port);
		portfd = 0;
		return -1;
	}

	/* check to see that the file is a terminal */
	if (!isatty(portfd)) {
		printf("ERROR: %s does not refer to a terminal\n", port);
		return -1;
	}

	/* get port attributes, store in oldterm */
	if (tcgetattr(portfd, &oldterm) < 0) {
		printf("ERROR: Can't get port attributes (%s)\n", port);
		return -1;
	}

	/* get port flags, save in oldflags */
	if ((oldflags = fcntl(portfd, F_GETFL)) < 0) {
		printf("ERROR: Can't get port flags (%s)\n", port);
		return -1;
	}

	portterm = oldterm;
	portflags = oldflags;

	portterm.c_cflag = getBaudrate() | CRTSCTS | CS8 | CLOCAL | CREAD | O_NDELAY;
	if (!rtscts) {
		portterm.c_cflag &= ~CRTSCTS;
	}
	portterm.c_iflag	= IGNPAR;
	portterm.c_oflag	= 0;
	portterm.c_lflag	= 0;
	portterm.c_cc[VTIME]	= 0;
	portterm.c_cc[VMIN]	= 0;

	tcflush(portfd, TCIOFLUSH);

	if (tcsetattr(portfd,TCSANOW,&portterm) < 0) {
		printf("ERROR: Can't set port attributes (%s)\n", port);
		return -1;
	}

	/* set non-blocking */
	if (fcntl(portfd, F_SETFL, (portflags |= O_NONBLOCK)) < 0) {
		printf("ERROR: Can't set port flags (%s)\n", port);
		return -1;
	}
	return 0;
}

// Then port will be closed from a forked child use 0
int bt_close_port(int final) 
{
	int retval = 0;
	//printf("INFO: Close port\n");

	if (!portfd) { /* already closed */
		printf("INFO: Already closed ?\n");
		return -1;
	}

	if (final) {
		/* restore old settings */
		if (tcsetattr(portfd, TCSADRAIN, &oldterm) < 0) {
			retval = -1;
		}

		if (fcntl(portfd, F_SETFL, oldflags) < 0) {
			retval = -1;
		}
	}
	retval = close(portfd);
	portfd = 0;

	return retval;
}

int bt_read(char* buf, int len) 
{
	return read(portfd, buf, len);
}

int bt_read_tmo(char* buf, int len, int timeout) 
{
	int ok;
	fd_set rdfds;
	struct timeval tv;

	FD_ZERO(&rdfds);
	FD_SET(portfd, &rdfds);

	/* block until something to read or timeout occurs.  select() is damn cool */
	if (timeout < 0) {
		ok = select(portfd + 1, &rdfds, NULL, NULL, NULL);
	} else {
		tv.tv_sec = timeout / 1000000;
		tv.tv_usec = (timeout % 1000000);
		ok = select(portfd + 1, &rdfds, NULL, NULL, &tv);
	}

	if (ok > 0) {
		ok = read(portfd, buf, len);
		if (ok == 0) {
			logger("DEBUG", "EOF during read()");
			return EOF;
		}
	
		/////////////////////////////////////
		//if(ok > 0){
		//	char buf2[1024];
		//	strncpy(buf2,buf,ok);
		//	sprintf(tmp, "READ: %s", buf2);
		//	logger("DEBUG", tmp);
		//}
		//////////////////////////////////////
		return ok;
	
	} else {
		
		return 0;

		//if (ok < 0) {
		//    return -1;
		//} else {
		//    errno = ETIMEDOUT;
		//    return 0;
		//}
	}
}

int  bt_readchar(char* c, int timeout) 
{
	char buf[1];
	*c = 0;
        
	int rv =  bt_read_tmo(buf, 1, timeout);

	if (rv == 1) {
        	*c =  buf[0];
        	return 1;
        }
	if (rv == 0) return EOF - 1;
	
	//  else - EOF
	bt_close_port(0);
	return EOF;
}

int bt_write(char* buf, int len) 
{
	return write(portfd, buf, len);
}

int bt_getfd(void) {
	return portfd;
}

// max should be >= 100
int bt_put_command(char* command,
		   char* answer,
		   int   max,
		   int   timeout,
		   char* expect) 
{
	int count=0;
	int readcount;
	char tmp2[100];
	int timeoutcounter=0;
	int found=0;

	logger("DEBUG",command);
	if (expect != NULL) {
		logger("DEBUG",expect);
	}
	
	// send command
	if (command && command[0]) {
		write(portfd,command,strlen(command));
		tcdrain(portfd);
	}

	if (max == 0) {
		return 0;
	}
	
	answer[0]=0;
	do {
		// try to read some bytes.
		usleep(100000);
		//write(1,".",1);
		timeoutcounter++;

		// read data
		readcount=read(portfd,tmp2,sizeof(tmp2)-1);
		if (readcount<0) {
			readcount=0;
		}
		tmp2[readcount]=0;
	
		// add read bytes to the output buffer
		if (readcount) {
			strcat(answer,tmp2);
			count+=readcount;
			  
			// if we have more time to read, check if we got already the expected string
			if ((timeoutcounter<timeout) && (found==0)) {
				
				// check if it's the expected answer
				if ((strstr(answer,"OK\r") || strstr(answer,"ERROR")) && expect == NULL) {
					found=1;
				}
			
				if (expect && expect[0]) {
					if (strstr(answer,expect)) {
						sprintf(tmp, "Got expected %s (iteration %d)", answer, timeoutcounter);
						logger("DEBUG", tmp);
						found=1;
					}
				}
				    
				// if found then set timoutcounter to read only 0.1s after that and not more
				if (found) {
					//timeoutcounter=timeout-1;
					break;
				}
			}
		}
	}
	// repeat until timout
	while (timeoutcounter<timeout && count < max);

	if (getLog()) {
		char *a2 = answer;

		while (a2[0] == '\r' || a2[0] == '\n')  {
			a2++;
		}
		logger("DEBUG",a2);
	}

	return count;
}

//
// Support for stdin non-blocking IO (actually only Input)
//
// need a non-blocking IO because of another connection to front-end
// at the same time
//
#define NB_ENABLE  1
#define NB_DISABLE 0

static char stdinbuf[1024];
static int  cnt = 0;

static int kbhit()
{
	struct timeval tv;
        fd_set fds;
        tv.tv_sec = 0;
        tv.tv_usec = 0;
        FD_ZERO(&fds);
        FD_SET(STDIN_FILENO, &fds); //STDIN_FILENO is 0
        select(STDIN_FILENO+1, &fds, NULL, NULL, &tv);
        return FD_ISSET(STDIN_FILENO, &fds);
}

static void nonblock(int state)
{
    struct termios ttystate;

    //get the terminal state
    tcgetattr(STDIN_FILENO, &ttystate);

    if (state==NB_ENABLE)
    {
        //turn off canonical mode
        ttystate.c_lflag &= ~ICANON;
        //minimum of number input read.
        ttystate.c_cc[VMIN] = 1;
    }
    else if (state==NB_DISABLE)
    {
        //turn on canonical mode
        ttystate.c_lflag |= ICANON;
    }
    //set the terminal attributes.
    tcsetattr(STDIN_FILENO, TCSANOW, &ttystate);

}

void openStdin()
{
	//printf("openStdin\n");      
	nonblock(NB_ENABLE);
}

void closeStdin()
{
	//printf("closeStdin\n");      
	nonblock(NB_DISABLE);
}

int readStdin(char* buf, int max)
{
        int ch = '1';
        
        //nonblock(NB_ENABLE);
        
        while (kbhit() /*&& cnt < max*/) {
                //printf("readStdin kbhit() OK\n");      
                ch = getchar();
                //printf("readStdin got %d %c\n",cnt,ch);      
        	if (ch == '\n' || ch == EOF) break;
                
        	stdinbuf[cnt] = ch;
        	cnt++;
        }
        
        if (cnt == max || ch == '\n') {
        	stdinbuf[cnt] = '\0';
                strcpy(buf,stdinbuf);
                int i = cnt;
                cnt = 0;
                return i;
        }
         
	buf[0] = '\0';
        return 0;
} 

//
// Support for local server mode sockets
//
int unix_open_port(char* port) 
{
	sprintf(tmp, "unix_open_port >%s<", port);
	logger("INFO", tmp);
	
	struct sockaddr_un serveraddr;

	portfd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (portfd < 0) {
		sprintf(tmp, "can't open UNIX %s", port);
		logger("ERROR", tmp);
		printf("ERROR: can't open UNIX %s\n", port);
		portfd = 0;
		return -1;
	}
	memset(&serveraddr, 0, sizeof(serveraddr));
	serveraddr.sun_family = AF_UNIX;
      	strncpy(serveraddr.sun_path, port, sizeof serveraddr.sun_path - 1);

	int ret = connect(portfd, (struct sockaddr *)&serveraddr, SUN_LEN(&serveraddr));
	if (ret < 0) {
		sprintf(tmp, "can't open UNIX socket %s", port);
		logger("ERROR", tmp);
		printf("ERROR: can't open UNIX socket %s\n", port);
		portfd = 0;
		return -1;
	}

	return 0;
}

int unix_close_port() 
{
	int retval = 0;
	//printf("INFO: Close port\n");

	if (!portfd) { /* already closed */
		printf("INFO: Already closed ?\n");
		return -1;
	}

	retval = close(portfd);
	portfd = 0;

	return retval;
}
/*
int unix_read_port(char* buf, int l)
{
	int ret = 0;
	int bytesReceived = 0;
	printf("SOCKET: read fd %d\n", portfd);
	
	while (bytesReceived < l-1)
	{
		int rc = recv(portfd, & buf[bytesReceived], l - bytesReceived, 0);
		printf("unix_read_port %d\n",rc);
		if (rc < 0) {
	   		//logger("ERROR","recv() failed");
			ret = EOF-1;
	   		break;
		} else if (rc == 0) {
	   		//logger("INFO","Read 0 bytes");
			ret = EOF;
	   		break;
		}

		bytesReceived += rc;
		ret = bytesReceived;
	}
	buf[bytesReceived] = '\0';
	
	return ret;
}
*/
//
// Support server mode sockets
//

int openSocketPort(int type, int port, char *path) 
{
	struct sockaddr_in tcp_addr;
	struct sockaddr_un un_addr;
	#ifdef USE_BT
	struct sockaddr_rc bt_addr;
	#endif
	
	struct sockaddr*   socketaddr = NULL;

	int addFamily = 0;
	int proto     = 0;

	if (sportfd) { 
		logger("ERROR", "socket was already opened");
		//close(sportfd);
		//sportfd = 0;
		return 1;
	}

	if (type == SERVER_TCP) {
		addFamily = AF_INET;
		proto	  = IPPROTO_TCP;
	} else if (type == SERVER_BT) {
		#ifdef USE_BT
		addFamily = PF_BLUETOOTH;
		proto	  = BTPROTO_RFCOMM;
		#endif
	} else if (type == SERVER_UX) {
		addFamily = AF_UNIX;
		proto	  = 0;
	} else {
		logger("ERROR", "incorrect input");
		return -1;
	}

	if ((sportfd = socket(addFamily, SOCK_STREAM, proto)) < 0) {
		logger("ERROR", "opening socket");
		printf("ERROR: opening socket\n");
	
		return -1;
	}
	
	//socklen_t optlen = 0;
	//setsockopt(sportfd, SOL_SOCKET, SO_REUSEADDR, NULL,optlen);
	
	/*
	// Set non-blocking mode
	if (-1 == (oldflags = fcntl(sportfd, F_GETFL, 0))) {
	    oldflags = 0;
	}
	fcntl(sportfd, F_SETFL, oldflags | O_NONBLOCK);
	*/
	

	if (type == SERVER_TCP) {
	
		memset((void *) &tcp_addr, 0, sizeof(tcp_addr));
		tcp_addr.sin_family = AF_INET;
		tcp_addr.sin_addr.s_addr = INADDR_ANY;
		tcp_addr.sin_port = htons(port);
	
		socketaddr=(struct sockaddr *)&tcp_addr;
		
	} else if (type == SERVER_BT){	// SERVER_BT
		#ifdef USE_BT
		memset((void *) &bt_addr, 0, sizeof(bt_addr));
		
		// bind socket to the specified port of the first available local bluetooth adapter
		bt_addr.rc_family = AF_BLUETOOTH;
		bt_addr.rc_bdaddr = *BDADDR_ANY; 
		bt_addr.rc_channel = (uint8_t) port;

		char tmpstring[512];
		//sprintf(tmpstring, "sdptool add --channel=%i SP", port);
		//sprintf(tmpstring, "sdptool add --channel=%i SP;sdptool setattr `sdptool search --bdaddr local SP|grep \"Service RecHandle\"|tail -1|cut -f 3 -d \" \"` 0x100 anyRemote", port);
		sprintf(tmpstring, "bash -c \'A=`sdptool search --bdaddr local SP|grep \"Service Name\"|grep anyRemote|wc -l`; if [ \"x$A\" == \"x0\" ]; then sdptool add --channel=%i SP;sdptool setattr `sdptool search --bdaddr local SP|grep \"Service RecHandle\"|tail -1|cut -f 3 -d \" \"` 0x100 anyRemote; fi\'", port);
		system(tmpstring);
		sprintf(tmp, "registered SP for channel %i", port);
		logger("INFO", tmp);

		socketaddr=(struct sockaddr *)&bt_addr;
		#endif
	} else if (type == SERVER_UX && path != NULL){
	
		memset(&un_addr, 0, sizeof(un_addr));
		un_addr.sun_family = AF_UNIX;
		strncpy(un_addr.sun_path, path, sizeof un_addr.sun_path - 1);
		printf("ERROR: SOCKET %s\n", path);
		socketaddr=(struct sockaddr *)&un_addr;
	
	} else {
		logger("ERROR", "incorrect input 2");
		return -1;
	}

	if (bind(sportfd, (struct sockaddr *) socketaddr, sizeof(*socketaddr)) < 0) {
		logger("ERROR", "on binding");
		printf("ERROR: on binding %d->%s\n", errno, strerror(errno));
	
		return -1;
	}
	
	if (type == SERVER_UX) {
		uSocket = strdup(path);
	}
	return 1;
}

int closeSocketPort(int type) 
{
	logger("INFO", "closeSocketPort");

	if (portfd) { 
		close(portfd);
		portfd = 0;
	}
	if (sportfd) { 
		close(sportfd);
		sportfd = 0;
	}

	if (type == SERVER_UX) {
		unlink(uSocket);
		free(uSocket);
	}
	
	return 1;
}

int listenAndAcceptSocketConn(int type) 
{
	int cnt, sz;

	struct sockaddr* socketaddr = NULL;
	struct sockaddr_in ip_addr;

	#ifdef USE_BT
	struct sockaddr_rc bt_addr;
	bdaddr_t ba;
	#endif

	logger("INFO", "listenAndAcceptSocketConn");
	cnt = 0;

	if (type == SERVER_BT) {
		#ifdef USE_BT
		socketaddr=(struct sockaddr *)&bt_addr;
		sz = sizeof(bt_addr);
		#endif
	} else if (type == SERVER_TCP) {
		socketaddr=(struct sockaddr *)&ip_addr;
		sz = sizeof(ip_addr);
	}

	while (1) {	       
		listen(sportfd,5);
		 
		portfd = accept(sportfd, (struct sockaddr *) socketaddr, (socklen_t *)&sz);
		    	 
		if (portfd == -1 && errno == EAGAIN) {
			
			if (cnt >= 60) {    // Print to log every minute
				logger("INFO", "listenAndAcceptSocketConn: waiting for connection");
				//printf(".");
				cnt = 0;
			}
			fflush(stdout);
	
			sleep(1);
			cnt++;
	
			continue;
		}	 
		 
		if (portfd < 0) {
			logger("ERROR", "on accept");
			printf("ERROR: on accept %d\n", errno);
			return -1;
		/*} else {
			// Set non-blocking mode
			if (-1 == (flags = fcntl(portfd, F_GETFL, 0))) {
			    flags = 0;
			}
			fcntl(portfd, F_SETFL, flags | O_NONBLOCK);
		*/
		}
		logger("INFO", "listenAndAcceptSocketConn: accepted");

	
		#ifdef USE_BT
		if (type == SERVER_BT) {
			baswap(&ba, &bt_addr.rc_bdaddr);

			sprintf(tmp, "listenAndAcceptSocketConn: remote BT address is %s", batostr(&ba));
			logger("INFO", tmp);
		}
		#endif
	
		break;
	}
	return 1;
}

int writeSocketConn(char* command, int count)
{
	logger("DEBUG", "writeSocketConn");

	// send command
	if (count > 0) {
        
		memset(tmp, 0, MAXMAXLEN);
        	strcat(tmp, "writeSocketConn ");
                
                int logSz = (count > 256 ? 255 : count);
                
                // it is possible to get binary data here
                memcpy(tmp, command, logSz); // Do not dump long commands
                tmp[logSz] = '\0';
        	logger("DEBUG", tmp);
		
		sprintf(tmp, "writeSocketConn %d bytes", count);
		logger("INFO", tmp);

        	int n = write(portfd,command,count);        
        	if (n < 0) {
        		logger("ERROR", "error writing to socket");
        		return EXIT_NOK;
        	}
        	//tcdrain(portfd);
	}
	return EXIT_OK;
}

int writeBytesSocketConn(char* command) 
{
	//logger("DEBUG", "writeBytesSocketConn");

	// send command
	if (command && command[0]) {
		
		char byteStr[MAXCKPDLEN];
		strncpy(byteStr,command,MAXCKPDLEN-1);

		sprintf(tmp, "writeBytesSocketConn >%s<", byteStr);
		logger("DEBUG", tmp);

		char* bStr = strtok(byteStr,",");
		while (bStr != NULL) {
			
			//sprintf(tmp, "Next byte is >%s<", bStr);
			//logger("DEBUG", tmp);

			char bStripped[4];
		
			while (*bStr == ' ') {
				bStr++;
			}
			int i = 0;
			while (*bStr != ' ' && i < 3) {  // 0 < ... < 256
				bStripped[i] = *bStr;
				bStr++;
				i++;
			}
			bStripped[i] = '\0';

			sprintf(tmp, "Next byte is >%s<", bStripped);
			logger("DEBUG", tmp);
		
			unsigned char byte2write[2];
			byte2write[0] = (unsigned char) atoi(bStripped);
			byte2write[1] = '\0';
		
			if ( write(portfd, byte2write, 1) < 0) {
				logger("ERROR", "error writing to socket");
				return EXIT_NOK;
			}
			bStr = strtok(NULL,",");
		}
	}
	//logger("DEBUG", "writeBytesSocketConn EXIT");
	return EXIT_OK;
}

//
// Frontend support (ganyremote/kanyremote2) 
//

int sockfd_fe = 0;

int connectFrontEnd(int portno)
{
	int flags;
	sprintf(tmp, "connect to frontend >%d<",portno );
	logger("DEBUG", tmp);
	
	struct sockaddr_in serv_addr;
	struct hostent     *server;

	if (portno <= 0) {
		portno = 5050;
	}
	sockfd_fe = socket(AF_INET, SOCK_STREAM, 0);
	if (sockfd_fe < 0) {
		logger("ERROR", "opening socket for frontend");
		return EXIT_NOK;
	}
	
        if (-1 == (flags = fcntl(sockfd_fe, F_GETFL, 0)))
        	flags = 0;
     	fcntl(sockfd_fe, F_SETFL, flags | O_NONBLOCK);
	   
	server = gethostbyname("localhost");
	if (server == NULL) {
		return EXIT_NOK;
	}

	memset((void *) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	bcopy((char *)server->h_addr, (char *)&serv_addr.sin_addr.s_addr, server->h_length);
	serv_addr.sin_port = htons(portno);
	
	if (connect(sockfd_fe,(const struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) { 
		logger("ERROR", "connect socket for frontend");
		return EXIT_NOK;
	}
	
	return EXIT_OK;
}

int disconnectFrontEnd()
{
	if (sockfd_fe) { 
		close(sockfd_fe);
		sockfd_fe = 0;
	}
	return EXIT_OK;
} 

int sendToFrontEnd(char *buf)
{
	if (buf == NULL) {
		return EXIT_OK;
	}
	
	sprintf(tmp, "send to frontend >%s<", buf);
	logger("DEBUG", tmp);

	int n = write(sockfd_fe, buf,strlen(buf));
	if (n < 0) { 
		logger("ERROR", "ERROR writing to socket");
		return EXIT_NOK;
	}

	return EXIT_OK;
}

int readFromFrontEnd(char *buf, int size)
{
	//logger("INFO", "readFromFrontEnd");
	if (buf == NULL) {
		return EXIT_OK;
	}
 
        bzero(buf,size);
        int n = read(sockfd_fe,buf,size-1);
	if (n < 0) { 
		//logger("ERROR", "ERROR reading from socket");
		return EXIT_NOK;
	} else {
		*(buf+n) = '\0';
		if (n > 0) {
			sprintf(tmp, "Got from frontend (%d) >%s<", n, buf);
			logger("DEBUG", tmp);
			
			int i = 0;
			int k = 0;
			for (i=0;i<n;i++) {
				if (*(buf+i) != '\0') {
					if( i > k) {
						*(buf+k) = *(buf+i);
					} 
					k++;
				}
			}
			*(buf+k) = '\0';
			sprintf(tmp, "Got from frontend (%d) >%s<", n, buf);
			logger("DEBUG", tmp);
		}
	}
	
	return EXIT_OK;
}
