//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008,2009,2010 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <termios.h>

#include "parse.h"
#include "utils.h"

#define EMPTY_STR   	"EMPTY"

extern char tmp[MAXMAXLEN];

extern type_key* findExact(mode *mode, char *key);

extern mode       *modes;
extern mode       *currentMode;
extern mode       *internalMode;
extern mode 	  *defMode;
extern CONF       conf;

type_alarm *alarms = NULL;

/* ----------------- Interface functions ------------------------- */

// caller function must free data
static char* dupVarValue(char *name, int size)
{       
        varData* v = searchVar(name, size);
	if (v != NULL && v->value != NULL && v->value[0] != '\0') {
		char *d = calloc(1,v->size+1);
		strncpy(d,v->value,v->size);
		d[v->size] = '\0';
		return d;
	}
	return NULL;
}

int getLog()
{       
        varData* v = searchVar("Logging", 7);
        if (v != NULL && strncmp(v->value,"false",5) != 0) {
		return 1;
	}
	return 0;
}

int getDebug()
{
        varData* v = searchVar("Logging", 7);
        if (v != NULL && strncmp(v->value,"debug",5) == 0) {
		return 1;
	}
	return 0;
}

char* getDevice()
{
        return dupVarValue("Device", 6);
}

int getBaudrate()
{
	char br[16];	// enough for 234000 
	memset(br,0,16);
	
        varData* v = searchVar("Baudrate", 12);
        if (v != NULL) {
	        strncpy(br,v->value,v->size);
		
		int rate = atoi(br);
		
		switch (rate) {
			case 300:   return B300;
			case 1200:  return B1200;
			case 2400:  return B2400;
			case 9600:  return B9600;
			case 19200: return B19200;
			case 38400: return 38400;
			#ifdef B57600
			case 57600: return B57600;
			#endif
			#ifdef B115200
			case 115200: return B115200;
			#endif
			#ifdef B230400
			case 230400: return B230400;
			#endif
			default:
				WARNING2("bad baudrate %d, defaulting to 9600", rate);
		 		return B9600;
  		}
	}
	return B19200;
}

int getRetrySecs()
{
	char rs[16];	// should be enough 
	memset(rs,0,16);
        varData* v = searchVar("RetrySeconds", 12);
        if (v != NULL) {
	        strncpy(rs,v->value,v->size);
		return atoi(rs);
	}
	return 10;
}

int autoConnect()
{
        varData* v = searchVar(AUTOCONN_STR, 11);
        if (v != NULL && strncmp(v->value,"true",4) == 0) {
		return 1;
	}
	return 0;
}

int getIViewer()
{
        varData* v = searchVar("IViewer", 7);
        if (v != NULL && strncmp(v->value,"true",4) == 0) {
		return 1;
	}
	return 0;
}


int getAutoRepeat()
{
        varData* v = searchVar(AUTOREPEAT_STR, 10);
        if (v != NULL && strncmp(v->value,"true",4) == 0) {
		return 1;
	}
	return 0;
}

static type_key* repeatCMD = NULL;

void setRepeatNow(type_key* repeat) 
{
	repeatCMD = repeat;
}

type_key* repeatNow()
{
	return repeatCMD;
}

int getUseScreen()
{
        varData* v = searchVar("TwoWayComm", 10);
        if (v != NULL && strncmp(v->value,"true",4) == 0) {
		return 1;
	}
	return 0;
}

char* getCharset()
{
        return dupVarValue(CHARSET_STR, 7);
}

char* getToMainMenu()
{
        return dupVarValue("ToMainMenu", 10);
}

char* getServiceName()
{
        return dupVarValue("ServiceName", 11);
}

char* getAT_CMER(int what)
{
	if (what == CMER_ON) {
	        char *cmer = dupVarValue("CmerOn", 6);

		if  (cmer == NULL) {
			if (conf.model == MODEL_MOTOROLA) {
				return strdup(DEF_AT_CMER_ON_MOTOROLA);
			} else if (conf.model == MODEL_SE) {
				return strdup(DEF_AT_CMER_ON_SE);
			} else if (conf.model == MODEL_SAGEM) {
				return strdup(DEF_AT_CMER_ON_SAGEM);
			} else if (conf.model == MODEL_SIEMENS) {
				return strdup(DEF_AT_CMER_ON_SIEMENS);
			} else {
				return strdup(DEF_AT_CMER_ON_DEFAULT);
			}
		} else {
			return cmer;
		}
	}

	if (what == CMER_OFF) {
		char *cmer = dupVarValue("CmerOff", 7);
                
		if  (cmer == NULL) {
			if (conf.model == MODEL_MOTOROLA) {
				return strdup(DEF_AT_CMER_OFF_MOTOROLA);
			} else if (conf.model == MODEL_SE) {
				return strdup(DEF_AT_CMER_OFF_SE);
			} else if (conf.model == MODEL_SAGEM) {
				return strdup(DEF_AT_CMER_OFF_SAGEM);
			} else if (conf.model == MODEL_SIEMENS) {
				return strdup(DEF_AT_CMER_OFF_SIEMENS);
			} else {
				return strdup(DEF_AT_CMER_OFF_DEFAULT);
			}
		} else {
			return cmer;
		}
	}
	return NULL;
}

int getFrontEnd()
{
	return conf.frontEnd;
}

int getHttp()
{
	return conf.http;
}

int usePassword()
{
	return conf.pass;
}


char* getCfgDir(void) 
{
	return conf.cfgDir;
}

char *getMode (void) 
{
	return (currentMode == NULL ? NULL : currentMode->name);
}

void setModel (char *answer) 
{
	if (answer == NULL) {
		conf.model = MODEL_DEFAULT;
	} else if (strstr(answer,STR_MOTOROLA)) {
		conf.model = MODEL_MOTOROLA;
	} else if (strstr(answer,STR_SE)) {
		conf.model = MODEL_SE;
	} else if (strstr(answer,STR_SAGEM)) {
		conf.model = MODEL_SAGEM;
	} else if (strstr(answer,STR_SIEMENS)) {
		conf.model = MODEL_SIEMENS;
	} else {
		conf.model = MODEL_DEFAULT;
	}
}

int getModel (void) 
{
	return conf.model;
}

/////////////////////////////////////////////////////////////////////////////////////////////////

void addAlarm(char *file, char *macro)
{
	sprintf(tmp,"addAlarm() >%s< >%s<\n", file,macro);
	logger("DBG",tmp);
        
	type_alarm * v = (type_alarm *) calloc(sizeof(type_alarm),1);
        v->file  = strdup(file);
        v->macro = strdup(macro);
	
        // Insert in head
        if (alarms == NULL) { 		// Insert first
        	v->next = NULL;
        } else {
        	v->next = alarms;
        }
        alarms = v;
}

static type_alarm *tk = NULL;

char* getFirstAlarm(char** macro)
{
	if (alarms == NULL) {
	        //logger("DBG","no any alarm found");
        	return NULL;
        }
	tk = alarms;
    	
	*macro = tk->macro;
 	return tk->file;
}
	
char* getNextAlarm(char** macro)
{
        if (tk != NULL) {
        	tk = tk->next;
        }
        
        if (tk == NULL) {
        	return NULL;
        }
    	
	*macro = tk->macro;
 	return tk->file;
}

/////////////////////////////////////////////////////////////////////////////////////////////////

static type_key* findStartingWith(mode *mode, char *key)
{
        if (mode == NULL) {
        	return NULL;
        }
	type_key* It = mode->keys;

	while (It && It->key != NULL) {
		char *start = strstr(It->key,key);

		if (start && start == It->key) {    // We got a part of multi-key or parametrized command
			
			logger("DBG","Found part of multi-key or parametrized command");
			return It;
		} 
		It = (type_key*) It->next;
	}    

	return It;
}

static type_key* findItemInMode(mode *mode, char *key, int *flag, cmdParams *params) 
{
	if (mode == NULL) {
		logger("DBG","findItemInMode: mode is null");
		return NULL;
	}
	int canLog = (strcmp(mode->name,"_INTERNAL_") != 0);
	
	if(canLog) {
		sprintf(tmp,"findItemInMode >%s,%s<", mode->name, key);
		logger("DBG",tmp);
	}
	
	// Prepare to search as parametrized command, control presence of both ( and )
	char *start  = index(key,'(');
	char *finish = rindex(key,')');

	if ((start != NULL && finish == NULL) ||  // Command was read partially?
            (start == NULL && finish != NULL)) {  // Command was incorectrly formed ?
		if(canLog) {
			logger("DBG","findItemInMode: Incorrectly formed parametrized command. One brace is absent");
		}
		return NULL;
	}
        
	// Clean-up
	if (params != NULL) {
		params->index[0] = '\0';
		params->value[0] = '\0';
	}

	type_key* It = findExact(mode,key);	// Search exact command
	if (It) {
		//logger("DBG","Found exact command");
		*flag = FLAG_EXACT;
		return It;
	} 

	It = findStartingWith(mode,key);	// Search as part of multikey sequence
	if (It) {
		//logger("DBG","Found part of multi-key command");
		*flag = FLAG_MULTIKEY;
		return It;
	} 

	if (params == NULL) {
		if(canLog) {
			logger("DBG","findItemInMode: No parameters suspected. Item does not found.");
		}
		return NULL;
	}

	// Search as parametrized command

	if (start != NULL && finish != NULL && start != key) {        // Do not match "(...)=..."
		It = mode->keys;
	
		char tag  [MAXARGLEN];memset(tag,  0,MAXARGLEN);
		char index[6];        memset(index,0,6);
		char value[MAXARGLEN];memset(value,0,MAXARGLEN);
	
		strncpy(tag, key, start-key);
		*(tag+(start-key)) = '\0';
	
		char *comma = strstr(start+1,",");
		if (comma == NULL || comma > finish) { // Parametrized command from Java Client should be in a form like List(1,String1)
			if(canLog) {
				logger("DBG","findItemInMode: Incorrectly formed parametrized command");
			}
			return NULL;
		}

		if (comma-start > 6) { // 65535 = max value in Comman Fusion iViewer
			if(canLog) {
				logger("ERR","findItemInMode: Received incorrect index!");
			}
			return NULL;
		}
		strncpy(index,start+1,comma-start-1);
		*(index+(comma-start-1)) = '\0';

		strncpy(value,comma+1,finish-comma-1);
		*(value+(finish-comma-1)) = '\0';

		if(canLog) {
			sprintf(tmp,"Parametrized command parsed as >%s< >%s< >%s< ", tag,index,value);
			logger("DBG",tmp);
		}
		
		// Try to search explicitly specified parametrized command (for ex List(1) or List(commandX))
		// By index
		strcat(tag,"(");
		strcat(tag,index);
		strcat(tag,")");
		It = findExact(mode,tag);

		if (It) {
			if(canLog) {
				logger("DBG","Found exact (explicitly specified by index) parametrized command");
			}
			*flag = FLAG_EXACT;
			return It;
		} 

		// Then by value
		*(tag+(start-key+1)) = '\0';
		strcat(tag,value);
		strcat(tag,")");
		It = findExact(mode,tag);

		if (It) {
			if(canLog) {
				logger("DBG","Found exact (explicitly specified by value) parametrized command");
			}
			*flag = FLAG_EXACT;
			return It;
		} 
	
		// Finally... Search command like ListItem($$)
		*(tag+(start-key)) = '\0';
		
		char* keyAndBrace = calloc(1,strlen(tag)+3);
		strcpy(keyAndBrace,tag);
		strcat(keyAndBrace,"($");
		
                if(canLog) {
			DEBUG2("Search findStartingWith() >%s< ", keyAndBrace);
                }
		It = findStartingWith(mode,keyAndBrace);
		free(keyAndBrace);
		if (It) {
			if(canLog) {
	                	DEBUG2("Found parametrized command >%s,%s< ", index,value);
			}
			strcpy(params->index, index);
			strcpy(params->value, value);
	
			*flag = FLAG_PARAMETR;
			return It;
		} 
	}
	if(canLog) {
		logger("DBG","findItemInMode: Item does not found");
	}
	return NULL;
}

type_key* findItem(char *keyIn, int *flag, cmdParams *params) 
{
	//logger("DBG","findItem()"); 
	type_key *tk = NULL;

	if (keyIn == NULL) {
        	return NULL;
        }

	char key[MAXARGLEN];
	memset(key,0,MAXARGLEN);

	if (*keyIn == '\0') {
		logger("INF","Got empty key");   
		strcpy(key, EMPTY_STR);
	} else {
		strncpy(key, keyIn, MAXARGLEN - 1);
	}
	 
	if (currentMode != NULL) {
		tk = findItemInMode(currentMode, key, flag, params);
		if (tk == NULL && currentMode != defMode) {
			logger("DBG","findItem() search in default");
			tk = findItemInMode(defMode, key, flag, params);
		}
	}
	
	//  Search in "internal" commands
	if (tk == NULL) {
	        logger("DBG","findItem() internal mode");
		tk = findItemInMode(internalMode, key, flag, params);
		if (tk != NULL && *flag == FLAG_MULTIKEY) {
			logger("DBG","findItem() multikey in internal command ?");
		}
	}
	
        /*if (tk && tk->cmd) {
        	printf("TRACE %s->%s:%s \n",keyIn,tk->cmd->descr,(tk->cmd->exec ? tk->cmd->exec : "NULL"));
        } else {
        	printf("TRACE %s->NOT FOUND?\n",keyIn);
        }*/
	return tk;
}

cmdItem *getCommand(type_key* item)
{
	if (!item) {
		logger("DBG","getCommand(): Empty type_key was got");
		return NULL;
	}
	if (item->key == NULL) {
		logger("DBG","Empty key was got");
		return NULL;
	}

	sprintf(tmp,"getCommand for >%s<", item->key);
	logger("DBG",tmp);   

	return item->cmd;
}

//
// Print cfg stuff
//

void printKeysInMode(type_key* It)
{    
        while (It && It->key != NULL) {
        	if (It->cmd) {
    			sprintf(tmp, "\t%s = ", It->key);
                        
        		cmdItem* item = It->cmd;
        		while (item) { 
        			
                                strcat(tmp, id2Cmd(item->type));
                                
                                if (item->tdata || item->descr || item->exec) {
        				strcat(tmp, "(");
                                }
                                
                                int pComma = 0;
        			if (item->descr) {
                                	//printf("descr  %s\n",item->descr);
        				strcat(tmp, item->descr);
					pComma = 1;
                                }
                                if (item->tdata ) {
                                        char iBuff[16];
                                        sprintf(iBuff,",%d",item->tdata->timeout);
                               		strcat(tmp, iBuff);
                                        sprintf(iBuff,",%d",item->tdata->times);
                               		strcat(tmp, iBuff);
                                        strcat(tmp, ",");
                                }
        			strcat(tmp, "<|>");
                                if (item->exec) {
                                        if (pComma) {
						strcat(tmp, ",");
        				}
					strcat(tmp, item->exec);
                                }
                                
                                if (item->tdata || item->descr || item->exec) {
        				strcat(tmp, ")");
        			}
        			strcat(tmp, ";");
        			item = item->next;
                                
                                if (item) {
        				strcat(tmp, "\\");
                                }
                                logger("CFG", tmp);
                                sprintf(tmp, "\t\t");
        	
        		}
    		} else {
        		sprintf(tmp, "\t%s = no command", It->key);
                        logger("CFG", tmp);
        	}    
	
		It = (type_key*)It->next;
        }
        return;
}

void printKeys()
{    
	mode *mp = modes;
	while (mp) {
		
		sprintf(tmp, "%s %s", MODE_STR, mp->name);
		logger("CFG", tmp);
		 
		type_key* It = mp->keys;
	
		printKeysInMode(It);

		sprintf(tmp, "%s %s", MODE_END_STR, mp->name);
		logger("CFG", tmp);
	
		mp = (mode*)mp->next;
	}

	return;
}

void printConf() 
{
	//printf("printConf ENTER\n");fflush(stdout);
	printTime();
        sprintf(tmp, "anyRemote v%s", PACKAGE_VERSION);
        logger("CFG", tmp);
	printKeys   ();
	printVars   ();

	return ;
}

//
// Clean-up stuff
//

void freeCmds(cmdItem* item)
{
	cmdItem* mi;
        while (item) {
        	mi = item->next;
        	if (item->descr != NULL) {
        		free(item->descr);
        		item->descr = NULL;
        	}
         	if (item->exec != NULL) {
        		free(item->exec);
        		item->exec = NULL;
        	}
         	if (item->tdata != NULL) {
        		free(item->tdata);
        		item->tdata = NULL;
        	}
		
		free(item);
        	item = mi;
        }
	return;
}

void freeCfgEx(mode* pmodes, type_alarm* palarms)
{
	//logger("DBG", "freeCfgEx()");
	
	// Keys
	while(pmodes) {
		mode	 *md = pmodes;
		//printf("mode -> %s\n",md->name);
		type_key *mk = md->keys;
	
		while (mk) {
			type_key* It = mk;
			mk = (type_key*) mk->next;
			
                        freeCmds(It->cmd);
 
 			It->cmd = NULL;
                        free(It->key);
			free(It);
		}
		pmodes = (mode*) pmodes->next;
		free(md);
	}
	        
	// Alarms
        type_alarm *mk = palarms;
	while (mk) {
		type_alarm* It = mk;
		mk = (type_alarm*) mk->next;
                free(It->file);
                free(It->macro);
		free(It);
	}
}

void freeCfg(void)
{
	freeCfgEx(modes, alarms);
        
        // special handling for "internal" mode
        if (internalMode != NULL) {
		type_key *mki = internalMode->keys;
		while (mki) {
			type_key* It = mki;
			mki = (type_key*) mki->next;
		
                	freeCmds(It->cmd);
 
 			It->cmd = NULL;
                	free(It->key);
			free(It);
		}
		free(internalMode);
		internalMode = NULL;
        }
        
        free(conf.cfgDir);
}
