

#define LOCAL_DEBUG
#include "debug.h"

#include "acfg.h"
#include "meta.h"
#include "filereader.h"
#include "fileio.h"

#include <regex.h>

#include <iostream>
#include <fstream>
#include <string>
#include <deque>
#include <map>
#include <algorithm>

using namespace MYSTD;

namespace rechecks
{
bool CompileExpressions();
bool CompileUncExpressions(const string & req, const string & tgt);
}


namespace acfg {

// internal stuff:
char alphabet[65] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
string sPopularPath("/debian/");

string dontcache, dontcacherq, dontcachetgt;

tStringMap localdirs;
NoCaseStringMap mimemap;

MapNameToString n2sTbl[] = {
		{  "Port", 			&port , 0}
		,{ "CacheDir", 	&cachedir, 0 }
		,{ "LogDir", 	&logdir , 0}
		,{ "SocketPath", 	&fifopath, 0}
		,{ "PidFile", 	&pidfile, 0}
		//,{ "Proxy",		&proxy, 0}
		,{ "ReportPage",&reportpage, 0}
		,{ "VfilePattern", &vfilepat, 0}
		,{ "PfilePattern", &pfilepat, 0}
		,{ "WfilePattern", &wfilepat, 0}
		,{ "AdminAuth",  &adminauth, 0}
		,{ "BindAddress", &bindaddr, 0}
		,{ "UserAgent", &agentname, 0}
		,{ "DontCache",	&dontcache, 0}
		,{ "DontCacheRequested",	&dontcachetgt, 0}
		,{ "DontCacheResolved",	&dontcacherq, 0}
};

MapNameToInt n2iTbl[] = {
		{ "Debug", 		&debug, 0 , 10}
		,{ "OfflineMode", 	&offlinemode , 0, 10}
		,{ "ForeGround", 	&foreground , 0, 10}
		,{ "Verbose", 		0, "Option is deprecated, ignoring the value." , 10}
		,{ "ForceManaged", 	&forcemanaged , 0, 10}
		,{ "StupidFs", 		&stupidfs , 0, 10}
		,{ "VerboseLog",	&verboselog , 0, 10}
		,{ "ExTreshold",	&extreshhold, 0, 10}
		,{ "MaxSpareThreadSets",	&tpstandbymax, "Deprecated option name, mapped to MaxStandbyConThreads", 10}
		,{ "MaxStandbyConThreads",	&tpstandbymax, 0, 10}
		,{ "MaxConThreads",	&tpthreadmax, 0, 10}
		,{ "DnsCacheSeconds", &dnscachetime, 0, 10}
		,{ "UnbufferLogs", &debug , 0, 10}
		,{ "ExAbortOnProblems", &exfailabort, 0, 10}
		,{ "ExposeOrigin", &exporigin, 0, 10}
		,{ "LogSubmittedOrigin", &logxff, 0, 10}
		,{ "OldIndexUpdater", &oldupdate, "Option is deprecated, ignoring the value." , 10}
		,{ "RecompBz2", &recompbz2, 0, 10}
		,{ "NetworkTimeout", &nettimeout, 0, 10}
		,{ "MinUpdateInterval", &updinterval, 0, 10}
		,{ "ForwardBtsSoap", &forwardsoap, 0, 10}

		,{ "DirPerms", &dirperms, 0, 8}
		,{ "FilePerms", &fileperms, 0, 8}
};

#define _iterPos(it, start) (it-start.begin())/sizeof(it)
#define sProblemLoc szPath<< ':'<< _iterPos(it, lines)

string sFilterSet(SPACECHARS "#");
#define IsValidButIrrelevantLine(x) (x.empty() || stmiss != sFilterSet.find(x[0]))
#define BARF(x) { cerr << x << endl; exit(EXIT_FAILURE); }

void _ReadRewriteFiles(const string & sFile, const string & sRepName);
void _ReadBackendsFiles(const string & sFile, const string &sRepName);


struct fct_lt_host
{
  bool operator()(const tHttpUrl &a, const tHttpUrl &b) const
  {
    return strcasecmp(a.sHost.c_str(), b.sHost.c_str()) < 0;
  }
};
typedef multimap<tHttpUrl,const string*, fct_lt_host> tMapUrl2StringPtr; 
typedef tMapUrl2StringPtr::iterator tUrl2RepIter;
tMapUrl2StringPtr mapUrl2pVname;

typedef map<const string, tHostiVec> tMapString2Hostivec;
tMapString2Hostivec mapRepName2Backends;

string * _GetStringPtr(const string &key)  {
	for(unsigned int i=0; i<_countof(n2sTbl); i++) {
		if(0==strcasecmp(key.c_str(), n2sTbl[i].name))
		{
			if(n2sTbl[i].warn)
				cerr << "Warning, " << key << ": " << n2sTbl[i].warn << endl;

			return n2sTbl[i].ptr;
		}
	}
	return NULL;
}

int * _GetIntPtr(const string &key, int &base)  {
	for(unsigned int i=0; i<_countof(n2iTbl); i++) {
		if(0==strcasecmp(key.c_str(), n2iTbl[i].name))
		{
			if(n2iTbl[i].warn)
				cerr << "Warning, " << key << ": " << n2iTbl[i].warn << endl;
			base = n2iTbl[i].base;
			return n2iTbl[i].ptr;
		}
	}
	return NULL;
}

inline void _FixPostPreSlashes(string &val)
{
	// fix broken entries

	if (val.empty() || val.at(val.length()-1) != '/')
		val.append("/");
	if (val.at(0) != '/')
		val.insert(0, "/", 1);
}

inline bool _ReadMainConfiguration(const string & szFilename)
{
	filereader reader;
	reader.OpenFile(szFilename);
	reader.CheckGoodState(true);
	string sLine, key, val;
	for (bool bNotEof=true; bNotEof;)
	{
		bNotEof=reader.GetOneLine(sLine);
		if (IsValidButIrrelevantLine(sLine))
			continue;
#ifdef DEBUG
		cerr << sLine <<endl;
#endif
		if(! SetOption(sLine))
			BARF("Error reading main options, terminating.");
		// XXX: move that to SetOption... if (debug>4)
		//	cout << key << " -> "<< val <<endl;
	}
	return true;
}

const string * _CheckBEentryGetNamePtr(const string & sRepName)
{
	// needs a reliably stored string for the pointer. Backend description may not exist,
	// then create a dummy one with no contents and point at this string. Iterators on map are
	// guaranteed to be stable so point at its key variable.
	tMapString2Hostivec::iterator
			itHostiVec = mapRepName2Backends.find(sRepName);
	if (itHostiVec == mapRepName2Backends.end())
	{
		mapRepName2Backends[sRepName]=tHostiVec(0);
		itHostiVec=mapRepName2Backends.find(sRepName); // must refer to that string on the heap
		if (debug>4)
			cout << "created empty backend entry for "<< sRepName <<endl;
	}
	return & itHostiVec->first;
}

inline void _AddRemapInfo(bool bAsBackend, const string & token,
		const string &repname)
{
	if (0!=token.compare(0, 5, "file:"))
	{
		tHttpUrl url;
		if(! url.SetHttpUrl(token))
			BARF(token + " <-- bad URL detected");
		_FixPostPreSlashes(url.sPath);
		
		if (bAsBackend)
			mapRepName2Backends[repname].push_back(url);
		else
			mapUrl2pVname.insert(pair<tHttpUrl,const string*>(
					url, _CheckBEentryGetNamePtr(repname)));
	}
	else
	{
		string sPath=token.substr(5);
		if (sPath.empty())
			BARF("Bad file spec for repname, file:?");
		
		MakeAbsolutePath(sPath, confdir);
		
		tStrDeq srcs=ExpandFilePattern(sPath, true);
		for(tStrDeq::const_iterator it=srcs.begin(); it!=srcs.end(); it++)
		{
			if (bAsBackend)
				_ReadBackendsFiles(*it, repname);
			else
				_ReadRewriteFiles(*it, repname);
		}
	}
}

inline string _GetBase64Auth(const string & sUserColonPass)
{
	int cols, bits, c, char_count;

	char_count = 0;
	bits = 0;
	cols = 0;
	tStrPos pos(0);
	string out;
	while ( pos<sUserColonPass.size())
	{
		c=sUserColonPass[pos++];
		bits += c;
		char_count++;
		if (char_count == 3)
		{
			out+=(alphabet[bits >> 18]);
			out+=(alphabet[(bits >> 12) & 0x3f]);
			out+=(alphabet[(bits >> 6) & 0x3f]);
			out+=(alphabet[bits & 0x3f]);
			cols += 4;
			bits = 0;
			char_count = 0;
		}
		else
		{
			bits <<= 8;
		}
	}
	if (char_count != 0)
	{
		bits <<= 16 - (8 * char_count);
		out+=(alphabet[bits >> 18]);
		out+=(alphabet[(bits >> 12) & 0x3f]);
		if (char_count == 1)
		{
			out+=('=');
			out+=('=');
		}
		else
		{
			out+=(alphabet[(bits >> 6) & 0x3f]);
			out+=('=');
		}
	}
	return out;
}

bool SetOption(const string &sLine, bool bQuiet)
{
	string::size_type posCol = sLine.find(":");
	string::size_type posEq = sLine.find("=");
	if (posEq==stmiss && posCol==stmiss)
	{
		if(!bQuiet)
			cerr << "Not a valid configuration directive: " << sLine <<endl;
		return false;
	}
	string::size_type pos;
	if (posEq!=stmiss && posCol!=stmiss)
		pos=min(posEq,posCol);
	else if (posEq!=stmiss)
		pos=posEq;
	else
		pos=posCol;

	string key=sLine.substr(0, pos);
	string value=sLine.substr(pos+1);
	trimString(key);
	trimString(value);
	if(key.empty())
		return false; // weird
	
	string * psTarget;
	int * pnTarget;
	int nNumBase(10);

	if ( NULL != (psTarget = _GetStringPtr(key)))
		*psTarget=value;
	else if ( NULL != (pnTarget = _GetIntPtr(key, nNumBase)))
	{
		const char *pStart=value.c_str();
		if(! *pStart)
		{
			cerr << "Missing value for " << key << " option!" <<endl;
			return false;
		}
		
		errno=0;
		char *pEnd(0);
		long nVal = strtol(pStart, &pEnd, nNumBase);

		if(RESERVED_DEFVAL == nVal)
		{
			cerr << "Bad value for " << key << " (protected value, use another one)" <<endl;
			return false;
		}

		*pnTarget=nVal;

		if (errno)
		{
			cerr << "Invalid number for " << key << " ";
			perror("option");
			return false;
		}
		if(*pEnd)
		{
			cerr << "Bad value for " << key << " option or found trailing garbage: " << pEnd <<endl;
			return false;
		}
	}

	else if(0==strcasecmp(key.c_str(), "Proxy"))
	{
		if (!proxy_info.SetHttpUrl(value))
		{
			cerr << "Invalid proxy specification, aborting..." << endl;
			exit(EXIT_FAILURE);
		}
		tStrPos pos = proxy_info.sHost.find('@');
		if (stmiss != pos)
		{
			proxy_info.sPath = string("Proxy-Authorization: Basic ")
					+ _GetBase64Auth(value.substr(0, pos)) + "\r\n";
			proxy_info.sHost.erase(0, pos + 1);
		}
		else
			proxy_info.sPath.clear(); // no auth at all, stop confusion

		if (proxy_info.sHost.empty())
		{
			cerr << "Invalid proxy specification, aborting..." << endl;
			exit(EXIT_FAILURE);
		}
		if (proxy_info.sPort.empty())
		{
			if (proxy_info.sPath.empty()) // guess unless there is any risk...
			{
				cerr << "Warning, unknown proxy port, assuming 80." << endl;
				proxy_info.sPort = "80";
			}
			else
			{
				cerr << "Error, unknown proxy port!" << endl;
				exit(EXIT_FAILURE);
			}
		}
	}
	else if(0==strcasecmp(key.c_str(), "LocalDirs"))
	{
		tStrVec tokens;
		Tokenize(value, ";", tokens);
		for(tStrVecIter it=tokens.begin(); it!=tokens.end(); it++)
		{
			trimString(*it);
			tStrPos pos = it->find_first_of(SPACECHARS);
			if(stmiss == pos)
			{
				cerr << "Cannot map " << *it << ", needed format: virtualdir realdir, ignoring it";
				continue;
			}
			string from=it->substr(0, pos);
			trimString(from, "/");
			string what=it->substr(pos);
			trimString(what, SPACECHARS "'\"");
			if(what.empty())
			{
				cerr << "Unsupported target of " << from << ": " << what << ", ignoring it" << endl;
				continue;
			}
			localdirs[from]=what;
		}

		filereader mreader;
		tStrVec toks;
		string line;
		for(mreader.OpenFile("/etc/mime.types"); mreader.CheckGoodState(false); )
		{
			if(!mreader.GetOneLine(line))
				break;
			if(IsValidButIrrelevantLine(line))
				continue;

			if(Tokenize(line, SPACECHARS, toks, false)>1)
			{
				for(UINT i=1; i<toks.size();i++)
					mimemap[toks[i]]=toks[0];
			}
		}

		return !localdirs.empty();
	}
	else if(0==strncasecmp(key.c_str(), "Remap-", 6))
	{
		string vname=key.substr(6, key.npos);
		tStrVec tokens;
		
		Tokenize(value, SPACECHARS, tokens);
		if(tokens.empty() || vname.empty())
		{
			if(!bQuiet)
				cerr << "Found invalid entry, ignoring " << key << ": " << value <<endl;
			return false;
		}
		bool bIsBackend=false;
		for(UINT i=0; i<tokens.size(); i++)
		{
			if(tokens[i]==";")
			{
				bIsBackend=true;
				continue;
			}
			if(startsWithSz(tokens[i], "#"))
				break;
			_AddRemapInfo(bIsBackend, tokens[i], vname);
		}
		
	}
	else
	{
		if(!bQuiet)
			cerr << "Warning, unknown configuration directive: " << key <<endl;
		return false;
	}
	return true;
}


//const string * GetVnameForUrl(string path, string::size_type * nMatchLen)
const string * GetRepNameAndPathResidual(const tHttpUrl & in, string & sRetPathResidual)
{
	sRetPathResidual.clear();
	
	pair<tUrl2RepIter,tUrl2RepIter> range=mapUrl2pVname.equal_range(in);
	if(range.first==range.second)
		return NULL;
	
	tStrPos bestMatchLen(0);
	string const * psBestHit(NULL);
		
	for (tUrl2RepIter & it=range.first; it!=range.second; it++)
	{
		// rewrite rule path must be a real prefix
		// it's also surrounded by /, ensured during construction
		const string & prefix=it->first.sPath;
		tStrPos len=prefix.length();
		if (in.sPath.size() > len && 0==in.sPath.compare(0, len, prefix))
		{
			if (len>bestMatchLen)
			{
				bestMatchLen=len;
				psBestHit=it->second;
			}
		}
	}
		
	if(psBestHit) sRetPathResidual=in.sPath.substr(bestMatchLen);
	return psBestHit;
	
}

tHostiVec * GetBackendVec(const string * vname)
{
	if(!vname)
		return NULL;
	tMapString2Hostivec::iterator it=mapRepName2Backends.find(*vname);
	if(it==mapRepName2Backends.end() || it->second.empty())
		return NULL;
	return & it->second;
}


void _ReadBackendsFiles(const string & sFile, const string &sRepName)
{

	int nAddCount=0;
	string sLine, key, val;
	tHttpUrl entry;

	filereader reader;
	reader.OpenFile(sFile.c_str());

	if(debug>4)
		cout << "Reading backend file: " << sFile <<endl;
	if(!reader.CheckGoodState(false))
	{
		if(debug>4)
			cout << "No backend data found, file ignored."<<endl;
		goto try_read_default;
	}
	
	
	for(bool bNotEof=true;bNotEof;)
	{
		bNotEof=reader.GetOneLine(sLine);
		//if(debug)
		//	cerr << "backends, got line: " << sLine <<endl;
		

		if( (startsWithSz(sLine, "http://") && entry.SetHttpUrl(sLine) )
				||
			(IsValidButIrrelevantLine(sLine) 
					&& ! entry.sHost.empty() 
					&& ! entry.sPath.empty()
			)
		)
		{
			_FixPostPreSlashes(entry.sPath);
#ifdef DEBUG
			cerr << "Backend: " << sRepName << " <-- " << entry.ToURI() <<endl;
#endif		

			mapRepName2Backends[sRepName].push_back(entry);
			nAddCount++;
			entry.clear();
		}
		else if(_ParseLine(sLine, key, val))
		{
			if(keyEq("Site", key))
				entry.sHost=val;
			/* TODO: not supported yet, maybe add later - push to a vector of hosts and add multiple later
			if(keyEq("Aliases", key))
			{
				val+=" ";
				for(string::size_type posA(0), posB(0);
					posA<val.length();
					posA=posB+1)
				{
					posB=val.find_first_of(" \t\r\n\f\v", posA);
					if(posB!=posA)
						hosts.push_back(val.substr(posA, posB-posA));
				}
			}
			*/
			else if(keyEq("Archive-http", key) || keyEq("X-Archive-http", key))
			{
				entry.sPath=val;
			}
		}
		else if(bNotEof && !IsValidButIrrelevantLine(sLine))
		{
			cerr << "Bad backend description, around line "
					<< reader.GetPositionDescription() << endl;
			exit(2);
		}
	}

try_read_default:
	if(nAddCount || endsWithSzAr(sFile, ".default"))
		return;
	if(debug>4)
		cout << "Trying to read replacement from " << sFile << ".default" <<endl;
	_ReadBackendsFiles(sFile+".default", sRepName);

}

void ShutDown()
{
	mapUrl2pVname.clear();
	mapRepName2Backends.clear();
}

void _ReadRewriteFiles(const string & sFile, const string & sRepName)
{

	filereader reader;
	if(debug>4)
		cout << "Reading rewrite file: " << sFile <<endl;
	reader.OpenFile(sFile.c_str());
	reader.CheckGoodState(true);

	tStrVec hosts, paths;
	string sLine, key, val;
	tHttpUrl url;
	
	for(bool bNotEof=true; bNotEof; )
	{
		bNotEof=reader.GetOneLine(sLine);

		if (0==sLine.compare(0, 7, "http://"))
		{ // TODO: The check above is optional since SetHttpUrl does that too, but it's 
			// more userfriendly on errors (more exact error message)
			if (url.SetHttpUrl(sLine))
			{
				_FixPostPreSlashes(url.sPath);
				pair<tHttpUrl, const string*> info(url,
						_CheckBEentryGetNamePtr(sRepName));
				mapUrl2pVname.insert(info);
			}
			else
			{
				cout << "Parse error, invalid URL" << sLine << " on line "
						<< reader.GetPositionDescription() <<endl;
				exit(2);
			}
			continue;
		}
		else if (IsValidButIrrelevantLine(sLine)) // end of block, eof, ... -> commit it
		{
			if (hosts.empty() && paths.empty())
				continue; // dummy run
			if ( !hosts.empty() && paths.empty())
			{
				cerr << "Warning, missing path spec for the site " << hosts[0] <<", ignoring mirror."<< endl;
				continue;
			}
			if ( !paths.empty() && hosts.empty())
			{
				cout << "Parse error, missing Site: field around line "
						<< reader.GetPositionDescription() <<endl;
				exit(2);
			}
			for (tStrVecIterConst itHost=hosts.begin();
			itHost!=hosts.end(); 
			itHost++)
			{
				for (tStrVecIterConst itPath=paths.begin();
				itPath!=paths.end(); 
				itPath++)
				{
					//mapUrl2pVname[*itHost+*itPath]= &itHostiVec->first;
					tHttpUrl url;
					url.sHost=*itHost;
					url.sPath=*itPath;
					pair<tHttpUrl,const string*> info(url, _CheckBEentryGetNamePtr(sRepName));
					mapUrl2pVname.insert(info);

#ifdef DEBUG
						cout << "Mapping: "<< *itHost << *itPath 
						<< " -> "<< sRepName <<endl;
#endif
				}
			}
			hosts.clear();
			paths.clear();
			continue;
		}
		else if(!_ParseLine(sLine, key, val))
		{
			cerr << "Error parsing rewrite definitions, around line " << reader.GetPositionDescription() <<endl;
			exit(1);
		}
		
		// got something, intepret it...
		if( keyEq("Site", key) || keyEq("Alias", key) || keyEq("Aliases", key))
			Tokenize(val, SPACECHARS, hosts, true);
		
		if(keyEq("Archive-http", key) || keyEq("X-Archive-http", key))
		{
			// help STL saving some memory
			if(sPopularPath==val)
				paths.push_back(sPopularPath);
			else
			{
				_FixPostPreSlashes(val);
				paths.push_back(val);
			}
			continue;
		}
	}
}



void ReadConfigDirectory(const char *szPath)
{
	// TODO: early abort when the dir does not exist!

	char buf[PATH_MAX];
	if(!realpath(szPath, buf))
  {
     cerr << "Failed to open config directory" << endl;
     exit(42);
  }
	confdir=buf; // pickup the last config directory

#if defined(HAVE_WORDEXP) || defined(HAVE_GLOB)
	tStrDeq srcs=ExpandFilePattern(confdir+SZPATHSEP"*.conf", true);
	for(tStrDeq::const_iterator it=srcs.begin(); it!=srcs.end(); it++)
		_ReadMainConfiguration(*it);
#else
	_ReadMainConfiguration(confdir+SZPATHSEP"acng.conf");
#endif
}

void PostProcConfig() 
{
	if(stupidfs)
		cerr << "Warning, support for primitive filesystems is not completely implemented!\n";
	
	// let's also apply the umask to the directory permissions
	{
		mode_t mask = umask(0);
		umask(mask); // restore it...
		dirperms &= ~mask;
		fileperms &= ~mask;
	}

    // postprocessing

#ifdef FORCE_CUSTOM_UMASK
	if(!sUmask.empty())
	{
		mode_t nUmask=0;
		if(sUmask.size()>4)
		{
			cerr << "Invalid umask length\n";
			exit(EXIT_FAILURE);
		}
		for(unsigned int i=0; i<sUmask.size(); i++)
		{
			unsigned int val = sUmask[sUmask.size()-i-1]-'0';
			if(val>7)
			{
				cerr << "Invalid umask value\n" <<endl;
				exit(EXIT_FAILURE);
			}
			nUmask |= (val<<(3*i));
		
		}
		//cerr << "Got umask: " << nUmask <<endl;
		umask(nUmask);
	}
#endif
	
   if(cachedir.empty() || cachedir[0] != CPATHSEP) {
      cerr << "Cache directory unknown or not absolute, terminating..." <<endl;
      exit(EXIT_FAILURE);
   }
   
   if(!rechecks::CompileExpressions())
   {
	   cerr << "An error occured while compiling file type regular expression!" <<endl;
	   exit(EXIT_FAILURE);
   }
   
   if(acfg::tpthreadmax < 0)
	   acfg::tpthreadmax = numeric_limits<int>::max();
	   
   // get rid of duplicated and trailing slash(es)
	for(tStrPos pos; stmiss != (pos = cachedir.find(SZPATHSEP SZPATHSEP )); )
		cachedir.erase(pos, 1);

   if(!pidfile.empty() && pidfile.at(0) != CPATHSEP)
   {
	   cerr << "Pid file path must be absolute, terminating..."  <<endl;
	         exit(EXIT_FAILURE);
   }
   
   if(!acfg::agentname.empty())
	   acfg::agentheader=string("User-Agent: ")+acfg::agentname + "\r\n";
   
   if(!adminauth.empty())
	   adminauth=string("Basic ")+_GetBase64Auth(adminauth);
   
   // create working paths before something else fails somewhere
   if(!fifopath.empty())
	   mkbasedir(acfg::fifopath);
   if(!cachedir.empty())
	   mkbasedir(acfg::cachedir);
   if(! pidfile.empty())
	   mkbasedir(acfg::pidfile);

   if(nettimeout < 5) {
	   cerr << "Warning, NetworkTimeout too small, assuming 5." << endl;
	   nettimeout = 5;
   }

   if(RESERVED_DEFVAL == forwardsoap)
	   forwardsoap = !forcemanaged;


   if(!rechecks::CompileUncExpressions(dontcachetgt.empty() ? dontcache : dontcachetgt,
		   dontcacherq.empty() ? dontcache : dontcacherq))
   {
	   cerr << "An error occured while compiling regular expression for non-cached paths!" <<endl;
	   exit(EXIT_FAILURE);
   }

   if (acfg::debug>3)
	{
		for (UINT i=0; i<_countof(n2sTbl); i++)
			if(n2sTbl[i].ptr)
				cout << n2sTbl[i].name << " = " << *(n2sTbl[i].ptr) <<endl;

		if (acfg::debug > 4)
		{
			cerr << "escaped version:" << endl;
			for (UINT i = 0; i < _countof(n2sTbl); i++)
				if (n2sTbl[i].ptr)
				{
					cout << n2sTbl[i].name << " = ";
					for (const char *p = n2sTbl[i].ptr->c_str(); *p; p++)
						if('\\' == *p)
							cout << "\\\\";
						else
							cout << *p;
					cout <<endl;
				}
		}

		for (UINT i=0; i<_countof(n2iTbl); i++)
			if(n2iTbl[i].ptr)
				cout << n2iTbl[i].name << " = \"" << *(n2iTbl[i].ptr) << "\"\n";
	}

#ifndef DEBUG
   if(acfg::debug>=5)
	   cout << "\n\nAdditional debugging information not compiled in.\n\n";
#endif
   
   
   /*
   // help STL saving some memory
   for(tUrl2RepIter it=mapUrl2pVname.begin(); it!=mapUrl2pVname.end(); it++)
   {
	   tUrl2RepIter suc=it;
	   suc++;
	   if(suc==mapUrl2pVname.end()) break;
	   if(suc->first.sPath==it->first.sPath)
		   const_cast<string&>(suc->first.sPath)=it->first.sPath;
   }
   */
}
}

namespace rechecks
{
regex_t reIfiles, rePfiles, reWfiles;
deque<regex_t> vecReqPatters, vecTgtPatterns;

bool CompileExpressions()
{
	//printf("%s\\n%s\\n", a, b);

    if(regcomp(&rePfiles, acfg::pfilepat.c_str(), REG_EXTENDED))
    	return false;
    if(regcomp(&reIfiles, acfg::vfilepat.c_str(), REG_EXTENDED))
    	return false;
    if(regcomp(&reWfiles, acfg::wfilepat.c_str(), REG_EXTENDED))
    	return false;

    return true;
}

eFileKind GetFiletype(const string & in) {
	LOGSTART("rechecks::getFiletype");
    dbgline;
    if(!regexec(&reIfiles, in.c_str(), 0, NULL, 0))
        return FILE_INDEX;
    dbgline;
    if(!regexec(&rePfiles, in.c_str(), 0, NULL, 0))
        return FILE_PKG;
    dbgline;
    return FILE_INVALID;
}

bool MatchWhitelist(const string & in)
{
	LOGSTART("MatchWhiteList");
	bool bPersistent=!regexec(&reWfiles, in.c_str(), 0, NULL, 0);
	LOG(in <<" is " << (bPersistent ? "persistent" : "mortal") <<  " file");
	return bPersistent;
}

inline bool CompileUncachedRex(const string & token, bool bIsTgtPattern, bool bRecursiveCall)
{
	deque<regex_t> & patvec = bIsTgtPattern ? vecTgtPatterns : vecReqPatters;

	if (0!=token.compare(0, 5, "file:")) // pure pattern
	{
		UINT pos = patvec.size();
		patvec.resize(pos+1);
		return 0==regcomp(&patvec[pos], token.c_str(), REG_EXTENDED);
	}
	else if(!bRecursiveCall) // don't go further than one level
	{
		string sPath=token.substr(5);
		if (sPath.empty())
			BARF("Bad file spec for repname, file:?");

		MakeAbsolutePath(sPath, acfg::confdir);

		tStrDeq srcs = ExpandFilePattern(sPath, true);
		for(tStrDeq::const_iterator it=srcs.begin(); it!=srcs.end(); it++)
		{
			filereader reader;
			if(!reader.OpenFile(*it))
			{
				cerr << "Error opening pattern file: " << *it <<endl;
				return false;
			}
			string line;
			while(reader.GetOneLine(line))
			{
				if(!CompileUncachedRex(*it, bIsTgtPattern, true))
					return false;
			}
		}
		return true;
	}

	cerr << token << " is not supported here" <<endl;
	return false;
}


bool CompileUncExpressions(const string & req, const string & tgt)
{
	tStrVec pats;
	Tokenize(req, SPACECHARS, pats, false);
	/*
	 *
	 if(acfg::debug>10)
			cerr << "req: " << req << ", tgt: " << tgt << ",#req: " << pats.size()<< endl;
			*/
	for (tStrVec::const_iterator it = pats.begin(); it != pats.end(); it++)
		if (!CompileUncachedRex(*it, false, false))
			return false;
	Tokenize(tgt, SPACECHARS, pats, false);
	for (tStrVec::const_iterator it = pats.begin(); it != pats.end(); it++)
		if (!CompileUncachedRex(*it, true, false))
			return false;
	return true;
}



bool MatchUncacheableRequest(const string & in)
{
	LOGSTART2("MatchUncacheableRequest", in << " against " << vecReqPatters.size() << " patterns");
	for(deque<regex_t>::const_iterator it=vecReqPatters.begin();
			it!=vecReqPatters.end(); it++)
	{
		if(!regexec(& (*it), in.c_str(), 0, NULL, 0))
			return true;
	}
	return false;
}

bool MatchUncacheableTarget(const string &in)
{
	for(deque<regex_t>::const_iterator it=vecTgtPatterns.begin();
			it!=vecTgtPatterns.end(); it++)
	{
		if(!regexec(& (*it), in.c_str(), 0, NULL, 0))
			return true;
	}
	return false;
}

}

void mkbasedir(const string & path)
{
    for(UINT pos=0; pos<path.size(); pos=path.find("/", pos+1))
    {
        if(pos>0)
            mkdir(path.substr(0,pos).c_str(), acfg::dirperms);
    }
}
/*
int main(int argc, char **argv)
{
	if(argc<2)
		return -1;
	
	acfg::tHostInfo hi;
	cout << "Parsing " << argv[1] << ", result: " << hi.SetUrl(argv[1])<<endl;
	cout << "Host: " << hi.sHost <<", Port: " << hi.sPort << ", Path: " << hi.sPath<<endl;
	return 0;
}
*/

