#! /usr/bin/perl
# This is a library file for Apt-cacher to allow code
# common to Apt-cacher itself plus its supporting scripts
# (apt-cacher-report.pl and apt-cacher-cleanup.pl) to be
# maintained in one location.

# This function reads the given config file into the
# given hash ref. The key and value are separated by
# a '=' and will have all the leading and trailing
# spaces removed.
sub read_config
{
    # set the default config variables
    my %config = (
		  cache_dir => '/var/log/cache/apt-cacher',
		  logdir => '/var/log/apt-cacher',
		  admin_email => 'root@localhost',
		  generate_reports => 0,
		  expire_hours => 0,
		  http_proxy => '',
		  use_proxy => 0,
		  http_proxy_auth => '',
		  use_proxy_auth => 0,
		  debug => 0,
		  clean_cache => 0,
		  allowed_hosts_6 => '*',
		  allowed_hosts => '*',
		  limit => 0,
		  daemon_port => 3142
		 );

    ($config_file) = @_;

    open CONFIG, $config_file or die $!;

    read(CONFIG, $buf, 50000);
    $buf=~s/\\\n#/\n#/mg; # fix broken multilines
    $buf=~s/\\\n//mg; # merge multilines

    for(split(/\n/, $buf))
	{
	next if(/^#/); # weed out whole comment lines immediately

	s/#.*//;   # kill off comments
	s/^\s+//;	# kill off leading spaces
	s/\s+$//;	# kill off trailing spaces

		if ($_)
		{
			my ($key, $value) = split(/\s*=\s*/);	# split into key and value pair
			$value = 0 unless ($value);
	    #print "key: $key, value: $value\n";
			$config{$key} = $value;
	    #print "$config{$key}\n";
		}
	}

	close CONFIG;

	return \%config;
}

# check directories exist and are writable
# Needs to run as root as parent directories may not be writable
sub check_install()
{
    # Die if we have not been configured correctly
    die "$0: No cache_dir directory!\n" if (!-d $$cfg{cache_dir});

    my $uid = $$cfg{user}=~/^\d+$/ ? $& : getpwnam($$cfg{group});
    my $gid = $$cfg{group}=~/^\d+$/ ? $& : getgrnam($$cfg{group});

    if (!defined ($uid || $gid)) {
	die "Unable to get user:group";
    }

    foreach my $dir ($$cfg{cache_dir}, $$cfg{logdir}, "$$cfg{cache_dir}/private",
		     "$$cfg{cache_dir}/import", "$$cfg{cache_dir}/packages",
		     "$$cfg{cache_dir}/headers", "$$cfg{cache_dir}/temp") {
	if (!-d $dir) {
	    print "Warning: $dir missing. Doing mkdir($dir, 0755)\n";
	    mkdir($dir, 0755) || die "Unable to create $dir";
	    chown ($uid, $gid, $dir) || die "Unable to set ownership for $dir";
	}
    }
    for $file ("$$cfg{logdir}/access.log", "$$cfg{logdir}/error.log") {
	if(!-e $file) {
	    print "Warning: $file missing. Creating.\n";
	    open(my $tmp, ">$file") || die "Unable to create $file";
	    close($tmp);
	    chown ($uid, $gid, $file) || die "Unable to set ownership for $file";
	}
    }
}

# Convert a human-readable IPv4 address to raw form (4-byte string)
# Returns undef if the address is invalid
sub ipv4_normalise ($)
{
	return undef if $_[0] =~ /:/;
	my @in = split (/\./, $_[0]);
	return '' if $#in != 3;
	my $out = '';
	foreach my $num (@in)
	{
		return undef if $num !~ /^[[:digit:]]{1,3}$/o;
		$out .= pack ("C", $num);
	}
	return $out;
}

# Convert a human-readable IPv6 address to raw form (16-byte string)
# Returns undef if the address is invalid
sub ipv6_normalise ($)
{
	return "\0" x 16 if $_[0] eq '::';
	return undef if $_[0] =~ /^:[^:]/  || $_[0] =~ /[^:]:$/ || $_[0] =~ /::.*::/;
	my @in = split (/:/, $_[0]);
	return undef if $#in > 7;
	shift @in if $#in >= 1 && $in[0] eq '' && $in[1] eq ''; # handle ::1 etc.
	my $num;
	my $out = '';
	my $tail = '';
	while (defined ($num = shift @in) && $num ne '')
	{
		return undef if $num !~ /^[[:xdigit:]]{1,4}$/o;
		$out .= pack ("n", hex $num);
	}
	foreach $num (@in)
	{
		return undef if $num !~ /^[[:xdigit:]]{1,4}$/o;
		$tail .= pack ("n", hex $num);
	}
	my $l = length ($out.$tail);
	return $out.("\0" x (16 - $l)).$tail if $l < 16;
	return $out.$tail if $l == 16;
	return undef;
}

# Make a netmask from a CIDR network-part length and the IP address length
sub make_mask ($$)
{
	my ($mask, $bits) = @_;
	return undef if $mask < 0 || $mask > $bits;
	my $m = ("\xFF" x ($mask / 8));
	$m .= chr ((-1 << (8 - $mask % 8)) & 255) if $mask % 8;
	return $m . ("\0" x ($bits / 8 - length ($m)));
}

sub extract_sums {
   my ($name, $hashref) = @_;
   my ($cat, $listpipe, $indexbase);

   $cat = ($name=~/bz2$/ ? "bzcat" : ($name=~/gz$/ ? "zcat" : "cat"));

   if($name=~/^(.*_)Index$/) {
       $indexbase=$1;
       $indexbase=~s!.*/!!g;
   }

   open($listpipe, "-|", $cat, $name);
   my $file;
   while(<$listpipe>) {
      if(/^\s(\w{40})\s+\d+\s(\S+)\n/) {
	 $sum=$1;
	 # Index format similar to Sources but filenames need to be corrected
	 # FIXME: this is sha1, not md5. Different format, can be detected by
	 # the length (40), though
	 $file=$indexbase.$2.".gz";
     }
     elsif(/^\s(\w{32})\s\d+\s(\S+)\n/) {
	 $sum=$1;
	 $file=$2;
      }
      elsif(/^MD5sum:\s+(.*)$/) {
	 $sum=$1;
      }
      elsif(/^Filename:\s+(.*)$/) {
	 $file=$1;
	 $file=~s/.*\///;
      }
      if(defined($file) && defined($sum)) {
	 $$hashref{$file}=$sum;
	 undef $file;
	 undef $sum;
      }
   };
   close($listpipe);
   my $ret = ! ($? >> 8);
   return $ret;
}


my $exlock;
my $exlockfile;

sub define_global_lockfile {
    $exlockfile=shift;
}

sub set_global_lock {

    die ("Global lock file unknown") if !defined($exlockfile);

    my $msg=shift;
    $msg="" if !defined($msg);

    debug_message("Entering critical section $msg") if defined (&debug_message);

    #may need to create it if the file got lost
    my $createstr = (-f $exlockfile) ? "" : '>';

    open($exlock, $createstr.$exlockfile);
    if ( !$exlock || !flock($exlock, LOCK_EX)) {
	debug_message("unable to achieve a lock on $exlockfile: $!") if defined (&debug_message);
	die "Unable to achieve lock on $exlockfile: $!";
    }
}

sub release_global_lock {
		    debug_message("Exiting critical section") if defined (&debug_message);
   flock($exlock, LOCK_UN);
}

sub setup_ownership {
    my $uid=$$cfg{user};
    my $gid=$$cfg{group};

    if($chroot) {
	if($uid || $gid) {
	    # open them now, before it is too late
	    # FIXME: reopening won't work, but the lose of file handles needs to be
	    # made reproducible first
	    &open_log_files;
	}
	chroot $chroot || die "Unable to chroot, aborting.\n";
	chdir $chroot;
    }

    if($gid) {
	if($gid=~/^\d+$/) {
	    my $name=getgrgid($gid);
	    die "Unknown group ID: $gid (exiting)\n" if !$name;
	}
	else {
	    $gid=getgrnam($gid);
	    die "No such group (exiting)\n" if !defined($gid);
	}
	$) = $gid;
	$( = $gid;
	$) =~ /^$gid\b/ && $( =~ /^$gid\b/ || barf("Unable to change group id");
    }

    if($uid) {
	if($uid=~/^\d+$/) {
	    my $name=getpwuid($uid);
	    die "Unknown user ID: $uid (exiting)\n" if !$name;
	}
	else {
	    $uid=getpwnam($uid);
	    die "No such user (exiting)\n" if !defined($uid);
	}
	$> = $uid;
	$< = $uid;
	$> == $uid && $< == $uid || barf("Unable to change user id");
    }
}

sub barf {
	my $errs = shift;

	die "--- $0: Fatal: $errs\n";
}

######### HOOKS ###########
#
# arg: file to be scanned and added to DB
sub import_sums {
   return 1;
}

# purpose: ?create?, lock the DB file and establish DB connection
sub db_init {
   return 1;
}

# purpose: create hasher object
sub data_init {
   return 1;
}

# purpose: append data to be scanned
sub data_feed {
   return 1;
}

# args: filename only or filename and sum
sub check_sum {
   return 1;
}

# args: filename and sum
sub store_sum {
}

#### common code for installation scripts ####
sub remove_apache {
    foreach $apache ("apache", "apache-ssl", "apache2") {
	# Remove the include lines from httpd.conf
	my $httpdconf = "/etc/$apache/httpd.conf";
	if (-f $httpdconf) {
	    $old = $httpdconf;
	    $new = "$httpdconf.tmp.$$";
	    $bak = "$httpdconf.bak";
	    my $done;

	    open(OLD, "< $old")         or die "can't open $old: $!";
	    open(NEW, "> $new")         or die "can't open $new: $!";

	    while (<OLD>) {
		$done += s/# This line has been appended by the Apt\-cacher install script/ /;
		$done += s/Include \/etc\/apt\-cacher\/apache.conf/ /;
		(print NEW $_)          or die "can't write to $new: $!";
	    }

	    close(OLD)                  or die "can't close $old: $!";
	    close(NEW)                  or die "can't close $new: $!";
	
	    if (!$done) {
		unlink $new;
		last;
	    }
	
	    rename($old, $bak)          or die "can't rename $old to $bak: $!";
	    rename($new, $old)          or die "can't rename $new to $old: $!";
#	    if (-f "/etc/init.d/$apache")
#	  {
#	      `/etc/init.d/$apache restart`;
#	  }
	}
    }
}

1;
