#
# bts.rb - ruby interface for debian bts
# Copyright (c) 2002 Masato Taruishi <taru@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Currently, this interface has only acquires to create bugs.

require 'debian/bug'
require 'net/http'
require 'uri.rb'
require 'find'
require 'ftools'
require 'zlib'
require 'thread'

module Debian
  module BTS
    class Acquire
      module Cache

	def create_cache_dir(cache_dir)
	  ::File.mkpath(cache_dir)
	end

	def expire(cache_dir, timer)
	  now = Time.now
	  create_cache_dir(cache_dir)
	  Find.find(cache_dir) do |path|
	    if FileTest.file?(path)
	      mtime = ::File.mtime(path)
	      if now - mtime > timer
		::File.delete(path)
	      end
	    end
	  end
	end

	def encode(path)
	  URI.escape(path, /\/|&/)
	end

	def read(cache_dir, path)
	  cache_path = cache_dir + "/" + encode(path)
	  if FileTest.exist?(cache_path)
	    open(cache_path) do |file|
	      file.read
	    end
	  else
	    nil
	  end
	end

	def write(cache_dir, path, body)
	  cache_path = cache_dir + "/" + encode(path)
	  open(cache_path, "w") do |file|
            file.write body
	  end
	end

	module_function :expire, :read, :write, :encode,
	  :create_cache_dir

      end

      def initialize(cache_dir = nil, timer = nil)
	@cache_dir = cache_dir
	begin
          Cache.expire(@cache_dir, timer) if @cache_dir != nil
        rescue Errno::EACCES
          $stderr.puts "W: #{$!}"
          @cache_dir = nil
        end
	@use_cache = (@cache_dir != nil)
	@mutex = Mutex.new
        @mutex_hash = {}
	@buf = {}
	@user_agent = nil
      end

      def read (path, use_cache = @use_cache)
	$stderr.puts "Reading #{path}... " if $DEBUG
	@mutex.synchronize {
          @mutex_hash[path] = Mutex.new if @mutex_hash[path] == nil 
	}
        @mutex_hash[path].synchronize {
          if @buf[path] == nil && use_cache
	    @buf[path] = Cache.read(@cache_dir, encode(path))
          end
	  if @buf[path] == nil
	    @buf[path] = read_real(path)
	    begin
              Cache.write(@cache_dir, encode(path), @buf[path]) if @cache_dir
            rescue Errno::EACCES
              $stderr.puts "W: #{$!}"
            end
	  end
        }
	@buf[path]
      end

      def write(path, buf)
	Cache.write(@cache_dir, encode(path), buf)
      end

      def cache_path(path)
	@cache_dir + "/" + encode(path)
      end

      def read_real(path)
	raise "Not Implemented"
      end

      def encode(path)
	raise "Not Implemented"
      end

      attr_accessor :use_cache

      class HTTP < Acquire
	def initialize(host = "bugs.debian.org", port = 80, cache_dir = nil, timer = nil)
	  super(cache_dir, timer)
	  @host = host
	  @port = port
	  @extra_headers = {}
	  if ENV["http_proxy"] != nil
	    uri = URI::split(ENV["http_proxy"])
	    @proxy_host = uri[2]
	    @proxy_port = uri[3]
	    begin
	      Net::HTTP::Proxy(@proxy_host, @proxy_port).start(@host,@port)
	    rescue
	      $stderr.puts "Disabling unavailable proxy configuration: #{ENV["http_proxy"]}: #{$!}"
              @proxy_host = @proxy_port = nil
	    end
	    puts "http_proxy: #{ENV['http_proxy']} : #{@proxy_host} #{@proxy_port}" if $DEBUG
	  end
          puts "http://#{host}:#{port}/" if $DEBUG
	end

	def read_real(path)
	  # creates instance every time to be thread safe
	  puts "getting #{path}" if $DEBUG
	  val = nil
	  Net::HTTP::Proxy(@proxy_host, @proxy_port).start(@host, @port) { |http|
	    val = http.get(path, @extra_headers).body
	  }
	  val
	end

	def encode(path)
	  URI.escape(path, /\/|&/)
	end

	def user_agent
	  @extra_headers['User-Agent']
	end

	def user_agent=(agent)
	  @extra_headers['User-Agent'] = agent
	end

      end

      class File < Acquire

        def initialize(dir=".")
          super( )
          @dir = dir
        end

        def read_real(path)
          $stderr.puts "File: read #{path}" if $DEBUG
          if @buf[path] == nil
            open(@dir + "/" + path) do |io|
              @buf[path] = io.read
            end
          end
          @buf[path]
        end

      end

      class LDAP < Acquire
        LDAPSEARCH = "ldapsearch"
        LDAPSEARCH_OPTION = "-x -LLL"

        def initialize( host = "bugs.debian.org", basedn = "dc=bugs,dc=debian,dc=org" )
          super()
          @host = host
          @basedn = basedn
        end

        def read_real( filter )
          $stderr.puts "#{LDAPSEARCH} #{LDAPSEARCH_OPTION} -h #{@host} -b #{@basedn} '#{filter}'" if $DEBUG
          open("|#{LDAPSEARCH} #{LDAPSEARCH_OPTION} -h #{@host} -b #{@basedn} '#{filter}'") do |i|
            @buf[filter] = i.read
          end
          @buf[filter]
        end

      end

    end


    class Parser

      def initialize(acquire)
	@acquire = acquire
      end

      attr_reader :acquire

      class Index < Parser

        class StringIO
          def initialize(buf)
            @buf = buf
	    @ptr = 0
	  end

          def read ( size = nil )
	    if @buf == nil
	      return nil
	    end 
            if size
              buf = @buf[@ptr,size]
              @ptr += size
              if @buf.size < @ptr
                @buf = nil
              end
	    else
	      buf = @buf
	      @buf = nil
	    end
            buf
	  end
	end

	# pkg   num  date? status submitter severity tags
	REGEX = /^(\S+) (\d+) (\d+) (\S+) \[(.+)\] (\S+)( (.+))?/o

	def initialize(acquire, indexdir = "/")
	  super(acquire)
	  @bugs = {}
	  @buf = nil
	  @indexdir = indexdir
	  puts "indexdir = #{@indexdir}" if $DEBUG
	end

	def fetch_bugs (severity)
	  if @bugs[severity]
	    @bugs[severity]
	  else
	    bugs = Bugs.new
	    path=@indexdir + "index.db-#{severity}.gz"
	    puts "fetching #{path}.. " if $DEBUG
	    io = StringIO.new( @acquire.read(path) )
	    Zlib::GzipReader.wrap(io).each do |line|
	      REGEX =~ line
	      pkgname = $1
	      id = $2
	      time = $3
	      sev = $6
	      stat = $4
	      tag = $7.split(' ') if $7 != nil
	      #             pkg, id, sev, st, desc, tag, merge 
	      bug = Bug.new(pkgname, id,  sev, stat, "", tag, [], Time::at(time.to_i))
	      bugs << bug
	    end
	    @bugs[severity] = bugs
	  end
	end

	def parse(pkg, severities = ["critical", "grave"])
	  bugs = Bugs.new
	  severities.each do |severity|
	    fetch_bugs(severity).each do |bug|
	      if pkg == bug.pkg_name
	        i = bug.bug_number.to_i % 100
  	        i = "0#{i}" if i < 10
	        path = @indexdir + "db-h/#{i}/#{bug.bug_number}.status"
	        puts "fetching #{path}.. " if $DEBUG
	        buf = @acquire.read(path).split("\n")
	        bug.desc = buf[2]
	        bug.mergeids = buf[8].split(' ') if buf[8]
	        p bug if $DEBUG
	        bugs << bug
	      end
	    end
	  end
	  bugs
	end
      end

      class CGI < Parser
	
	module CGIParser
	  None = 0
	  BugIntern = 1
	  
	  def each_severity(body)
	    stat = None
	    severity = ""
	    body.each_line do |line|
	      case stat
	      when BugIntern
		severity << line
		if /<\/UL>/ =~ line
		  yield severity
		  severity = ""
		  stat = None
		end
		
	      when None
		if /<H2>(.*)<\/H2>/ =~ line
		  severity << line
		  stat = BugIntern
		end
	      end
	    end
	  end
	  
	  def each_bug(io)
	    each_severity(io) do |severity|
	      serv = nil
	      stat = nil
	      bugs = ""
	      severity.each_line do |line|
		case line
		when /<HR><H2>.*<\/a>(\S+).+- (\S+).*<\/H2>/
		  serv = $1.downcase!
		  stat  = $2
		when /<li><a href="(.*)">(#\S*): (.*)<\/a>/
		  yield serv, stat, bugs if /<li>/ =~ bugs
		  link = $1
		  bug_number = $2
		  desc = $3
		  bugs = line
		else
		  bugs << line
		end
	      end
	      yield serv, stat, bugs if /<li>/ =~ bugs
	    end
	  end
	  
	  def parse(pkg, body)
	    bugs = Bugs.new
	    each_bug body do |serv, stat, bug|
	      tags = []
	      bug_number = -1
	      desc = ""
	      link = ""
	      mergeids = []
	      puts "PARSING: serv: #{serv}, #{stat}" if $DEBUG
	      bug.each_line do |line|
		puts line if $DEBUG
		case line
		when /Tags: (.+);/
		  tags = $1.gsub!("<strong>", "").gsub!("</strong>", " ").split
		when /<li><a href="(.*)">#(\S*): (.*)<\/a>/
		  link = $1
		  bug_number = $2
		  desc = $3
		when /^merged with (.+);/
		  merged_line = $1
		  merged_line.gsub!(/<[^>]+>/, "")
		  merged_line.gsub!("#","")
		  mergeids = merged_line.split(", ")
		end
	      end
	      bug = Bug.new("#{pkg}", bug_number, serv, stat, desc, tags, mergeids)
	      bugs << bug
	      puts "Created: #{bug.inspect}" if $DEBUG
	    end
	    bugs
	  end

	  module_function :parse, :each_bug, :each_severity

	end

	def parse(pkg, severities = [])
          if severities.empty?
            severities = ""
          else
            severities = "&sev-inc=" + severities.join('&sev-inc=')
          end
	  esc_pkg = URI.escape(pkg + severities)
	  path = "/cgi-bin/pkgreport.cgi?pkg=#{esc_pkg}"
	  buf = @acquire.read(path)
	  CGIParser.parse(pkg, buf)
	end

	def submit_version(bug)
	  path = "/cgi-bin/bugreport.cgi?bug=#{bug.bug_number}"
	  buf = @acquire.read(path)
	  buf.each_line do |line|
	    if /^Package:\s*([^<>]+)\s*$/ =~ line
	      if $1 != bug.pkg_name
		return nil
	      end
	    end
	    if /^Version:\s*(\S+)/ =~ line
	      return $1
	    end
	  end
	  return nil
	end

	def first_contact(bug)
	  path = "/cgi-bin/bugreport.cgi?bug=#{bug.bug_number}"
	  buf = @acquire.read(path)
	  contact = ""
	  stat = 0
	  buf.each_line do |line|
	    case stat
	    when 0
	      if /^$/ =~ line
		stat = 1
	      end

	    when 1
	      if /<\/pre>/ =~ line
		break
	      end
	      contact << line
	    end
	  end
	  return contact
	end

	def desc(bug)
	  path = "/cgi-bin/bugreport.cgi?bug=#{bug.bug_number}"
	  buf = @acquire.read(path)
	  buf.each_line do |line|
	    case line
	    when /<H1>.+<BR>(.*)<\/H1>/
	      return $1
	    end
	  end
	  return nil
	end

      end

      class DSA < Parser

	def parse(pkg, severities = [])
          bugs = Bugs.new
          buf = @acquire.read("/security/2004/index.en.html")
          puts buf if $DEBUG
          buf.each_line do |line|
            if /(DSA-\d+) ([^<]+).*- ([^<]+)/ =~ line
              if pkg == $2 then
                id = $1
                desc = $3
                bug = Bug.new(pkg, id, "debian-security-announce", "", desc)
                bugs << bug
                puts "Created: #{bug.inspect}" if $DEBUG
              end
              bugs
            end
          end
          bugs
	end

      end

      class ReleaseCritical < Parser
	
	module RCParser

	  BodyRe = /^<a name="([^"]+)"/
	  BugRe = /^<A NAME="(\d+)".*\[[^\]]+\] (.*)/
	  Stat = [ HEADER = 0, BUGBODY = 1 ]
	  
	  def each_pkg(body)
	    pkg_str = ""
	    stat = HEADER
	    body.each_line do |line|
	      case stat
	      when HEADER
		if BodyRe =~ line
		  stat = BUGBODY
		  pkg_str = line
		end
	      when BUGBODY
		if BodyRe =~ line
		  yield pkg_str
		  pkg_str = line
		else
		  pkg_str << line
		end
	      end
	    end
	  end
	  
	  def each_bug(body)
	    each_pkg(body) do |pkgstr|
	      pkg = ""
	      pkgstr.each_line do |line|
		case line
		when BodyRe
		  pkg = $1
		when BugRe
		  yield pkg, $1, $2
		end
	      end
	    end
	  end
	  
	  def parse(pkgname, body)
	    bugs = Bugs.new
	    each_bug(body) do |pkg, bug, desc|
	      bug = Bug.new(pkg, bug, "release-critical", "", desc)
	      bugs << bug
	      puts "Created: #{bug.inspect}" if $DEBUG
	    end
	    bugs
	  end
	  module_function :parse, :each_bug, :each_pkg
	end

	def parse(pkg, severities = [])
	  buf = @acquire.read("/release-critical/other/all.html")
	  RCParser.parse(pkg, buf)
	end

      end


      class LDIF < Parser

        def each_entry(buf)
          entry = ""
          buf.each_line do |line|
            case line
            when ""
              yield entry
              entry = ""
            else
              entry += line
            end
          end
        end

        def parse(pkg, severities = [])
          bugs = Bugs.new
          sevfilt = ""
          severities.each do |s|
            sevfilt += "(debbugsSeverity=#{s})"
          end
          filter = "(&(debbugsPackage=#{pkg})(|#{sevfilt}))"
          buf = @acquire.read(filter)
          each_entry(buf) do |entry|
            id = nil
            severity = nil
            stat = nil
            desc = nil
            tags = nil
            mergeids = nil
            time = nil
            entry.each_line do |line|
              attr_type = line.split(": ")[0]
              attr_val = line.split(": ")[1, -1]
              case attr_type
              when "debbugsID"
                id = attr_val
              when "debbugsSeverity"
                severity = attr_val
              when "debbugsStat"
                stat = attr_val
              when "debbugsTag"
                tags = [] if tags == nil
                tags.append(attr_val)
              when "debbugsMergedWith"
                mergeids = [] if mergeids == nil
                mergeids.append(attr_val)
              when "debbugsTitle"
                desc = attr_val
              when "debbugsDate"
                time = Time::at(attr_val)
              end
              bug = Bug.new( pkg, id, severity, stat, desc, tags,
                             mergeids, time )
              Bugs << bug
            end
          end
          bugs
        end
      end
    end      
  end
end
