# -*- coding: utf-8 -*-
#####################################################################
#  Rafael Proença <cypherbios@ubuntu.com>
#  Laudeci Oliveira <laudeci@gmail.com> 
#
#  Copyright 2006 APTonCD DevTeam.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#####################################################################
import gtk
from math import ceil
import datetime
from time import time

from APTonCD.core import *
from APTonCD.core import gui 
from APTonCD.core import constants
from APTonCD.core.metapackage import MetaPackage
from APTonCD.core import mediainfo
from APTonCD.core.mkisofs import Mkisofs
from APTonCD.core.scanpkgs import AptFtpArchive
from APTonCD.core.utils import get_icon
from APTonCD.core import utils
from APTonCD.widgets import *
from APTonCD.widgets.progresswindow import ProgressDialog
from APTonCD.widgets import stepswidget
from APTonCD.create.burnproperties import BurnProperties
from APTonCD.create.finisheddialog import FinishedDialog

WINDOW_NAME = 'create'
    
class CreateWindow(controller.IController):
    """
        Create window
    """
 
    def __init__(self, controller = None, temp_path = '', packages_path = '', packages_file_list='', non_interactive = False):
        """
            Constructor
        """
        self.controller = controller
        self.glade = gui.get_glade(constants.CREATE_GUI, WINDOW_NAME)
        
        self.non_interactive = non_interactive
        if temp_path == '':
            self.temp_path = constants.TEMP_FOLDER
        else:
            self.temp_path = temp_path
        self.packages_path = packages_path
        
        self.window = gui.get_widget(self.glade, WINDOW_NAME)
        #self.window.set_icon(get_icon('aptoncd'))
        #if is inside a container
        if self.controller:
            self.window.set_decorated(False)
            self.content = gui.set_parent_widget_from_glade(self.glade, WINDOW_NAME, self.controller.createContainer)
            self.window.destroy()
            gui.setCursorToBusy(self.get_main_window(), True)
        else:
            #Window Widgets
            if packages_file_list =='' or self.non_interactive == False:
                self.window.show()
            gui.setCursorToBusy(self.window, True)
        
        self.ListContainer = gui.get_widget(self.glade,'ListContainer')       
        self.lblPkgDesc = gui.get_widget(self.glade, 'lblPkgDesc')
        self.pkgImg = gui.get_widget(self.glade, 'pkgImg')

        self.btnAdd = gui.get_widget(self.glade, 'btnAdd')
        self.btnBurn = gui.get_widget(self.glade, 'createBurn')

        self.packageList = PackageList.PackageView(self)
        self.ListContainer.add(self.packageList)
        self.ListContainer.show_all()
        self.bind_signals()
        #gui.connect(self.tgBtnCreate, 'toggled',self.on_BtnCreate_toggle)
        self.values = {
			'media_type': 'CD',
			'media_size': constants.CD,
			'destination': utils.get_home_folder(),
			'isoname': '',
			'createmeta': False
			}
        if packages_file_list =='':
            #self.packageList.can_select_old = self.ckbtnOldVersions.get_active()
            #self.packageList.auto_select_packages = self.ckbtnAutoSelectDepends.get_active()
        
            gui.processEvents()
        
            if self.controller:
                self.controller.show_status_message(constants.MESSAGE_0001)
        else:
            self.process_list(packages_file_list)
        
        

    def process_list(self, packages_file_list):
        pack_files = []
        file_list = open(packages_file_list,'r')
        for n in file_list.readlines():
            pack_files.append(n.replace('\n',''))
        file_list.close()
        self.packageList.add_from_file_list(pack_files)
        if self.non_interactive:
            self.btnBurn.emit('clicked')
        #self.window.destroy()
        
    def __show_message(self, text = ''):
        if self.controller:
            self.controller.show_status_message(text)
            
    def rebuild_list(self):
        self.controller.progressbar.show()
        self.__show_message(constants.MESSAGE_0001)
        self.packageList.clear()
        self.packageList.load_package_folder(self.packages_path, False, False)
        #self.__show_message(constants.MESSAGE_0066)
        
    def get_main_window(self):
        """
            implements IController interface method.
            for access the class windows object.
        """
        if self.controller:
            return self.controller.get_main_window()
        else:
            return self.window
    
    def get_parent_widget(self):
        """
            implements IController interface method.
            for container widget
        """
        if self.controller:
            return self.controller.get_parent_widget()
        else:
            return self.ListContainer
    
    def get_parent(self):
        return self.controller
    
    def bind_signals(self):
        #signals connection
        gui.connect(self.window, 'destroy', gtk.main_quit)
        #gui.connect(self.ckbtnOldVersions, 'toggled', self.on_check)
        #gui.connect(self.ckbtnAutoSelectDepends, 'toggled', self.on_check)
        gui.connect(self.btnAdd, 'clicked', self.on_btnAdd_clicked)
        gui.connect(self.btnBurn,'clicked', self.on_btnBurn_clicked)
    
    def on_add_package(self,widget):
        
        file_add = filedialog.ChooseFileDialog(constants.MESSAGE_0013, self.window, constants.MESSAGE_0015, '*.deb')
        files = file_add.run()
        file_add.destroy()
        if len(files) > 0:
            self.packageList.add_from_file_list(files)
    
    def on_add_folder(self,widget):
        file_add = filedialog.ChooseFolderDialog(constants.MESSAGE_0014, self.window)
        files = file_add.run()
        file_add.destroy()
        if len(files) > 0:
            self.packageList.load_package_folder(files[0],custom = True)    
    
    def on_btnBurn_clicked(self, widget):
        burn_window = BurnProperties(self)
        response = burn_window.run()
        if response == gtk.RESPONSE_APPLY:
            self.values['media_type'] = burn_window.get_media_type()
            self.values['media_size'] = burn_window.get_media_size()
            self.values['destination'] = burn_window.get_destination()
            self.values['isoname'] = burn_window.get_iso_name()
            self.values['createmeta'] = burn_window.get_metapackage()
            burn_window.destroy()
            self.create_aptoncd()
        else:
            burn_window.destroy()
        
    def on_btnAdd_clicked(self, widget):
        #self.btnAdd.set_state(gtk.STATE_SELECTED)
        widget.freeze_child_notify()
        c = contextMenu.ContextMenu(self)
        c.addMenuItem(constants.MESSAGE_0056, self.on_add_package, get_icon('deb',16))
        c.addMenuItem(constants.MESSAGE_0057, self.on_add_folder, get_icon('gnome-fs-directory',16))
        gui.popup_emoticons_under_button(c, widget, gui.get_top_window(widget))
        widget.thaw_child_notify()
        #c.popup(None, None, None, 1 , 0)
        #self.btnAdd.set_state(gtk.STATE_NORMAL)
    
    def show_selected(self, text, img):
        self.lblPkgDesc.set_markup(text)
        self.pkgImg.set_from_pixbuf(img)
        
    def show_selected_count(self, value):
        if self.controller:
            self.controller.show_status_message(constants.MESSAGE_0007 % value)
        
    def show(self):
        self.window.show()
    
    def hide(self):
        self.window.hide()
            
    def copy_files(self, isoFileList, tmpdir):
        """
            this will copy each package from every cd to the destination.
        """
        self.controlFile = {}
        #start mounting the cd/dvd layout
        for cds in isoFileList.keys():
            #show the first step in bold with an arrow at its left
            self.steps.set_current(0)
            media_folder_name = self.values['media_type'] + cds.__str__()
            current_msg = constants.MESSAGE_0020 + ' (' + media_folder_name + ')'
            self.steps.set_current_text(0,current_msg )
            self.progress.progress_text = constants.MESSAGE_0002
            
            tmpmetapackageDir = utils.join_path(tmpdir , 'metapackage/' + media_folder_name)
            tmppackages = utils.join_path(tmpdir , 'packages')

            #if we are creating a metapackage
            #we must create a DEBIAN folder
            
            if self.values['createmeta']:
                utils.mkdir(utils.join_path(tmpmetapackageDir, 'DEBIAN'),True)
            else:
                utils.removePath(tmpmetapackageDir )

            #metapackage configuration
            if self.values['createmeta']:
                controlFile =  utils.join_path(tmpmetapackageDir, 'DEBIAN/control')
                utils.del_file(controlFile)
                self.mPack = MetaPackage(controlFile)
            
            #cd destination folder
            destination = utils.join_path(tmppackages, media_folder_name)
            utils.mkdir(destination, True)
            
            #destination for each package inside this media
            pkg_destination = utils.join_path(destination , 'packages')
            utils.mkdir(pkg_destination, True)
            
            #start to copy files from source to destination
            self.progress.stop = len(isoFileList[cds])
            
            for index, pkg in enumerate(isoFileList[cds]):
                #update the progress bar
                self.progress.update_progress(index + 1)
                
                #if we must create metapackage we will add this
                #package to the meta list
                if self.values['createmeta']:
                    self.mPack.appendPackage(pkg.package)
                
                #set the folder destination of the package
                filedestination = utils.join_path(pkg_destination, pkg.deb_filename)
                
                #if user has canceled
                #stop iterating
                if self.progress.cancel_status:
                    errorMsg = constants.MESSAGE_0027
                    return False, errorMsg
                
                #copy file to destination
                pkg.copy_to(filedestination )
                
                #update the progress message
                self.progress.progress_text = constants.MESSAGE_0029 % pkg.package
                gui.processEvents()
            
            #close the packfile for this cd
            if self.values['createmeta']:
                self.mPack.write()
                
        #update processes
        #marking first as done and starting a new one
        self.steps.set_current_text(0,constants.MESSAGE_0020)
        self.steps.set_done(0)
            
        return True, None
    
    def scan_packages(self, isoFileList, tmpdir):
        
        self.time_start = 1
        self.steps.set_current(1)
        self.progress.update_progress(0)
        self.progress.progress_widget.set_text( '')
        self.progress.progress_text = constants.MESSAGE_0019
        gui.processEvents()
        
        for cds in isoFileList.keys():
            #cd destination folder
            media_folder_name = self.values['media_type'] + cds.__str__()
            
            current_msg = constants.MESSAGE_0021 + ' (' + media_folder_name + ')'
            self.steps.set_current_text(1,current_msg )
            
            tmpmetapackageDir = utils.join_path(tmpdir , 'metapackage/' + media_folder_name)
            tmppackages = utils.join_path(tmpdir , 'packages')
        
            destination = utils.join_path(tmppackages, media_folder_name)
            pkg_destination = utils.join_path(destination , 'packages')
            
            controlFile =  utils.join_path(tmpmetapackageDir, 'DEBIAN/control')
            #create the deb file for metapackage
            command = 'dpkg-deb -b %s  %s > /dev/null 2> /dev/null' % \
                      (tmpmetapackageDir.replace(' ','\ '), utils.join_path(pkg_destination.replace(' ','\ '), self.mPack.get_mtFileName()) )
            result = utils.run_command(command)
            #remove the file needed to create the metapackage
            utils.del_file(controlFile)
            
            packages_files = utils.join_path(destination.replace(' ','\ ') , 'Packages')
            command = 'apt-ftparchive packages packages/'
        
            apt_ftp = AptFtpArchive()
            self.progress.stop = 100
            apt_ftp.set_progress_hook(self.on_update)
            apt_ftp.execute(command, destination, packages_files)
            
            #creates a tar file using bz2 compression
            result, path = utils.compress(packages_files)
            result, msg = utils.zip_file(utils.join_path(destination,'Packages.gz'), utils.join_path(destination,'Packages'))
            
            packages_files = utils.join_path(destination.replace(' ','\ ') , 'Release')
            command = 'apt-ftparchive release  .'
            apt_ftp.set_progress_hook(self.on_update)
            apt_ftp.execute(command, destination, packages_files)
            
            #Create aptoncd.inf file
            infoFile = utils.join_path(destination, 'aptoncd.info')
            info = mediainfo.MediaInfo(infoFile)
            info.write()
            
            #Creates .disk/info
            diskinfodir = utils.join_path(destination ,'.disk')
            utils.mkdir(diskinfodir, True)
            infoDiskFile = utils.join_path(diskinfodir,'info')
            infoDisk = mediainfo.aptDiskInfo(infoDiskFile, media_folder_name)
            infoDisk.write()
            
            #write README.diskdefines
            infoDiskDefinesFile = utils.join_path(destination, 'README.diskdefines')
            infoDiskDefines = mediainfo.aptDiskDefines(infoDiskDefinesFile,media_folder_name)
            infoDiskDefines.write()
            
            gui.processEvents()
            
            utils.removePath(tmpmetapackageDir)
            
        self.progress.cancel_pulse()
        self.steps.set_current_text(1,constants.MESSAGE_0021)
        self.steps.set_done(1)
        self.progress.update_progress(0)
        return True, None
    
    def create_iso(self, isoFileList, tmpdir):
        
        self.steps.set_current(2)
        
        today = datetime.date.today().strftime('%Y%m%d')
        #this will hold all isos that we will create
        #from user selection
        self.isos = []
        
        for cds in isoFileList.keys():
            media_folder_name = self.values['media_type'] + cds.__str__()
            

            current_msg = constants.MESSAGE_0022 + ' (' + media_folder_name + ')'
            self.steps.set_current_text(2, current_msg )
            tmppackages = utils.join_path(tmpdir , 'packages')
        
            destination = utils.join_path(tmppackages, media_folder_name)

            if self.values['isoname'] == '':
                fileNames = utils.list_dir(self.values['destination'], 'aptoncd-' + today +"-" + self.values['media_type'] + cds.__str__())
                if len(fileNames) > 0:
                    fileDestiny = 'aptoncd-' + today +"-" + self.values['media_type'] + cds.__str__() +'-'+str(len(fileNames)) +'.iso'
                else:
                    fileDestiny = 'aptoncd-' + today +"-" + self.values['media_type'] + cds.__str__() +'.iso'
            else:
                fileNames = utils.list_dir(self.values['destination'], self.values['isoname'] +"-" + self.values['media_type'] + cds.__str__())
                if len(fileNames) >0:
                    fileDestiny = self.values['isoname'].replace('.iso','') + "-" + self.values['media_type'] + cds.__str__() +'-' +str( len(fileNames)) +'.iso'
                else:
                    fileDestiny = self.values['isoname'].replace('.iso','') + "-" + self.values['media_type'] + cds.__str__() +'.iso'
            
            finalDestiny = utils.join_path(self.values['destination'], fileDestiny)
            self.isos.append(finalDestiny)
            
            #update processes
            #marking first as done and starting a new one
            self.progress.progress_text = constants.MESSAGE_0032 + ' '+ self.values['media_type'] + cds.__str__() + ' ('+fileDestiny+')'
            gui.processEvents()
            
            
            #now create the iso file
            command = ['-iso-level', '4' ,'-pad', '-l','-r', '-J', '-joliet-long', '-v', '-V', 'APTonCD', '-hide-rr-moved', '-o', finalDestiny.replace(' ','\ '),  destination.replace(' ','\ ')]
            print command
            iso_file = Mkisofs()
            self.progress.stop = 100
            iso_file.set_progress_hook(self.progress.update_percent)
            iso_file.create_iso(command)
            
        self.steps.set_current_text(2,constants.MESSAGE_0022)
        self.steps.set_done(2)
        self.progress.update_progress(0)
    
    def create_aptoncd(self):
        
        #this will hold error information.
        errorMsg = ""
        
        self.progress = ProgressDialog(self.get_main_window())
        self.progress.can_cancel_progress = True
        self.progress.title = constants.MESSAGE_0024 
        self.progress.description = constants.MESSAGE_0025 

        #defines the steps to create this iso.
        process = [constants.MESSAGE_0020, constants.MESSAGE_0021, \
                   constants.MESSAGE_0022, constants.MESSAGE_0023]
        self.steps = stepswidget.ProgressSteps(process)
        self.progress.add_widget(self.steps)
        
        #set temp variables for packages locations
        tmpdir = utils.join_path(self.temp_path, 'aptoncd') #"/tmp/aptoncd/"
        tmpmetapackageDir = utils.join_path(tmpdir , 'metapackage')
        tmppackages = utils.join_path(tmpdir , 'packages')
        
        #creates a temporary location to work with .deb files
        utils.mkdir(tmpdir, True)
        
        #get what files will be on each cd
        if self.values['media_type'] == 'CD':
            isoFileList = self.CreateIsoList(constants.CD)
        else:
            isoFileList = self.CreateIsoList(constants.DVD)
        
        #we will starting changing here to break process in parts
        # first, copy files from cds to locations
        
        result, msg = self.copy_files(isoFileList,tmpdir)
        self.scan_packages(isoFileList,tmpdir)

        self.create_iso(isoFileList,tmpdir)
        
        self.steps.set_current(3)
        gui.processEvents()
        #clean folders created previously    
        utils.removePath(tmpdir)
        self.steps.set_done(3)
        
        burn_list = {}
        index = 0
        for index, burns in enumerate(utils.get_burn_applications(constants.BURNS_DATA)):
        	item = utils.whereis(burns[0])
        	
        	if item:
        		burn_list[index]= { 'name':burns[0], 'location':item , 'parameter':burns[1]}
        		index += 1
        		        
        if index >= 1:
        	dlgburn = FinishedDialog(self, burn_list)
        	self.progress.destroy()
        	result, index = dlgburn.run()
        	
        	if result == 1:
        		for iso in self.isos:
        			command = burn_list[index]['location'] + ' '+ burn_list[index]['parameter'] + iso + ' &'
        			utils.run_command(command)
        	dlgburn.destroy()  
        
        self.get_main_window().set_sensitive(True)
        
        self.main_window = gui.get_glade(constants.MAIN_GUI, 'main')
        self.notebook = gui.get_widget(self.main_window, 'notebook')
        self.notebook.set_current_page(0)
    
    def get_iso_location(self):
        return self.values['destination']
    
    def on_update(self, value):
        if self.time_start == 1:
            self.time_start = 2
            self.progress.pulse()
        gui.processEvents()
        
    def CreateIsoList(self, size = constants.CD):
        """
            This will set what package will be inside
            each cd/dvd.
        """
        cds = {}
        currentCD = 1
        actualSize = 0
        
        for n in self.packageList.packages_list:
            pkg = n[PackageList.COL_PACKAGE]
            if pkg.selected:
                if not currentCD in cds: cds[currentCD] = []
                if ( actualSize + pkg.size) > size:
                        actualSize = 0
                        currentCD += 1
                        cds[currentCD] = []
                cds[currentCD].append(pkg)
                actualSize += pkg.size
                
        return cds
    
    def get_packages_size(self):
        """
            Return quantity of cds/dvds needed to
            hold selected packages.
        """
        cds = int(ceil(float(self.packageList.total_selected_size) / float(constants.CD)))
        if cds < 1: cds = 1
        dvds = int(ceil(float(self.packageList.total_selected_size) / float(constants.DVD)))
        if dvds < 1 : dvds = 1
        
        return cds, dvds


