# -*- coding: utf-8 -*-
#####################################################################
#  Rafael Proença <cypherbios@ubuntu.com>
#  Laudeci Oliveira <laudeci@gmail.com> 
#
#  Copyright 2006 APTonCD DevTeam.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#####################################################################
import gtk
import gtk.glade
import pango
import gobject
import os
import apt_pkg

from APTonCD.core.package import DebPackage
from APTonCD.core import constants
from APTonCD.core import utils
from APTonCD.core.gui import processEvents
from APTonCD.core.utils import get_icon
from APTonCD.core import gui 
from APTonCD.widgets import contextMenu
from APTonCD.widgets import dlgSkipped 
from APTonCD.widgets.propertywindow import PackagePropety
from APTonCD.widgets.progresswindow import ProgressDialog

def TreeViewSearch( model, column, key, it, data):
    """
        Do case insensitive wildcard search on package name.
    """
    key = key.lower()
    title = model.get_value(it, COL_PACKAGE).package
    title = title.lower()
    return title.find(key) == -1

class PackageView(gtk.TreeView):
    """This class represent the package list in the main window"""
    def __init__ (self, controller = None):
        """
        Constructor
        """
        super(PackageView, self).__init__()

        #gtk.TreeView.__init__ (self)

        #self.scroll = None
        self.controller = controller

        #packages dictionary for fast search of a package
        self.packages = {}

        #packages selection options
        self.select_old = False # allow old packages selection
        self.autoselect_dependents = False # do a selection on dependents packages

        #create the default liststore
        self.store = PackageModel(self)

        #self.sort_model = gtk.TreeModelSort(self.store)
        #self.sort_model.set_sort_column_id(COL_TITLE, gtk.SORT_ASCENDING)

        self.create_ui()
        self.connect_signals()

        self.set_enable_search(True)
        self.set_search_column(1)

        #columns will resize to fit it's content 
        self.columns_autosize()
        self.cellRendererText.set_property('ellipsize', pango.ELLIPSIZE_END)
        self.cache = None

    def create_ui(self):
        """
            This procedure will create all UI objects.
        """

        # prepares list for drag and drop operations
        self.enable_model_drag_dest([('application/x-deb', 0, 0), ('text/uri-list', 0, 1)], gtk.gdk.ACTION_PRIVATE)
        # Create all cell renders
        self.cellRendererToggle = gtk.CellRendererToggle()
        self.cellRendererText = gtk.CellRendererText()
        #self.cellRendererText.set_property('ellipsize', pango.ELLIPSIZE_END)
        self.cellRenderSize = gtk.CellRendererText()

        # add the check column to the tree view widget
        column = gtk.TreeViewColumn('', self.cellRendererToggle, active = COL_CHECK)
        column.set_sort_column_id(COL_CHECK)
        column.set_clickable(True)
        self.append_column(column)

        # add the pixbuf column to the tree view widget
        column = gtk.TreeViewColumn(constants.MESSAGE_0004, self.cellRendererText, markup = COL_TITLE)
        column.set_sort_column_id(COL_TITLE)
        column.set_clickable(True)
        column.set_resizable(True)
        column.set_expand(True)
        column.set_min_width(300)
        self.append_column(column) 

        self.colVersion = gtk.TreeViewColumn(constants.MESSAGE_0026, self.cellRendererText, markup = COL_VERSION)
        self.colVersion.set_sort_column_id(COL_VERSION)
        self.colVersion.set_clickable(True)
        self.colVersion.set_resizable(True)
        self.colVersion.set_visible(False)
        self.append_column(self.colVersion)

        self.cellRenderSize.set_property('xalign', 1.0)
        self.colSize = gtk.TreeViewColumn('Size', self.cellRenderSize, markup = COL_SIZE)
        self.colSize.set_property('alignment',1.0)
        self.colSize.set_sort_column_id(COL_SIZE)    
        self.colSize.set_clickable(True)
        self.append_column(self.colSize)

        # this will make the list searchable
        self.set_enable_search(True)
        self.set_search_column(COL_TITLE)
        self.set_search_equal_func(TreeViewSearch, None)

        self.set_rules_hint(True)
        self.controller.show_selected('',None)


    def show_column(self, col_id = 0):
        if col_id == 0:
            self.colSize.set_visible( not self.colSize.get_visible())
        elif col_id == 1:
            self.colVersion.set_visible( not self.colVersion.get_visible())


    def connect_signals(self):
        """
            This procedure will connect widgets to its signal handler.
        """
        self.connect('button_press_event', self.on_PackageList_right_click)
        self.connect('row-activated', self.on_row_activated)
        self.connect('cursor_changed', self.on_cursor_changed)
        self.cellRendererToggle.connect ("toggled", self.toggled_item)
        self.connect("drag_data_received",self.on_drag_data_received)

    def on_cursor_changed(self, widget):
        """
            Show package's info in the top panel when a package row is selected.
        """
        iter, pkg = self.get_selected_iter()
        if pkg:
            pkgText = self.store.get_value(iter, COL_TITLE) 
            img = get_icon(pkg.package, 48)

            if pkg.bad:
                #show a grayed pixlated image
                img = utils.grayscale(img, 0.3, True)
            elif not pkg.installed:
                #show an image with some brightness
                img = utils.grayscale(img, 0.3)

            self.controller.show_selected(pkgText,img)

    def on_drag_data_received(self, widget, context, x, y, selection, target_type, timestamp):
        """
            Drop package handler.
            It must handle a single file or a list of files.
        """
        file_list = []
        uri = selection.data.strip()
        uri_splitted = uri.split() # we may have more than one file dropped

        packages_error_list = '\n\n'
        msgError = constants.MESSAGE_0005

        #iterate for every file
        for uri in uri_splitted:
            if uri.endswith(".deb"):
                ipath = utils.parse_url(uri)
                file_list.append(ipath)

                #path,filename = utils.split_path(ipath)
        if len(file_list)>0:
            self.add_from_file_list(file_list)

    # Control Procedures
    def clear(self):
        self.controller.show_selected('',None)
        self.set_model(None)
        self.store.clear()
        self.store = PackageModel(self)

    def add_list(self, values):
        """
            Add itens to the internal liststore
        """
        # Removes the model so the addition becomes quickly
        self.set_model(None)

        for value in values:
            self.store.append(value)

        # set model back
        self.set_model(self.store)

    def get_selected_row(self):
        selection = self.get_selection()
        model, paths = selection.get_selected_rows()
        return paths[0]

    def get_selected(self):
        """
            Return the package object in the selected line.
        """
        selection = self.get_selection()
        model, iter, = selection.get_selected()
        return  self.store.get_value(iter, COL_PACKAGE) 

    def get_selected_iter(self):
        """
            Return the package object in the selected line.
        """
        selection = self.get_selection()
        model, iter, = selection.get_selected()
        if iter:
            return  iter , self.store.get_value(iter, COL_PACKAGE) 
        else:
            return None, None

    def __load_dialog_skip(self, list_skip=[], msg =''):
        n = dlgSkipped.SkippedDialog(self.controller, list_skip)
        n.set_message(msg)
        n.run()
        n.destroy()

    #add files from the button menu
    #Add package
    def add_from_file_list(self, packages_names):
        """
            Loads all packages files from a given path in to the list.
        """
        gui.setCursorToBusy(self.controller.get_parent_widget(),True)
        self.controller.get_main_window().set_sensitive(False)
        #self.clear()

        list_bad = []
        not_added = []

        f_len = len(packages_names)
        pbar_shown = False
        #only show progressbar if user selects more than 5 packages
        if len(packages_names) > 5:

            if self.controller.get_parent():
                progress = self.controller.get_parent().progressbar
                progress.show()
                pbar_shown = True
            else:
                pbar_shown = False
            processEvents()

        indx = 1
        # now get all packages in cache
        for index in range(f_len):
            percent = (float(index) / f_len)
            filename = os.path.basename(packages_names[index])
            pathname = os.path.dirname(packages_names[index])
            iter = self.store.add_item(filename,pathname, True)
            if iter == None:
                not_added.append(filename)
            else:
                #self.get_selection().select_iter(iter)
                if self.store.get_value(iter, COL_PACKAGE).bad:
                    list_bad.append(self.store.get_value(iter, COL_PACKAGE).deb_filename) 

            #process pending events / update the window
            if pbar_shown:
                progress.set_fraction(percent)
                progress.set_text(str(int(percent * 100)) + '%' )
                processEvents()

        if pbar_shown:
            progress.set_fraction(0)
            progress.hide()
            progress.set_text('0%')
            progress.set_fraction(0)

        #update the parent count labels
        #update the parent count labels
        self.update_parent_count()
        self.controller.get_main_window().set_sensitive(True)
        self.set_model(self.store)
        self.get_selection().select_path((0,0))
        self.grab_focus()
        gui.setCursorToNormal(self.controller.get_parent_widget())

        #show packages not inserted due to already exist in list
        if len(not_added)>0:
            self.__load_dialog_skip(not_added, constants.MESSAGE_0017)

        if len(list_bad)>0:
            self.__load_dialog_skip(list_bad, constants.MESSAGE_0016)

    def load_package_folder(self, path_to_scan, custom = False, recursive = False):
        """
            Loads all packages files from a given path in to the list.
        """
        gui.setCursorToBusy(self.controller.get_parent_widget(),True)
        self.controller.get_main_window().set_sensitive(False)
        #self.installed = [] #a list of installed packages
        self.store._load_cache()

        list_bad = []
        not_added = []

        progress = self.controller.get_parent().progressbar
        progress.show()

        processEvents()

        # now get all packages in cache
        packages_files = utils.get_deb_from_folder(path_to_scan, recursive)
        f_len = len(packages_files)
        for index, pkg in enumerate(utils.get_deb_from_folder(path_to_scan, recursive)):
            #add the package to the list
            index+=1
            percent = (float(index) / f_len)
            iter = self.store.add_item(pkg,path_to_scan, custom)
            if iter == None:
                not_added.append(os.path.basename(pkg))
            else:
                if self.store.get_value(iter, COL_PACKAGE).bad:
                    list_bad.append(self.store.get_value(iter, COL_PACKAGE).deb_filename)
            progress.set_fraction(percent)
            progress.set_text(str(int(percent * 100)) + '%' )
            #process pending events / update the window
            if index % 2 == 0:
                processEvents()

        processEvents()

        self.store.uncheck_minor_version()
        #self.count_packages()

        progress.set_fraction(0)
        progress.hide()
        progress.set_text('0%')
        progress.set_fraction(0)
        #update the parent count labels
        self.update_parent_count()
        self.controller.get_main_window().set_sensitive(True)
        self.set_model(self.store)
        self.get_selection().select_path((0,0))
        self.grab_focus()
        gui.setCursorToNormal(self.controller.get_parent_widget())

        #show packages not inserted due to already exist in list
        if len(not_added)>0:
            self.__load_dialog_skip(not_added)

        if len(list_bad)>0:
            self.__load_dialog_skip(list_bad)

    def update_parent_count(self, values = None):
        """
            This function will show count itens in main window.

        """
        if not values:
            values = self.store.get_count()

        self.controller.show_selected_count( (\
                                                      values['count'], utils.humanize_file_size(values['total_size']) \
                                                    , values['selected'], utils.humanize_file_size(values['selected_size']) \
                                                    , values['custom'], utils.humanize_file_size(values['custom_size'])))

    def show_package_property(self, pkg, image):
        prop = PackagePropety(pkg, self.controller, image)
        prop.show()

    # Events Handling
    def toggled_item(self, data, row):
        """Set a function to be called when the value of a row is toggled.
        The  function will be called with two arguments, the clicked item
        in the row and a string for which row was clicked."""
        gui.setCursorToBusy(self, True)
        iter = self.store.get_iter((int(row),))

        pkg = self.store.get_value(iter, COL_PACKAGE)

        if not pkg.selected:
            self.store.check(pkg)
        else:
            self.store.uncheck(pkg)

        try:
            self.controller.btnBurn.set_sensitive( self.store.get_count()['selected'] > 0)
        except:
            self.controller.btnRestorePackages.set_sensitive( self.store.get_count()['selected'] > 0)

        gui.setCursorToNormal(self)
        return

    def check_uncheck_all(self, check_type = 0):
        pkg = self.get_selected()
        gui.setCursorToBusy(self, True)
        tmp = self.autoselect_dependents
        self.autoselect_dependents = False
        for n in range(len(self.store)):
            pkg = self.store.get_value(self.store.get_iter((n,)),COL_PACKAGE)
            if check_type == 0 :
                self.store.check(pkg)
            elif check_type == 1:
                self.store.uncheck(pkg)
            elif check_type == 2:
                if not pkg.selected:
                    self.store.check(pkg)
                else:
                    self.store.uncheck(pkg)
        self.autoselect_dependents = tmp
        self.update_parent_count()
        gui.setCursorToNormal(self)
        try:
            self.controller.btnBurn.set_sensitive( self.store.get_count()['selected'] > 0)
        except:
            self.controller.btnRestorePackages.set_sensitive( self.store.get_count()['selected'] > 0)

    def menu_clicked(self, widget):
        pkg = self.get_selected()
        """
            This procedure will decide what to do when a user select
            an option from de contextMenu.
        """
        try:
            command =  widget.get_child().get_text()
        except Exception, msg:
            print str(msg)
            command = ''

        if command == constants.MESSAGE_0009:
            self.check_uncheck_all(0)
        elif command == constants.MESSAGE_0010:
            self.check_uncheck_all(1)
        elif command == constants.MESSAGE_0011:
            self.check_uncheck_all(2)
        elif command == constants.MESSAGE_0012:
            iter, selected = self.get_selected_iter()
            self.show_package_property(selected, get_icon(selected.package, 48))
        elif command == constants.MESSAGE_0062:
            os.system('/usr/bin/gdebi-gtk ' + os.path.join(pkg.source_path,pkg.deb_filename) + ' 2> /dev/null &')
        else:
            pass

    def on_row_activated(self, treeview, path, view_column):
        """
            This procedure will handle list doubleClick.
        """
        self.toggled_item('', path[0])

    def on_remove_clicked(self, widget):
        """
            This will remove a package from the list.
            Only Custom packages can be removed.
        """
        iter, selected = self.get_selected_iter()
        if selected:
            if gui.question_message(constants.MESSAGE_0008 % selected.package, self.controller.get_main_window()):
                self.store.remove_package(iter)
                self.update_parent_count()
        return

    def on_PackageList_right_click(self, widget, event):
        """ 
            This function will handle the signal to show a popup menu sent by 
            a right click on PackageList widget.
        """
        if event.button == 3 and event.type == gtk.gdk.BUTTON_PRESS and len(self.store) > 0:
            c = contextMenu.ContextMenu(self)
            try:
                #just to make sure that something is selected
                x = int(event.x)
                y = int(event.y)
                pthinfo = widget.get_path_at_pos(x, y)
                if pthinfo is not None:
                    path, col, cellx, celly = pthinfo
                    widget.grab_focus()
                    widget.set_cursor( path, col, 0) 

                #now shows the menu with correct options    
                pkg = self.get_selected()
                c.addMenuItem(constants.MESSAGE_0009, self.menu_clicked, gtk.STOCK_YES,True)
                c.addMenuItem(constants.MESSAGE_0010, self.menu_clicked, gtk.STOCK_CANCEL, True)
                c.addMenuItem(constants.MESSAGE_0011, self.menu_clicked, None)
                if pkg.custom:
                    c.addMenuItem('-', None)
                    c.addMenuItem('',self.on_remove_clicked ,gtk.STOCK_REMOVE)

                if not pkg.installed and not pkg.bad and not pkg.old:
                    if os.path.isfile('/usr/bin/gdebi-gtk'):
                        c.addMenuItem('-', None)
                        c.addMenuItem(constants.MESSAGE_0062 , self.menu_clicked ,get_icon('deb',16), True)
                if not pkg.bad:
                    c.addMenuItem('-', None)
                    c.addMenuItem(constants.MESSAGE_0012 , self.menu_clicked ,gtk.STOCK_PROPERTIES, True)
                c.popup(None, None, None, event.button, event.get_time())
            except:
                pass

    def __get_packages_list(self): return self.store
    packages_list = property(fget = __get_packages_list, doc = 'Returns inner packages liststore.')

    def __get_selected_count(self): return self.store.get_count()['selected']
    total_selected_count = property(fget = __get_selected_count, doc = 'Return the lenght of selecteds packages.')

    def __get_selected_size(self): return self.store.get_count()['selected_size']
    total_selected_size = property(fget = __get_selected_size, doc = 'Return the size of selecteds packages.')

    def __set_can_select_old(self, value):  self.select_old = value
    can_select_old = property(fset = __set_can_select_old, doc = 'Set if old packages can be selected.')

    def __set_auto_select_dependents(self, value):  self.autoselect_dependents = value
    auto_select_packages = property(fset = __set_auto_select_dependents, doc = 'Set if dependents packages can be selected automatically.')


###### try to implement a custom liststore to make things easier to handle

PACKAGE_TEXT = {
                  'normal': '<b>%s</b> (<small>%s</small>)\n<small>%s</small>', 
                  'bad': '<span color = "red"><b>%s</b> (<small>%s</small>)</span>\n<span color = "red"><small>%s</small></span>',
                  'custom': '<span color = "blue"><b>%s</b> (<small>%s</small>)</span>\n<span color = "blue"><small>%s</small></span>',
                  'old': '<span color = "#606060"><b>%s</b> (<small>%s</small>)</span>\n<span color = "#606060"><small>%s</small></span>',
                  'old_installed': '<span color = "blue"><b>%s</b> (<small>%s</small>)</span>\n<span color = "#blue"><small>%s</small></span>',
                  'normal_size' : '<small>%s</small>', 
                  'bad_size' : '<span color = "red"><small>%s</small></span>',
                  'custom_size' : '<span color = "blue"><small>%s</small></span>',
                  'old_size' : '<span color = "#606060"><small>%s</small></span>',
                }

COUNT_TEXT = '%s/%s'

#columns indexes
(COL_CHECK, COL_TITLE, COL_PACKAGE, COL_VERSION, COL_SIZE) = range(5)

class PackageModel(gtk.ListStore):
    """
        List model for packages.
    """

    def __init__(self, parent = None):
        gtk.ListStore.__init__(self, gobject.TYPE_BOOLEAN, str, object, str, str)

        self.controller = parent
        self.dictionary = {}
        self.installed = {}
        self.count = {'count':0,'total_size':0, 'selected':0, 'selected_size':0, 'custom':0, 'custom_size':0 }

        #custom sort for the list
        self.set_sort_func(COL_CHECK, self._compare, COL_CHECK)
        self.set_sort_column_id(COL_CHECK, gtk.SORT_ASCENDING)

        self.set_sort_func(COL_SIZE, self._compare, COL_SIZE)
        self.set_sort_column_id(COL_SIZE, gtk.SORT_ASCENDING)

        self.set_sort_func(COL_VERSION, self._compare, COL_VERSION)
        self.set_sort_column_id(COL_VERSION, gtk.SORT_ASCENDING)

        self.set_sort_func(COL_TITLE, self._compare, COL_TITLE)
        self.set_sort_column_id(COL_TITLE, gtk.SORT_ASCENDING)

    def clear(self):
        self.dictionary.clear()
        self.installed.clear()
        self.count = {'count':0,'total_size':0, 'selected':0, 'selected_size':0, 'custom':0, 'custom_size':0 }

    def _load_cache(self):
        """
            Creates a local cache with packages using apt_pkg cache.
            This will be used to check if a package is installed or not.
        """
        apt_pkg.init()
        cache = apt_pkg.GetCache()
        for package in cache.Packages:
            current = package.CurrentVer
            if not current:
                continue
            current_str = package.CurrentVer.VerStr
            name = package.Name
            self.installed[name] = current_str
        print 'got installed'

    def __create_package(self, package_name, path_to_scan, custom = False):
        """
            This will  create the package object with respective values
        """
        pkg = DebPackage(package_name, path_to_scan)
        pkg.custom = custom
        if pkg.bad:
            #we should not select a bad package
            pkg.selected = False
        else:
            pkg.selected = True
        
        if pkg.package in self.installed:
            versionstr = self.installed[pkg.package]
            if pkg.version == versionstr:
                pkg.installed = True
            else:
                pkg.installed =False
                pkg.selected = True 
        else:
            pkg.installed = False

        return pkg

    def __get_package_list_text(self,  pkg):
        size_text = ''
        title_text = ''
        if pkg.bad:
            title_text = PACKAGE_TEXT['bad'] % (pkg.package, pkg.version,  utils.escape(pkg.short_description))
            size_text = PACKAGE_TEXT['bad_size']  % pkg.size_text
        elif pkg.custom:
            title_text = PACKAGE_TEXT['custom'] % (pkg.package, pkg.version,  utils.escape(pkg.short_description))
            size_text = PACKAGE_TEXT['custom_size']  % pkg.size_text
            #update custom count
            self.count['custom'] +=1
            self.count['custom_size'] += pkg.size
        elif pkg.old and pkg.installed:
            title_text = PACKAGE_TEXT['old_installed'] % (pkg.package, pkg.version,  utils.escape(pkg.short_description))
            size_text = PACKAGE_TEXT['old_size']  % pkg.size_text
        elif pkg.old:
            title_text = PACKAGE_TEXT['old'] % (pkg.package, pkg.version,  utils.escape(pkg.short_description))
            size_text = PACKAGE_TEXT['old_size']  % pkg.size_text
        else:
                title_text = PACKAGE_TEXT['normal'] % (pkg.package, pkg.version,  utils.escape(pkg.short_description))
                size_text = PACKAGE_TEXT['normal_size']  % pkg.size_text
        return title_text,  size_text
        
    def add_item(self, package_name, path_to_scan, custom = False):
        """
        """
        size_text = ''
        title_text = ''
        pkg = self.__create_package(package_name, path_to_scan, custom)
        if not (pkg.package + str(pkg.version) in self.dictionary.keys()):
            title_text , size_text = self.__get_package_list_text(pkg)
            

            #result =  self.append([pkg.selected, title_text, pkg, size_text, pkg.package + str(pkg.version)])
            result =  self.append([pkg.selected, title_text, pkg, str(pkg.version), size_text])
            #increase our packages.count
            #self.count = {'count':0,'total_size':0, 'selected':0, 'selected_size':0, 'custom':0, 'custom_size':0 }
            self.count['count'] +=1
            self.count['total_size'] += pkg.size
            if pkg.selected:
                self.count['selected'] +=1
                self.count['selected_size'] += pkg.size

            #update our dictionary
            self.dictionary[pkg.package + str(pkg.version)] = result
            return result
        else:
            return None
    def uncheck_minor_version(self):
        """
            Uncheck packages having old versions.
            for key in self.dictionary.keys():
            pkg = self.get_value(self.dictionary[key],COL_PACKAGE)
            if pkg.package in self.installed.keys():
                if pkg.version != self.installed[pkg.package]:
                    if not self.check_package_is_newer(pkg):
                        self.uncheck(pkg)
            else:
                if not self.check_package_is_newer(pkg):
                        self.uncheck(pkg)
        """
        for key in self.dictionary.keys():
            pkg = self.get_value(self.dictionary[key],COL_PACKAGE)
            pkg.old = not self.check_package_is_newer(pkg)
            if pkg.old and not pkg.installed:
                text, size = self.__get_package_list_text(pkg)
                self.set_value(self.dictionary[key], COL_TITLE, text)
                self.set_value(self.dictionary[key], COL_SIZE, size)
                self.uncheck(pkg)

    def check_package_is_newer(self, item):
        """
            Check if package is the greater version.
        """
        is_newer = True

        start = 0
        end = len(self) - 1
        while start <= end:
            medium = (start + end) / 2
            if self[medium][COL_PACKAGE].package == item.package:
                if item.compare_version(self[medium][COL_PACKAGE].version) < 0: 
                    is_newer = False
                    break
            if self[medium][COL_PACKAGE].package < item.package: start = medium + 1
            else: end = medium - 1

        nReturn = True

        for tmp in self:
            iter = tmp[COL_PACKAGE]
            if item.package == iter.package:
                if item.compare_version(iter.version) < 0:
                    nReturn = False
        return nReturn

    def check_package_version(self, item):
        """
            Check if package is the greater version.
        """
        is_newer = True

        start = 0
        end = len(self) - 1
        while start <= end:
            medium = (start + end) / 2
            if self[medium][COL_PACKAGE].package == item[COL_PACKAGE].package:
                if item[COL_PACKAGE].compare_version(self[medium][COL_PACKAGE].version) < 0: 
                    is_newer = False
                    break
            if self[medium][COL_PACKAGE].package < item[COL_PACKAGE].package: 
                start = medium + 1
            else: 
                end = medium - 1

        nReturn = True

        for tmp in self:
            iter = tmp[COL_PACKAGE]
            if item[COL_PACKAGE].package == iter.package:
                if item[COL_PACKAGE].compare_version(iter.version) < 0:
                    nReturn = False
            #elif item.get_pkg_Name() <  iter.get_pkg_Name():
            #    break
        return nReturn

    def remove_package(self, pkg):
        """
            this will remove a package from the list.
            Only a custom package should be removed.

            @pkg must be a DebPackage object. 
        """
        if isinstance(pkg, gtk.TreeIter):
            pkg = self.get_value(pkg, COL_PACKAGE)

        pacote = pkg.package + str(pkg.version)

        if pacote in self.dictionary.keys():
            iter = self.dictionary[pacote]
            self.remove(iter)
            del self.dictionary[pacote]
            if pkg.selected:
                self.count['selected'] -=1
                self.count['selected_size'] -= pkg.size
            self.count['count'] -=1
            self.count['total_size'] -= pkg.size
            self.count['custom'] -=1
            self.count['custom_size'] -= pkg.size
            return True
        else:
            return False

    def check(self, pkg, checking = False):
        """
            This will mark a package as checked.

            Bad packages should not be checked.
            Old packages could be checked if allowed

            @pkg must be a DebPackage object.
            @checking Boolean value that will no allow deeper dependencie checker

        """
        pacote = pkg.package + str(pkg.version)
        if pacote in self.dictionary.keys():
            iter = self.dictionary[pacote]

        #if you can set old packages return
        if (pkg.old and not self.controller.select_old) or pkg.bad:
            return False

        if not pkg.selected:
            pkg.selected = True
            self.set_value(iter,COL_CHECK, True)
            self.count['selected'] += 1
            self.count['selected_size'] += pkg.size

            if pkg.custom:
                self.count['custom'] += 1
                self.count['custom_size'] += pkg.size

            if self.controller.autoselect_dependents and not checking:
                self.__check_depends(pkg.depends)
        self.controller.update_parent_count(self.count)

    def uncheck(self, pkg):
        """
            This will uncheck selected packages and reduce packages count.

            @pkg must be a DebPackage object.
        """
        pacote = pkg.package + str(pkg.version)

        if pacote in self.dictionary.keys():
            iter = self.dictionary[pacote]
        if pkg.selected:
            pkg.selected = False
            self.set_value(iter,COL_CHECK, False)

            if pkg.custom:
                self.count['custom'] -= 1
                self.count['custom_size'] -= pkg.size

            self.count['selected'] -= 1
            self.count['selected_size'] -= pkg.size

        self.controller.update_parent_count(self.count)

    def get_package(self, name):
        """
            Return the DebPackage object from its name.
        """
        if name in self.dictionary.keys():
            return self.dictionary[name][COL_PACKAGE]
        else:
            return None

    def __check_depends(self, dependent):
        """
           checks dependents packages recursively.
           @dependent - A list of packages names that a package depends.
        """
        tmp = self
        newdependents = []
        newdependents.append(dependent)

        for dependent in newdependents:
            for dep in dependent:
                item = self.__get_by_name(dep)
                if item != None :

                    if self.check_package_version(item):
                        self.check(item[COL_PACKAGE], True)

                    if not item[COL_PACKAGE].depends in newdependents:
                        newdependents.append(item[COL_PACKAGE].depends)

    def __get_by_name(self, iterText):
        """
            Get a package Iter  selected by its name.
        """
        itemFound = None

        start = 0
        end = len(self) - 1
        while start <= end:
            medium = (start + end) / 2
            if self[medium][COL_PACKAGE].package == iterText:
                itemFound = self[medium]
                break
            if self[medium][COL_PACKAGE].package < iterText: start = medium + 1
            else: end = medium - 1

        return itemFound

    def get_count(self):
        """
            Return the count dictionary.
        """
        return self.count

    def _compare(self, model, iter1, iter2, column_id):
        """
            Function used to make a custom compare method for PackageList
        """
        try:
            pkg = model.get_value(iter1,COL_PACKAGE)
            if column_id == COL_SIZE:
                a = pkg.size
            elif column_id == COL_VERSION:
                a = pkg.version
            elif column_id == COL_CHECK:
                a = pkg.selected
            else:
                a = pkg.package + str(pkg.version)
        except:
            a = None
        try:
            pkg = model.get_value(iter2,COL_PACKAGE)
            if column_id == COL_SIZE:
                b = pkg.size
            elif column_id == COL_VERSION:
                b = pkg.version
            elif column_id == COL_CHECK:
                b = pkg.selected
            else:
                b = pkg.package + str(pkg.version)
        except:
            b = None
        return  cmp(a, b)
