/***************************************************************************
 $RCSfile: createuser.cpp,v $
                             -------------------
    cvs         : $Id: createuser.cpp,v 1.15 2003/06/11 18:48:16 aquamaniac Exp $
    begin       : Sat Jan 26 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@aquamaniac.de
*/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "createuser.h"



bool _checkMedium(HBCI::Pointer<AQMAPI> hbciif,
		  bool isFile, string &mediumName, string &mediumTypeName) {
  HBCI::Pointer<HBCI::MediumPluginList> mediumPlugins;
  list<Pointer<MediumPlugin> >::const_iterator it;
  HBCI::Error err;

  fprintf(stderr,
	  "Checking for plugin which supports this medium, please wait.\n");

  mediumPlugins=hbciif.ref().enumerateMediumPlugins();
  if (!mediumPlugins.isValid()) {
    fprintf(stderr,
	    "Error listing installed plugins.\n"
	    "Please report this incidence to:\n"
	    "martin@libchipcard.de\n");
    return false;
  }
  if (mediumPlugins.ref().plugins().empty()) {
    fprintf(stderr,
	    "There are no plugins installed.\n"
	    "At least one plugin is shipped with "
	    "OpenHBCI,\n"
	    "so I guess OpenHBCI is not properly "
	    "installed.\n"
	    "You can get OpenHBCI along with plugins from:\n"
	    "http://www.openhbci.de\n");
    return false;
  }

  for (it=mediumPlugins.ref().plugins().begin();
       it!=mediumPlugins.ref().plugins().end();
       it++) {
    if (!(((*it).ref().mediumType()==MediumTypeFile) ^ isFile)) {
      // medium types are equal, let the plugin if this is his medium
      if (hbciif.ref().debugLevel()>2)
	fprintf(stderr,
		"Checking whether plugin \"%s\" supports this medium \n",
		(*it).ref().mediumTypeName().c_str());
      err=(*it).ref().mediumCheck(mediumName);
      if (!err.isOk()) {
	// error, now check what kind of
	if (err.code()==HBCI_ERROR_CODE_BAD_MEDIUM) {
	  // this plugin definitely does NOT support this medium, try next
	  if (hbciif.ref().debugLevel()>0)
	    fprintf(stderr,
		    "Medium not supported by plugin \"%s\"\n",
		    (*it).ref().mediumTypeName().c_str());
	  continue;
	}
	if (err.code()==HBCI_ERROR_CODE_WRONG_MEDIUM) {
	  // this plugin supports this medium, however the names do not match
	  fprintf(stderr,
		  "Found a plugin which supports this medium.\n"
		  "However, the name does not match.\n"
		  "Please specify the correct name or leave it blank.\n");
	  mediumTypeName=(*it).ref().mediumTypeName();
	  return false;
	}
	else {
	  // more severe error
	  fprintf(stderr, "ERROR: %s\n",
                  err.errorString().c_str());
	  return false;
	}
      } // if err.isOk()
      else {
	// medium plugin found
	fprintf(stderr,
		"Medium supported by plugin \"%s\"\n",
		(*it).ref().mediumTypeName().c_str());
	mediumTypeName=(*it).ref().mediumTypeName();
	return true;
      }
    } // if mediumtype matches
  } // for

  fprintf(stderr, "No plugin seems to support this medium. Sorry.\n");
  mediumTypeName.erase();
  return false;
}




HBCI::Error reallyCreateUser(HBCI::Pointer<AQMAPI> hbciif,
			     const string &mediumname,
			     const string &mediumType,
			     bool readonly,
			     int country,
			     const string &instid,
			     const string &userid,
			     const string &customerid,
			     const string &role,
			     const string &server) {
  HBCI::Pointer<HBCI::Bank> bank;
  HBCI::Pointer<HBCI::User> user;
  HBCI::Pointer<HBCI::Customer> cust;
  HBCI::Pointer<HBCI::Medium> medium;
  HBCI::Error err;

  // create medium
  fprintf(stderr,
	  "Creating this user:\n"
	  "  Bank code: %s\n"
	  "  User Id  : %s\n"
	  "  Server   : %s\n",
	  instid.c_str(),userid.c_str(),server.c_str());
  if (country==0 ||
      instid.empty() ||
      userid.empty() ||
      server.empty())
    return HBCI::Error("reallyCreateUser()",
		       ERROR_LEVEL_NORMAL,
		       0,
		       ERROR_ADVISE_DONTKNOW,
		       "empty arguments");

  // create user and medium
  try {
    // check if the bank exists
    bank=hbciif.ref().findBank(country,instid);
    if (!bank.isValid()) {
      // need to create it
      bank=hbciif.ref().bankFactory(country,
				    instid,
				    server);
      hbciif.ref().addBank(bank);
    }

    // check if user exists
    user=bank.ref().findUser(userid);
    if (!user.isValid()) {
      // need to create him

      // check if medium exists
      medium=hbciif.ref().findMedium(mediumname);
      if (!medium.isValid())
	// need to create the medium
	  medium=hbciif.ref().createNewMedium(mediumType,
					      readonly,
					      country,
					      instid,
					      userid,
					      mediumname);
      // then the user himself
      user= hbciif.ref().userFactory(bank,
				     medium,
				     userid);
      // add customer to the system
      bank.ref().addUser(user);

      // done
    }
    // create customer, if given
    if (!customerid.empty()) {
      cust=hbciif.ref().customerFactory(user,
					customerid,
					role);
      user.ref().addCustomer(cust);
    }

  } // try
  catch (HBCI::Error lerr) {
    err=lerr;
  }

  return err;
}


HBCI::Error importUser(HBCI::Pointer<AQMAPI> hbciif,
		       string mediumname,
		       const string &mtype,
		       bool readonly,
		       int country,
		       const string &instid,
		       const string &userid,
		       const string &customerid,
		       const string &role,
		       const string &server) {
  string tmp;
  HBCI::Pointer<HBCI::Medium> medium;
  int i;
  int j;
  int lcountry;
  string linstid;
  string luserid;
  string lserver;
  HBCI::Error err;

  // now import the contexts from the medium
  medium=hbciif.ref().mediumFactory(mediumname,mtype);
  try {
    i=1;
    j=0;
    err=medium.ref().mountMedium();
    if (!err.isOk())
      return HBCI::Error("importUser",err);
    while(1) {
      // get context
      err=medium.ref().getContext(i,
				  lcountry,
				  linstid,
				  luserid,
				  lserver);
      if (!err.isOk())
	break;
      // add context, if it matches the criteria
      if (
	  (!country || lcountry==country) &&
	  (-1!=HBCI::parser::cmpPattern(linstid,
					instid,
					false) || instid.empty())
	 ) {
	mediumname=medium.ref().mediumName();
	err=reallyCreateUser(hbciif,
			     mediumname,
			     mtype,
			     readonly,
			     lcountry,
			     linstid,
			     userid.empty()?luserid:userid,
			     customerid,
			     role,
			     lserver.empty()?server:lserver);
	if (!err.isOk())
	  fprintf(stderr,"%s\n",
		  err.errorString().c_str());
	j++;
      }
      i++;
    } // while
    medium.ref().unmountMedium();
  } // try
  catch (HBCI::Error lerr) {
    fprintf(stderr,"Exception: %s\n",
	    lerr.errorString().c_str());
    return HBCI::Error("importUser",err);
  }
  if (!j) {
    fprintf(stderr,"No user added.\n");
    return HBCI::Error("importUser()",
		       ERROR_LEVEL_NORMAL,
		       0,
		       ERROR_ADVISE_ABORT,
		       "no user added");
  }

  return HBCI::Error();
}



HBCI::Error createUser(HBCI::Pointer<AQMAPI> hbciif,
		       HBCI::SimpleConfig &opt){
  int country;
  string userid;
  string customerid;
  string role;
  string instid;
  string mediumname;
  string server;
  string tmp;
  HBCI::Pointer<HBCI::Bank> bank;
  HBCI::Pointer<HBCI::User> cust;
  HBCI::Pointer<HBCI::Medium> medium;
  HBCI::Error err;
  string mtypename;
  bool import;
  bool readonly;
  bool isFile;

  cust.setDescription("createUser::cust (HBCI::User)");

  // get some variables
  country=opt.getIntVariable("country",280,opt.root());
  instid=opt.getVariable("instid","",opt.root());
  userid=opt.getVariable("user","",opt.root());
  customerid=opt.getVariable("customer","",opt.root());
  role=opt.getVariable("role","",opt.root());
  mediumname=opt.getVariable("medium","",opt.root());
  server=opt.getVariable("server","",opt.root());
  import=opt.findVariable("import",opt.root(),false).isValid();
  readonly=opt.findVariable("ro-medium",opt.root(),false).isValid();

  // get the medium type
  tmp=opt.getVariable("mediumtype","file",opt.root());
  mtypename.erase();
  if (-1!=HBCI::parser::cmpPattern(tmp,"file",false)) {
    isFile=true;
    if (mediumname.empty()) {
      fprintf(stderr,"File name needed (\"--mediumname=NAME\")\n");
      return HBCI::Error("createUser()",
			 ERROR_LEVEL_NORMAL,
			 0,
			 ERROR_ADVISE_DONTKNOW,
			 "medium name missing");
    }
    if (mediumname.at(0)!='/') {
      // make medium path absolute
      mediumname=HBCI::Directory::workingDirectory()+"/"+mediumname;
    }
  }
  else if (-1!=HBCI::parser::cmpPattern(tmp,"card",false)) {
    isFile=false;
  }
  else {
    if (!import) {
      fprintf(stderr,
	      "Please use only \"file\" or "
	      "\"card\" as argument to \"mediumtype\"\n");
      return HBCI::Error("createUser()",
			 ERROR_LEVEL_NORMAL,
			 0,
			 ERROR_ADVISE_DONTKNOW,
			 "bad medium type");
    }
  }

  if (import || !isFile) {
    // get the medium type name for this medium
    if (!_checkMedium(hbciif, isFile, mediumname, mtypename)) {
      fprintf(stderr, "Could not determine the correct medium plugin.\n");
      return HBCI::Error("createUser()",
			 ERROR_LEVEL_NORMAL,
			 0,
			 ERROR_ADVISE_DONTKNOW,
			 "bad medium type");
    }
  }
  else if (isFile && !import) {
    mtypename="RDHFile";
  }

  if (import)
    return importUser(hbciif,
		      mediumname,
		      mtypename,
                      readonly,
		      country,
		      instid,
		      userid,
		      customerid,
		      role,
		      server);
  return reallyCreateUser(hbciif,
			  mediumname,
			  mtypename,
			  readonly,
			  country,
			  instid,
			  userid,
			  customerid,
			  role,
			  server);
}


