# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::Widget::Inventory::Tree;

use Glib qw(TRUE FALSE);
use Gtk2;

use base qw(Gtk2::TreeView);

use Arch::Inventory qw(:type);

use constant MODEL_COL_COUNT  => 2;
use constant MODEL_COL_TYPES  => qw(Glib::String Glib::String);
use constant MODEL_COL_TITLES => qw(Tree Path); 
use constant MODEL_COL_NAME   => 0;
use constant MODEL_COL_PATH   => 1;

sub new ($$) {
	my $class     = shift;
	my $inventory = shift;

	my $self = bless Gtk2::TreeView->new(), $class;
	$self->set_model(Gtk2::TreeStore->new(MODEL_COL_TYPES));

	$self->_init();
	$self->show($inventory) if ($inventory);

	return $self;
}

sub _init ($) {
	my $self = shift;

	for my $col (MODEL_COL_NAME) {
		my $column = Gtk2::TreeViewColumn->new_with_attributes(
			(MODEL_COL_TITLES)[$col], Gtk2::CellRendererText->new(),
			text => $col,
		);
		$column->set_resizable(TRUE);

		$self->append_column($column);
	}

	$self->set_rules_hint(TRUE);
}

sub get_selected_path ($) {
	my $self = shift;

	my $iter = $self->get_selection()->get_selected();
	return $iter ? $self->get_model()->get($iter, MODEL_COL_PATH) : undef;
}

sub get_selected_parent_path ($) {
	my $self = shift;

	my $iter = $self->get_selection()->get_selected();
	$iter = $self->get_model()->iter_parent($iter) if (defined $iter);

	return $iter ? $self->get_model()->get($iter, MODEL_COL_PATH) : undef;
}

sub select_by_path ($$) {
	my $self = shift;
	my $path = shift;

	my $model = $self->get_model();
	my $iter  = $model->get_iter_first();
	while (defined $iter) {
		my $ipath = $model->get($iter, MODEL_COL_PATH);

		if ($ipath eq $path) {
			$self->expand_to_path($self->get_model->get_path($iter));
			$self->get_selection()->select_iter($iter);
			last;
		} elsif (($ipath eq '') || (substr($path, 0, length($ipath)+1) eq "$ipath/")) {
			$iter = $model->iter_children($iter);
		} else {
			$iter = $model->iter_next($iter);
		}
	}
}

sub show ($$) {
	my $self      = shift;
	my $inventory = shift;

	my $model = $self->get_model();
	$model->clear();

	return unless (defined $inventory);

	my $top = $model->append(undef);
	$model->set($top,
		MODEL_COL_NAME, '/',
		MODEL_COL_PATH, $inventory->{path},
	);

	tree_model_append($model, $top, $inventory);

	$self->expand_all();
	$self->get_selection()->select_iter($top);
}

sub tree_model_append ($$$) {
	my ($model, $parent, $entry) = @_;

	foreach my $name (sort keys %{$entry->{children}}) {
		my $child = $entry->{children}->{$name};

		next unless ($child->{type} eq DIRECTORY);

		my $iter = $model->append($parent);
		$model->set($iter,
			MODEL_COL_NAME, $name,
			MODEL_COL_PATH, $child->{path},
		);

		&tree_model_append($model, $iter, $child);
	}
}

1;

__END__
