# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::Widget::Revisions;

use Glib qw(TRUE FALSE);
use Gtk2;

use base qw(Gtk2::TreeView);

use Arch::Util qw(date2daysago);

use constant MODEL_COL_TYPES => qw(
	Glib::Boolean  
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::Boolean
	Glib::Uint
	Glib::Uint
);

use constant MODEL_COL_IS_REV   => 0;
use constant MODEL_COL_NAME     => 1;
use constant MODEL_COL_SUMMARY  => 2;
use constant MODEL_COL_DATE     => 3;
use constant MODEL_COL_CREATOR  => 4;
use constant MODEL_COL_REVISION => 5;
use constant MODEL_COL_STRIKE   => 6;
use constant MODEL_COL_WEIGHT   => 7;
use constant MODEL_COL_STYLE    => 8;

my @VIEW_COLUMNS = (
	{
		title      => 'Name',
		column     => MODEL_COL_NAME,
		attributes => {
			strikethrough => MODEL_COL_STRIKE,
			weight        => MODEL_COL_WEIGHT,
			style         => MODEL_COL_STYLE,
		},
	},
	{
		title      => 'Summary',
		column     => MODEL_COL_SUMMARY,
		attributes => {
			weight => MODEL_COL_WEIGHT,
			style  => MODEL_COL_STYLE,
		},
	},
	{
		title      => 'Date',
		column     => MODEL_COL_DATE,
		attributes => {
			weight => MODEL_COL_WEIGHT,
			style  => MODEL_COL_STYLE,
		},

	},
	{
		title      => 'Creator',
		column     => MODEL_COL_CREATOR,
		attributes => {
			weight => MODEL_COL_WEIGHT,
			style  => MODEL_COL_STYLE,
		},
	},
);

sub new ($) {
	my $class = shift;

	my $self = bless Gtk2::TreeView->new, $class;
	$self->set_model(Gtk2::TreeStore->new(MODEL_COL_TYPES));

	$self->_init;

	return $self;
}

sub _init ($) {
	my $self = shift;

	$self->set_rules_hint(TRUE);
	$self->set_search_column(MODEL_COL_NAME);

	foreach my $col (@VIEW_COLUMNS) {
		my $column = Gtk2::TreeViewColumn->new_with_attributes(
			$col->{title}, Gtk2::CellRendererText->new,
			text => $col->{column},
			%{$col->{attributes}},
		);
		$column->set_resizable(TRUE);
		$column->set_sort_column_id($col->{column});

		$self->append_column($column);
	}
}

sub get ($$;@) {
	my $self = shift;
	my $iter = shift;

	return $self->get_model->get($iter, @_);
}

sub get_selected ($;@) {
	my $self = shift;

	my $iter = $self->get_selection->get_selected;

	return $iter ? $self->get($iter, @_) : undef;
}

sub get_selected_revision ($) {
	my $self = shift;

	return $self->get_selected(MODEL_COL_REVISION);
}

sub get_selected_is_revision ($) {
	my $self = shift;

	return $self->get_selected(MODEL_COL_IS_REV);
}

sub clear ($) {
	my $self = shift;

	my $model = $self->get_model;
	$model->clear;
}

sub add_revisions ($$;%) {
	my $self = shift;
	my @revisions = @{shift()};
	my %args = @_;
	my $is_missing = $args{missing};

	while (@revisions) {
		my @revs = splice @revisions, 0, 25;

		my ($name, $summary, $date, $creator);

		$summary = scalar(@revs);
		$summary .= ' revision';
		$summary .= 's' if @revs != 1;
		$summary .= ' (missing)' if $is_missing;

		$name = $revs[0]->{name};
		$date = date2daysago($revs[0]->{date});
				
		if (@revs > 1) {
			$name .= ' .. ' . $revs[-1]->{name};
			$date .= ' .. ' . date2daysago($revs[-1]->{date});
		}

		$date .= ' days ago';

		$creator = $revs[0]->{creator};
		$creator = 'multiple creators'
			if grep { $_->{creator} ne $creator } @revs;

		$self->add_rev_bunch(
			$name,
			$summary,
			$date,
			$creator,
			$is_missing,
			$args{version},
			\@revs,
		);
	}
}

sub add_rev_bunch ($$$$$$$$) {
	my $self = shift;
	my ($name, $summary, $date, $creator, $is_missing, $version, $revisions) = @_;

	my $model = $self->get_model;

	my $parent = $model->append(undef);
	$model->set($parent,
		MODEL_COL_IS_REV,   0,
		MODEL_COL_NAME,     $name,
		MODEL_COL_SUMMARY,  $summary,
		MODEL_COL_DATE,     $date,
		MODEL_COL_CREATOR,  $creator,
		MODEL_COL_REVISION, undef,
		MODEL_COL_STRIKE,   $is_missing ? TRUE : FALSE,
		MODEL_COL_WEIGHT,   700,       # bold
		MODEL_COL_STYLE,    0,
	);

	model_append_revisions($model, $parent, $is_missing, $version, $revisions);

	my $first = $model->iter_nth_child(undef, 0);
	$self->get_selection->select_iter($first)
		if defined $first;
}

sub model_append_revisions ($$$$$) {
	my ($model, $parent, $is_missing, $version, $revisions) = @_;

	foreach my $rev (@$revisions) {
		my $iter = $model->append($parent);
		$model->set($iter,
			MODEL_COL_IS_REV,   1,
			MODEL_COL_NAME,     $rev->{name},
			MODEL_COL_SUMMARY,  $rev->{summary},
			MODEL_COL_DATE,     $rev->{date},
			MODEL_COL_CREATOR,  "$rev->{creator} <$rev->{email}>",
			MODEL_COL_REVISION, $version ? $version . '--' . $rev->{name} : $rev->{name},
			MODEL_COL_STRIKE,   $is_missing ? TRUE : FALSE,
			MODEL_COL_WEIGHT,   0,
			MODEL_COL_STYLE,    $is_missing ? 2 : 0,
		);
	}
}	

1;

__END__
