/*
    Copyright (C) 2002 Paul Davis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: gain_pan_meter.cc,v 1.37 2004/01/29 04:56:46 pbd Exp $
*/

#include <limits.h>

#include <ardour/io.h>
#include <ardour/dB.h>
#include <gtkmmext/utils.h>
#include <gtkmmext/fastmeter.h>
#include <gtkmmext/stop_signal.h>
#include <midi++/manager.h>
#include <pbd/fastlog.h>

#include "ardour_ui.h"
#include "gain_pan_meter.h"
#include "panner.h"
#include "utils.h"
#include "logmeter.h"

#include <ardour/session.h>
#include <ardour/route.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace Gtkmmext;
using namespace Gtk;
using namespace SigC;

GainPanMeter::GainPanMeter (IO& io, Session& s)
	: _io (io),
	  _session (s),
	  gain_slider (0),
	  // 0.781787 is the value needed for gain to be set to 0.
	  gain_adjustment (0.781787, 0.0, 1.0, 0.01, 0.1),
	  gain_display (&gain_adjustment, "MixerStripGainDisplay"),
	  gain_unit_label (_("dbFS")),
	  meter_pre_label (_("pre")),
	  top_table (1, 2)
	
{
	ignore_toggle = false;
	meter_menu = 0;

	gain_slider = manage (new VSliderController (ARDOUR_UI::instance()->mixer_slider_pix(),
						     &gain_adjustment,
						     & _io.midi_gain_control(),
						     false));

	gain_slider->fader().button_press_event.connect (slot (*this, &GainPanMeter::start_gain_touch));
	gain_slider->fader().button_release_event.connect (slot (*this, &GainPanMeter::end_gain_touch));

	/* XXX should we really bind to the MMC port, or a different port? */

	if (_session.mmc_port()) {
		_io.midi_gain_control().midi_rebind (_session.mmc_port(), 0);
		_io.set_midi_to_gain_function (slider_position_to_gain);
	}

	gain_display.set_print_func (_gain_printer, this);
	set_usize_to_display_given_text (gain_display, "-86.0", 2, 2);

	gain_unit_button.add (gain_unit_label);
	gain_unit_button.set_name ("MixerStripGainUnitButton");
	gain_unit_label.set_name ("MixerStripGainUnitButton");

	top_table.set_col_spacings (2);
	top_table.set_homogeneous (true);
	top_table.attach (gain_unit_button, 0, 1, 0, 1);

	Route* r;

	if ((r = dynamic_cast<Route*> (&_io)) != 0) {
		r->meter_change.connect (slot (*this, &GainPanMeter::meter_changed));
		meter_pre_button.add (meter_pre_label);
		meter_pre_button.set_name ("MixerStripMeterPreButton");
		meter_pre_label.set_name ("MixerStripMeterPreButton");
		meter_pre_button.set_active (r->meter_pre_fader());
		meter_pre_button.toggled.connect (slot (*this, &GainPanMeter::meter_pre_toggled));
		top_table.attach (meter_pre_button, 1, 2, 0, 1);
	}

	gain_display_box.set_spacing (2);
	gain_display_frame.set_shadow_type (GTK_SHADOW_IN);
	gain_display_frame.set_name ("BaseFrame");
	gain_display_frame.add (gain_display);
	gain_display_box.pack_start (gain_display_frame, false, false);

	peak_display.set_name ("MixerStripPeakDisplay");
	set_usize_to_display_given_text (peak_display, "-86.0", 2, 2);
	peak_display.add (peak_display_label);
	peak_display_frame.set_shadow_type (GTK_SHADOW_IN);
	peak_display_frame.set_name ("BaseFrame");
	peak_display_frame.add (peak_display);
	max_peak = minus_infinity();
	peak_display_label.set_text (_("-inf"));
	
	gain_display_box.pack_start (peak_display_frame, false, false);
	
	panner = new Panner (61, 61);
	panner->set_name ("MixerPanZone");

	pan_frame.add (*panner);
	pan_frame.set_shadow_type (GTK_SHADOW_IN);
	pan_frame.set_name ("BaseFrame");
	pan_box.pack_start (pan_frame, true, false);

	meter_metric_area.set_usize (18, -1);
	meter_metric_area.set_name ("MeterMetricsStrip");

	meter_packer.show ();
	gain_slider->show_all ();
	pan_box.show ();
	panner->show ();
	pan_box.show ();
	pan_frame.show ();

	meter_packer.set_spacing (2);
	fader_box.set_spacing (2);

	fader_box.pack_start (*gain_slider, false, false);

	hbox.set_spacing (4);
	hbox.pack_start (fader_box, false, false);
	hbox.pack_start (meter_packer, false, false);

	set_spacing (4);
	pack_start (top_table, false, false);
	pack_start (gain_display_box, false, false);
	pack_start (hbox, false, false);

	show_all ();

	_io.gain_changed.connect (slot (*this, &GainPanMeter::gain_changed));
	_io.pan_changed.connect (slot (*this, &GainPanMeter::pan_changed));

	meter_metric_area.expose_event.connect (slot (*this, &GainPanMeter::meter_metrics_expose));

	gain_adjustment.value_changed.connect (slot (*this, &GainPanMeter::gain_adjusted));
	panner->PuckMoved.connect (slot (*this, &GainPanMeter::panner_moved));
	panner->BypassToggled.connect (slot (*this, &GainPanMeter::panner_bypass_toggled));

	peak_display.button_release_event.connect (slot (*this, &GainPanMeter::peak_button_release));

	gain_changed (0);
	pan_changed (0);
}

void
GainPanMeter::set_width (Width w)
{
	switch (w) {
	case Wide:
		panner->set_usize (61, 61);
		peak_display_frame.show_all();
		break;
	case Narrow:
		panner->set_usize (31, 61);
		peak_display_frame.hide_all();
		break;
	}

	_width = w;
	setup_meters ();
}

gint
GainPanMeter::meter_metrics_expose (GdkEventExpose *ev)
{
	/* XXX optimize this so that it doesn't do it all everytime */

	double fraction;

	Gdk_Window win (meter_metric_area.get_window());
	Gdk_GC gc (meter_metric_area.get_style()->get_fg_gc (GTK_STATE_NORMAL));
	Gdk_Font font (meter_metric_area.get_style()->get_font());
	gint x, y, width, height, depth;
	gint pos;
	int  db_points[] = { -50, -10, -3, 0, 6 };
	unsigned int i;
	char buf[32];

	win.get_geometry (x, y, width, height, depth);

	for (i = 0; i < sizeof (db_points)/sizeof (db_points[0]); ++i) {
		fraction = log_meter (db_points[i]);
		pos = height - (gint) floor (height * fraction);

		snprintf (buf, sizeof (buf), "%d", db_points[i]);

		gint twidth;
		gint lbearing;
		gint rbearing;
		gint ascent;
		gint descent;
		
		gdk_string_extents (font,
				    buf,
				    &lbearing,
				    &rbearing,
				    &twidth,
				    &ascent,
				    &descent);
		
		win.draw_text (font, gc, width - twidth, pos + ascent, buf, strlen (buf));
		// win.draw_line (gc, 0, pos, width, pos);
	}

	return TRUE;
}

GainPanMeter::~GainPanMeter ()
{
	for (vector<MeterInfo>::iterator i = meters.begin(); i != meters.end(); i++) {
		if ((*i).meter) {
			delete (*i).meter;
		}
	}
}

void
GainPanMeter::update_meters ()
{
	vector<MeterInfo>::iterator i;
	unsigned long n;
	float peak;
	char buf[32];
	
	for (n = 0, i = meters.begin(); i != meters.end(); ++i, ++n) {
		if ((*i).packed) {
			peak = _io.peak_input_power (n);
			(*i).meter->set (log_meter (_io.peak_input_power (n)));
			if (peak > max_peak) {
				max_peak = peak;
				/* set peak display */
				snprintf (buf, sizeof(buf), "%.1f", max_peak);
				peak_display_label.set_text (buf);
			}
		}
	}
}

void
GainPanMeter::hide_all_meters ()
{
	bool remove_metric_area = false;

	for (vector<MeterInfo>::iterator i = meters.begin(); i != meters.end(); ++i) {
		if ((*i).packed) {
			remove_metric_area = true;
			meter_packer.remove (*((*i).meter));
			(*i).packed = false;
		}
	}

	if (remove_metric_area) {
		if (meter_metric_area.get_parent()) {
			meter_packer.remove (meter_metric_area);
		}
	}
}

void
GainPanMeter::setup_meters ()
{
	unsigned int nmeters;
	guint16 width;
	
	hide_all_meters ();

	Route* r;

	if ((r = dynamic_cast<Route*> (&_io)) != 0) {
		if (r->meter_pre_fader()) {
			nmeters = r->input_io().n_inputs();
		} else {
			nmeters = r->output_io().n_outputs();
		}
	} else {
		nmeters = _io.n_outputs();
	}

	if (nmeters == 0) {
		return;
	}

	if (_width == Wide) {
		meter_packer.pack_start (meter_metric_area, false, false);
		meter_metric_area.show_all ();
	}

	if (nmeters <= 2) {
		width = regular_meter_width;
	} else {
		width = thin_meter_width;
	}

	while (meters.size() < nmeters) {
		meters.push_back (MeterInfo());
	}

	for (unsigned int n = 0; n < nmeters; ++n) {
		if (meters[n].width != width) {
			delete meters[n].meter;
			meters[n].meter = new FastMeter ((unsigned long) floor (_session.meter_hold), width, FastMeter::Vertical);
			meters[n].width = width;

			meters[n].meter->add_events (GDK_BUTTON_RELEASE_MASK);
			meters[n].meter->button_release_event.connect
				(bind (slot (*this, &GainPanMeter::meter_button_release), n));
			meters[n].meter->button_release_event.connect_after (slot (do_not_propagate));
		}

		meter_packer.pack_start (*meters[n].meter, false, false);
		meters[n].meter->show_all ();
		meters[n].packed = true;
	}
}	

gint
GainPanMeter::peak_button_release (GdkEventButton* ev)
{
	/* reset peak label */
	
	max_peak = minus_infinity();
	peak_display_label.set_text (_("-inf"));

	return TRUE;
}

gint
GainPanMeter::meter_button_release (GdkEventButton* ev, unsigned int which)
{
	switch (ev->button) {
	case 1:
		meters[which].meter->clear();
		max_peak = minus_infinity();
		peak_display_label.set_text (_("-inf"));
		break;

	case 3:
		popup_meter_menu (ev);
		break;
	};

	return TRUE;
}

void
GainPanMeter::popup_meter_menu (GdkEventButton *ev)
{
	using namespace Menu_Helpers;

	if (meter_menu == 0) {
		meter_menu = new Gtk::Menu;
		MenuList& items = meter_menu->items();

		items.push_back (MenuElem ("-inf .. +0dBFS"));
		items.push_back (MenuElem ("-10dB .. +0dBFS"));
		items.push_back (MenuElem ("-4 .. +0dBFS"));
		items.push_back (SeparatorElem());
		items.push_back (MenuElem ("-inf .. -2dBFS"));
		items.push_back (MenuElem ("-10dB .. -2dBFS"));
		items.push_back (MenuElem ("-4 .. -2dBFS"));
	}

	meter_menu->popup (1, ev->time);
}

void
GainPanMeter::_gain_printer (char buf[32], Gtk::Adjustment& adj, void *arg)
{
	static_cast<GainPanMeter *>(arg)->gain_printer (buf, adj);
}

void
GainPanMeter::gain_printer (char buf[32], Gtk::Adjustment& adj)
{
	float v = adj.get_value();
	
	if (v == 0.0) {
		strcpy (buf, _("-inf"));
	} else {
		snprintf (buf, 32, "%.1f", coefficient_to_dB (slider_position_to_gain (v)));
	}
}

void
GainPanMeter::gain_adjusted ()
{
	if (!ignore_toggle) {
		_io.set_gain (slider_position_to_gain (gain_adjustment.get_value()), this);
	}
}

void
GainPanMeter::update_gain_display ()
{
	gfloat value = gain_to_slider_position (_io.effective_gain());

	if (gain_adjustment.get_value() != value) {
		ignore_toggle = true;
		gain_adjustment.set_value (value);
		ignore_toggle = false;
	}
}

void
GainPanMeter::gain_changed (void *src)
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &GainPanMeter::update_gain_display));
}

void
GainPanMeter::setup_pan ()
{
	unsigned long nouts = _io.n_outputs();
	unsigned long nins = _io.n_inputs();

	panner->drop_targets ();

	if (nouts <= 1 || nins > 1) {
		panner->hide_puck ();
		panner->allow_x_motion (false);
	} else {
		panner->show_puck ();
		panner->allow_x_motion (true);
	}
	
	if (nouts > 2) {
		panner->allow_y_motion (true);
	} else {
		panner->allow_y_motion (false);
	}

	/* if all pans are 1.0 then pans are considered bypassed */
	bool byp = true;
	for (unsigned long n = 0; n < nouts; ++n) {
		if (_io.pans()[n] != 1.0) {
			byp = false;
			break;
		}
	}
	panner->set_bypassed (byp);
	
	/* we don't pan if there is more than 1 input */

	if (nins > 1) {
		panner->set_bypassed (true);
		return;
	}

	switch (nouts) {
	case 0:
	case 1:
		/* relax */
		break;

	case 2:
		panner->add_target (0, 0.5);
		panner->add_target (1.0, 0.5);
		break;

	case 3:
		panner->add_target (0.5, 0);
		panner->add_target (0, 0.75);
		panner->add_target (1.0, 0.75);
		panner->allow_target_motion (false);
		break;	
	case 4:
		panner->add_target (0, 0);
		panner->add_target (1.0, 0);
		panner->add_target (1.0, 1.0);
		panner->add_target (0, 1.0);
		break;	

	case 5:
		panner->add_target (0, 0);
		panner->add_target (1.0, 0);
		panner->add_target (1.0, 1.0);
		panner->add_target (0, 1.0);
		panner->add_target (0.5, 0.75);
		break;

	default:
		for (unsigned long n = 0; n < nouts; ++n) {
			panner->add_target (0.1 * n, 0.1 * n);
		}
		break;
	}

	if (nouts <= 2) {
		panner->allow_target_motion (false);
	} else {
		panner->allow_target_motion (true);
	}
}

void
GainPanMeter::effective_pan_display ()
{
        float right;
	unsigned long nouts = _io.n_outputs();
	pan_t v;

	switch (nouts) {
	case 0:
	case 1:
		/* relax */
		break;

	case 2:
		v = _io.effective_stereo_pan();
                if (v > 1.0) v = 1.0;
                right = sin (acos (v));
                panner->move_puck (pan_value (v, right), 0.5);
		break;

	case 3:
		break;	
	case 4:
		break;	

	case 5:
		break;

	default:
		break;
	}
}

void
GainPanMeter::pan_changed (void *src)
{
	if (src == this) {
		return;
	}
	
	unsigned long nouts = _io.n_outputs();
	const vector<pan_t>& pans = _io.pans();

	switch (nouts) {
	case 0:
	case 1:
		/* relax */
		break;

	case 2:
		panner->move_puck (pan_value (pans[0], pans[1]), 0.5);
		break;

	case 3:
		break;	
	case 4:
		break;	

	case 5:
		break;

	default:
		break;
	}
}

void
GainPanMeter::panner_moved ()
{
	unsigned long nouts = _io.n_outputs();

	if (panner->bypassed()) {
		return;
	}

#define STEVE_HARRIS
#ifdef STEVE_HARRIS
	static const float BIAS = FLT_MIN;
	unsigned long i;
	float xp[nouts];
	float yp[nouts];
	float dsq[nouts];
	float f, fr;
	vector<pan_t> pans;
	float x;
	float y;

	/* the puck is target zero */

	panner->position (0, x, y);

	/* get the position of each speaker */

	for (i = 0; i < nouts; ++i) {
		panner->position (i+1, xp[i], yp[i]);
	}
	f = 0.0f;

	for (i = 0; i < nouts; i++) {
		dsq[i] = 1.0 - ((x - xp[i]) * (x - xp[i]) + (y - yp[i]) * (y - yp[i]) + BIAS);
		if (dsq[i] < 0.0) {
		   dsq[i] = 0.0;
		}
		f += dsq[i] * dsq[i];
	}
	fr = 1.0f / sqrtf(f);
	
	for (i = 0; i < nouts; i++) {
		pans.push_back (dsq[i] * fr);
	}
	
	_io.set_pans (pans, this);

#else
	float sval;

	switch (nouts) {
	case 0:
	case 1:
		/* relax */
		break;

	case 2:
		if (panner->fraction(1,2, sval) == 0) {
			equal_power_stereo_pan (sval);
		}
		break;

	case 3:
		break;	
	case 4:
		break;	

	case 5:
		break;

	default:
		break;
	}
#endif	
}

void
GainPanMeter::panner_bypass_toggled ()
{
	vector<pan_t> pans;
	unsigned long nouts = _io.n_outputs();
	unsigned long nins = _io.n_inputs();

	
	if (panner->bypassed()) {
		/* all pans to full gain */
		for (unsigned int i=0; i < nouts; ++i) {
			pans.push_back (1.0);
		}
		_io.set_pans (pans, this);
	}
	else if (nins <= 1) {
		// reset if not more than one input
		float scale = 1.0 / nouts;
		for (unsigned int i=0; i < nouts; ++i) {
			pans.push_back (scale);
		}
		_io.set_pans (pans, this);
		
		panner_moved();
	}
}

void
GainPanMeter::equal_power_stereo_pan (float val)

{
        double angle;
        double cosine;
        double sine;
	vector<pan_t> pans;

        /* val is assumed to vary from 0 to +1, but angle
           varies from -45 to +45 degrees, and must be given
           in radians.
        */

        angle = (2 * val - 1) * M_PI / 4;
        cosine = cos (angle);
        sine = sin (angle);
        
        pans.push_back (M_SQRT1_2 * (cosine - sine));
        pans.push_back (M_SQRT1_2 * (cosine + sine));
        
	_io.set_pans (pans, this);
}

float
GainPanMeter::pan_value(float left, float right)
{
	return 2 * asin ((right - left) / M_SQRT2) / M_PI + 0.5;
}

void
GainPanMeter::set_meter_strip_name (string name)
{
	meter_metric_area.set_name (name);
}

void
GainPanMeter::set_sensitive (bool yn)
{
	panner->set_sensitive (yn);
	static_cast<Gtkmmext::SliderController*>(gain_slider)->set_sensitive (yn);
}

void
GainPanMeter::meter_pre_toggled ()
{
	Route* r;

	if ((r = dynamic_cast<Route*> (&_io)) != 0) {
		bool x = meter_pre_button.get_active ();

		if (x != r->meter_pre_fader()) {
			r->set_meter_pre_fader (x, this);
		}
	}
}

void
GainPanMeter::meter_changed (void *src)
{
	Route* r;

	if ((r = dynamic_cast<Route*> (&_io)) != 0) {
		bool x = r->meter_pre_fader();
		if (x != meter_pre_button.get_active()) {
			meter_pre_button.set_active (x);
		}
		setup_meters ();
	}
}

gint
GainPanMeter::start_gain_touch (GdkEventButton* ev)
{
	_io.start_gain_touch ();
	return FALSE;
}

gint
GainPanMeter::end_gain_touch (GdkEventButton* ev)
{
	_io.end_gain_touch ();
	return FALSE;
}
