/*
    Copyright (C) 2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: marker_time_axis_view.cc,v 1.10 2004/02/29 23:33:55 pauld Exp $
*/

#include <algorithm>

#include <gtk--.h>
#include <gtkmmext/gtk_ui.h>

#include "marker_time_axis_view.h"
#include "marker_time_axis.h"
#include "marker_view.h"
#include "imageframe_view.h"
#include "imageframe_time_axis.h"
#include "canvas-simplerect.h"
#include "public_editor.h"
#include "rgb_macros.h"
#include "i18n.h"

using namespace ARDOUR ;

//---------------------------------------------------------------------------------------//
// Constructor / Desctructor
		
/**
 * Construct a new MarkerTimeAxisView helper time axis helper
 *
 * @param mta the TimeAxsiView that this objbect is the helper for
 */
MarkerTimeAxisView::MarkerTimeAxisView(MarkerTimeAxis& tv)
	: _trackview (tv)
{
	region_color = _trackview.color();
	stream_base_color = RGBA_TO_UINT (222,223,218,255);

	canvas_group = gtk_canvas_item_new (GTK_CANVAS_GROUP(_trackview.canvas_display), gtk_canvas_group_get_type (), 0);

	canvas_rect = gtk_canvas_item_new (GTK_CANVAS_GROUP(canvas_group),
		gtk_canvas_simplerect_get_type(),
		"x1", 0.0,
		"y1", 0.0,
		"x2", 1000000.0,
		"y2", (double)20,
		"outline_color_rgba", RGBA_TO_UINT(0,0,0,255),
		"fill_color_rgba", stream_base_color,
		0) ;
					   
	gtk_signal_connect(GTK_OBJECT(canvas_rect), "event", (GtkSignalFunc)PublicEditor::canvas_marker_time_axis_view_event, &_trackview) ;

	_samples_per_unit = _trackview.editor.get_current_zoom() ;

	_trackview.editor.ZoomChanged.connect (slot(*this, &MarkerTimeAxisView::reset_samples_per_unit));
}

/**
 * Destructor
 * Reposinsibly for destroying all marker items that may have been added to this time axis view
 *
 */
MarkerTimeAxisView::~MarkerTimeAxisView()
{
	// destroy everything upon this view
	for(MarkerViewList::iterator iter = marker_view_list.begin(); iter != marker_view_list.end(); ++iter)
	{
		MarkerView* mv = (*iter) ;
		
		MarkerViewList::iterator next = iter ;
		next++ ;
		marker_view_list.erase(iter) ;

		delete mv ;
		mv = 0 ;
		
		iter = next ;
	}
	
	if(canvas_rect)
	{
		gtk_object_destroy(GTK_OBJECT(canvas_rect)) ;
		canvas_rect = 0 ;
	}
	
	if(canvas_group)
	{
		gtk_object_destroy(GTK_OBJECT(canvas_group)) ;
		canvas_group = 0 ;
	}
}


//---------------------------------------------------------------------------------------//
// ui methods & data

/**
 * Sets the height of the time axis view and the item upon it
 *
 * @param height the new height
 */
int
MarkerTimeAxisView::set_height(gdouble h)
{
	if (h < 10.0 || h > 1000.0)
	{
		return -1 ;
	}
	
	gtk_object_set (GTK_OBJECT(canvas_rect), "y2", h, NULL);

	for (MarkerViewList::iterator i = marker_view_list.begin(); i != marker_view_list.end(); ++i)
	{
		(*i)->set_height(h) ;
	}

	return 0;
}

/**
 * Sets the position of this view helper on the canvas
 *
 * @param x the x position upon the canvas
 * @param y the y position upon the canvas
 */
int
MarkerTimeAxisView::set_position(gdouble x, gdouble y)
{
	gtk_canvas_item_set (canvas_group, "x", x, "y", y, NULL);
	return 0;
}

/**
 * Sets the current samples per unit.
 * this method tells each item upon the time axis of the change
 * 
 * @param spu the new samples per canvas unit value
 */
int
MarkerTimeAxisView::set_samples_per_unit(gdouble spp)
{
	if(spp < 1.0) {
		return -1 ;
	}
	
	_samples_per_unit = spp ;

	for(MarkerViewList::iterator i = marker_view_list.begin(); i != marker_view_list.end(); ++i)
	{
		(*i)->set_samples_per_unit(spp) ;
	}
	return(0) ;
}

/**
 * Sets the color of the items contained upon this view helper
 *
 * @param color the new base color
 */
void
MarkerTimeAxisView::apply_color(GdkColor& color)
{
	region_color = color;

	for (MarkerViewList::iterator i = marker_view_list.begin(); i != marker_view_list.end(); i++)
	{
		(*i)->set_color (region_color) ;
	}
}


//---------------------------------------------------------------------------------------//
// Child MarkerView Accessors/Mutators
		
/**
 * Adds a marker view to the list of items upon this time axis view helper
 * the new MarkerView is returned
 *
 * @param ifv the ImageFrameView that the new item is marking up
 * @param mark_text the text to be displayed uopn the new marker item
 * @param mark_id the unique id of the new item
 * @param start the position the new item should be placed upon the time line
 * @param duration the duration the new item should be placed upon the timeline
 * @param src the identity of the object that initiated the change
 */	 
MarkerView*
MarkerTimeAxisView::add_marker_view(ImageFrameView* ifv, std::string mark_type, std::string mark_id, jack_nframes_t start, jack_nframes_t dur, void* src)
{
	if(ifv->has_marker_view_item(mark_id))
	{
		return(0) ;
	}
	
	MarkerView* mv = new MarkerView(GTK_CANVAS_GROUP(canvas_group),
		 &_trackview,
		 ifv,
		 _trackview.editor.get_current_zoom(),
		 region_color,
		 mark_type,
		 mark_id,
		 start,
		 dur) ;
	
	ifv->add_marker_view_item(mv, src) ;
	marker_view_list.push_front(mv) ;
	
	mv->GoingAway.connect(bind (slot (*this,&MarkerTimeAxisView::remove_marker_view), (void*)this)) ;
	
	 MarkerViewAdded(mv,src) ; /* EMIT_SIGNAL */
	
	return(mv) ;
}

/**
 * Returns the named MarkerView or 0 if the named marker does not exist
 *
 * @param item_id the unique id of the item to search for
 * @return the named MarkerView, or 0 if it is not held upon this view
 */
MarkerView*
MarkerTimeAxisView::get_named_marker_view(std::string item_id)
{
	MarkerView* mv =  0 ;
	
	for(MarkerViewList::iterator i = marker_view_list.begin(); i != marker_view_list.end(); ++i)
	{
		if(((MarkerView*)*i)->get_item_name() == item_id)
		{
			mv = ((MarkerView*)*i) ;
			break ;
		}
	}
	return(mv) ;
}

/**
 * Removes the currently selected MarverView
 * Note that this method actually destroys the MarkerView too.
 * We assume that since we own the object, we are allowed to do this
 *
 * @param src the identity of the object that initiated the change
 * @see add_marker_view
 */
void
MarkerTimeAxisView::remove_selected_marker_view(void* src)
{
	std::string removed ;
	
	if (selected_time_axis_item)
	{
		MarkerViewList::iterator i ;
		if((i = find (marker_view_list.begin(), marker_view_list.end(), selected_time_axis_item)) != marker_view_list.end())
		{
			marker_view_list.erase(i) ;
			
			 MarkerViewRemoved(selected_time_axis_item->get_item_name(),src) ; /* EMIT_SIGNAL */

			delete(selected_time_axis_item) ;
			selected_time_axis_item = 0 ;
		}
	}
	else
	{
		//No selected marker view
	}
}

/**
 * Removes and returns the named MarkerView from the list of MarkerView held by this view helper
 *
 * @param item_id the MarkerView unique id to remove
 * @param src the identity of the object that initiated the change
 * @see add_marker_view
 */
MarkerView*
MarkerTimeAxisView::remove_named_marker_view(std::string item_id, void* src)
{
	MarkerView* mv = 0 ;
	
	MarkerViewList::iterator i = marker_view_list.begin() ;
	
	for(MarkerViewList::iterator iter = marker_view_list.begin(); iter != marker_view_list.end(); ++iter)
	{
		if(((MarkerView*)*i)->get_item_name() == item_id)
		{
			mv = ((MarkerView*)*i) ;
			marker_view_list.erase(i) ;
						
			 MarkerViewRemoved(mv->get_item_name(), src) ; /* EMIT_SIGNAL */
			
			// break from the for loop
			break;
		}
		i++ ;
	}
	
	return(mv) ;
}

/**
 * Removes mv from the list of MarkerView upon this TimeAxis
 *
 * @param mv the MarkerView to remove
 * @param src the identity of the object that initiated the change
 */
void
MarkerTimeAxisView::remove_marker_view(MarkerView* mv, void* src)
{
	MarkerViewList::iterator i;

	if((i = find (marker_view_list.begin(), marker_view_list.end(), mv)) != marker_view_list.end()) {
		marker_view_list.erase(i) ;
		
		// Assume this remove happened locally, else use remove_named_marker_time_axis
		// let listeners know that the named MarkerTimeAxis has been removed
		 MarkerViewRemoved(mv->get_item_name(), src) ; /* EMIT_SIGNAL */
	}
}


/**
 * Sets the duration of the selected MarkerView to the specified number of seconds
 *
 * @param sec the duration to set the MArkerView to, in seconds
 */
void
MarkerTimeAxisView::set_marker_duration_sec(double sec)
{
  if(get_selected_time_axis_item() != 0)
  {
	  get_selected_time_axis_item()->set_duration((jack_nframes_t) (sec * _trackview.editor.current_session()->frame_rate()), this) ;
  }
}


//---------------------------------------------------------------------------------------//
// Selected item methods

/**
 * Sets the currently selected item upon this time axis
 *
 * @param mv the item to set selected
 */
void
MarkerTimeAxisView::set_selected_time_axis_item(MarkerView* mv)
{
	selected_time_axis_item = mv ;
}

/**
 * Clears any selected item upon this time axis
 *
 */
void
MarkerTimeAxisView::clear_selected_time_axis_item()
{
	selected_time_axis_item = 0 ;
}
		
/**
 * Returnsthe currently selected item upon this time axis
 *
 * @return the currently selected item pon this time axis
 */
MarkerView*
MarkerTimeAxisView::get_selected_time_axis_item()
{
	return(selected_time_axis_item) ;
}




/**
 * convenience method to re-get the samples per unit and tell items upon this view
 *
 */
void
MarkerTimeAxisView::reset_samples_per_unit ()
{
	set_samples_per_unit(_trackview.editor.get_current_zoom()) ;
}
