/*
    Copyright (C) 2000-2002 Paul Davis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: mixer_strip.cc,v 1.130 2004/02/29 23:33:55 pauld Exp $
*/

#include <cmath>
#include <glib.h>

#include <sigc++/bind.h>

#include <gtkmmext/gtk_ui.h>
#include <gtkmmext/utils.h>
#include <gtkmmext/choice.h>
#include <gtkmmext/slider_controller.h>
#include <gtkmmext/stop_signal.h>

#include <ardour/ardour.h>
#include <ardour/session.h>
#include <ardour/route.h>
#include <ardour/audio_track.h>
#include <ardour/diskstream.h>
#include <ardour/send.h>
#include <ardour/insert.h>
#include <ardour/ladspa_plugin.h>
#include <ardour/connection.h>
#include <ardour/session_connection.h>

#include "ardour_ui.h"
#include "mixer_strip.h"
#include "mixer_ui.h"
#include "keyboard.h"
#include "plugin_manager.h"
#include "public_editor.h"

#include "plugin_ui.h"
#include "send_ui.h"
#include "io_selector.h"
#include "panner.h"
#include "doi.h"
#include "utils.h"

#include "i18n.h"

using namespace SigC;
using namespace ARDOUR;
using namespace Gtk;
using namespace Gtkmmext;

/* XPM */
static const gchar * small_x_xpm[] = {
"11 11 2 1",
" 	c None",
".	c #FFFFFF",
"           ",
"           ",
"  .     .  ",
"   .   .   ",
"    . .    ",
"     .     ",
"    . .    ",
"   .   .   ",
"  .     .  ",
"           ",
"           "};

/* XPM */
static const gchar * lr_xpm[] = {
"11 11 2 1",
" 	c None",
".	c #FFFFFF",
"           ",
"           ",
"   .   .   ",
"  .     .  ",
" .       . ",
"...........",
" .       . ",
"  .     .  ",
"   .   .   ",
"           ",
"           "};

static void 
speed_printer (char buf[32], Gtk::Adjustment& adj, void* arg)
{
	float val = adj.get_value ();

	if (val == 1.0) {
		strcpy (buf, "1");
	} else {
		snprintf (buf, 32, "%.3f", val);
	}
}

MixerStrip::MixerStrip (Mixer_UI& mx, Session& sess, Route& rt)
	: AxisView(sess),
	  RouteUI (rt, sess, _("mute"), _("solo"), _("RECORD")),
	  _mixer(mx),
	  pre_redirect_display (1),
	  post_redirect_display (1),
	  gpm (_route, sess),
	  button_table (9, 2),
	  automation_recording_button (""),
	  automation_mode_button (""),
	  automation_playback_button (_("aplay")),
	  polarity_button (_("polarity")),
	  speed_adjustment (1.0, 0.001, 4.0, 0.001, 0.1),
	  speed_spinner (&speed_adjustment, "MixerStripSpeedBase")

{
	if (set_color_from_route()) {
		set_color (unique_random_color());
	}

	input_selector = 0;
	output_selector = 0;
	group_menu = 0;
	pre_redirect_menu = 0;
	post_redirect_menu = 0;
	send_action_menu = 0;
	_marked_for_display = false;
	route_ops_menu = 0;
	ignore_comment_edit = false;
	redirect_drag_in_progress = false;
	ignore_toggle = false;
	ignore_speed_adjustment = false;
	
	width_button.add (*(manage (new Gtk::Pixmap (lr_xpm))));
	hide_button.add (*(manage (new Gtk::Pixmap (small_x_xpm))));

	pre_redirect_display.set_name ("MixerRedirectSelector");
	pre_redirect_display.column_titles_active ();
	pre_redirect_display.set_reorderable (true);
	pre_redirect_display.set_button_actions (0, (GTK_BUTTON_SELECTS|GTK_BUTTON_DRAGS));
	pre_redirect_display.set_button_actions (1, 0);
	pre_redirect_display.set_button_actions (2, 0);
	pre_redirect_display.set_button_actions (3, 0);
	pre_redirect_display.drag_begin.connect (slot (*this, &MixerStrip::redirect_drag_begin));
	pre_redirect_display.drag_end.connect (slot (*this, &MixerStrip::redirect_drag_end));
	pre_redirect_display.set_usize (-1, 48);
	pre_redirect_display.set_selection_mode (GTK_SELECTION_EXTENDED);
	pre_redirect_display.set_shadow_type (GTK_SHADOW_IN);
	pre_redirect_display.row_move.connect (slot (*this, &MixerStrip::redirects_reordered));

	pre_redirect_scroller.set_policy (GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

	pre_redirect_scroller.add (pre_redirect_display);
	pre_redirect_eventbox.add (pre_redirect_scroller);
	pre_redirect_hpacker.pack_start (pre_redirect_eventbox, true, true);

	post_redirect_display.set_name ("MixerRedirectSelector");
	post_redirect_display.column_titles_active ();
	post_redirect_display.set_reorderable (true);
	post_redirect_display.set_button_actions (0, (GTK_BUTTON_SELECTS|GTK_BUTTON_DRAGS));
	post_redirect_display.set_button_actions (1, 0);
	post_redirect_display.set_button_actions (2, 0);
	post_redirect_display.set_button_actions (3, 0);
	post_redirect_display.drag_begin.connect (slot (*this, &MixerStrip::redirect_drag_begin));
	post_redirect_display.drag_end.connect (slot (*this, &MixerStrip::redirect_drag_end));
	post_redirect_display.set_usize (-1, 48);
	post_redirect_display.set_selection_mode (GTK_SELECTION_EXTENDED);
	post_redirect_display.set_shadow_type (GTK_SHADOW_IN);
	post_redirect_display.row_move.connect (slot (*this, &MixerStrip::redirects_reordered));

	post_redirect_scroller.set_policy (GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	
	post_redirect_scroller.add (post_redirect_display);
	post_redirect_eventbox.add (post_redirect_scroller);
	post_redirect_hpacker.pack_start (post_redirect_eventbox, true, true);

	input_label.set_text (_("INPUT"));
	input_button.add (input_label);
	input_button.set_name ("MixerIOButton");
	input_label.set_name ("MixerIOButtonLabel");

	output_label.set_text (_("OUTPUT"));
	output_button.add (output_label);
	output_button.set_name ("MixerIOButton");
	output_label.set_name ("MixerIOButtonLabel");

	rec_enable_button.set_name ("MixerRecordEnableButton");
	rec_enable_button.unset_flags (GTK_CAN_FOCUS);

	solo_button.set_name ("SoloButton");
	mute_button.set_name ("MixerMuteButton");
	automation_recording_button.set_name ("MixerAutomationRecordingButton");
	automation_mode_button.set_name ("MixerAutomationModeButton");
	automation_playback_button.set_name ("MixerAutomationPlaybackButton");
	polarity_button.set_name ("MixerPhaseInvertButton");

	hide_button.set_events (hide_button.get_events() & ~(GDK_ENTER_NOTIFY_MASK|GDK_LEAVE_NOTIFY_MASK));

	width_button.unset_flags (GTK_CAN_FOCUS);
	hide_button.unset_flags (GTK_CAN_FOCUS);
	input_button.unset_flags (GTK_CAN_FOCUS);
	output_button.unset_flags (GTK_CAN_FOCUS);
	solo_button.unset_flags (GTK_CAN_FOCUS);
	mute_button.unset_flags (GTK_CAN_FOCUS);
	automation_recording_button.unset_flags (GTK_CAN_FOCUS);
	automation_mode_button.unset_flags (GTK_CAN_FOCUS);
	automation_playback_button.unset_flags (GTK_CAN_FOCUS);
	polarity_button.unset_flags (GTK_CAN_FOCUS);

	button_table.set_homogeneous (true);

	button_table.attach (name_button, 0, 2, 0, 1);
	button_table.attach (group_button, 0, 2, 1, 2);
	button_table.attach (input_button, 0, 2, 2, 3);

	button_table.attach (polarity_button, 0, 2, 3, 4);

	button_table.attach (solo_button, 0, 1, 4, 5);
	button_table.attach (mute_button, 1, 2, 4, 5);

	button_table.attach (automation_recording_button, 0, 1, 5, 6);
	button_table.attach (automation_mode_button, 1, 2, 5, 6);
	button_table.attach (automation_playback_button, 0, 2, 6, 7);

	using namespace Menu_Helpers;
	
	arec_menu.items().push_back (MenuElem (_("off"), 
					       bind (slot (_route, &IO::set_automation_recording_mode), AutomationList::Off, (void*) this)));
	arec_menu.items().push_back (MenuElem (_("write"),
					       bind (slot (_route, &IO::set_automation_recording_mode), AutomationList::Write, (void*) this)));
	arec_menu.items().push_back (MenuElem (_("touch"),
					       bind (slot (_route, &IO::set_automation_recording_mode), AutomationList::Touch, (void*) this)));

	amode_menu.items().push_back (MenuElem (_("trim")));
	amode_menu.items().push_back (MenuElem (_("abs")));

	ARDOUR_UI::instance()->tooltips().set_tip (automation_recording_button, _("automation record"));
	ARDOUR_UI::instance()->tooltips().set_tip (automation_mode_button, _("automation mode"));
	ARDOUR_UI::instance()->tooltips().set_tip (automation_playback_button, _("toggle automation playback"));

	if (is_audio_track()) {
		
		speed_adjustment.value_changed.connect (slot (*this, &MixerStrip::speed_adjustment_changed));
		
		speed_frame.set_name ("BaseFrame");
		speed_frame.set_shadow_type (GTK_SHADOW_IN);
		speed_frame.add (speed_spinner);
		
		speed_spinner.set_print_func (speed_printer, 0);

		ARDOUR_UI::instance()->tooltips().set_tip (speed_spinner, _("varispeed"));

		button_table.attach (speed_frame, 0, 2, 7, 8);
		button_table.attach (rec_enable_button, 0, 2, 8, 9);
	}
	
	name_button.add (name_label);
	name_button.set_name ("MixerNameButton");
	Gtkmmext::set_usize_to_display_given_text (name_button, "longest label", 2, 2);

	name_label.set_name ("MixerNameButtonLabel");
	name_label.set_text (_route.name());

	group_button.add (group_label);
	group_button.set_name ("MixerGroupButton");
	group_label.set_name ("MixerGroupButtonLabel");

	comment_area.set_name ("MixerTrackCommentArea");
	comment_area.set_editable (true);

	global_vpacker.set_border_width (4);
	global_vpacker.set_spacing (4);

	Gtk::VBox *whvbox = manage (new Gtk::VBox);

	width_button.set_name ("MixerWidthButton");
	hide_button.set_name ("MixerHideButton");

	width_button.clicked.connect (slot (*this, &MixerStrip::width_clicked));
	hide_button.clicked.connect (slot (*this, &MixerStrip::hide_clicked));

	width_hide_box.pack_start (width_button, false, true);
	width_hide_box.pack_end (hide_button, false, true);

	whvbox->pack_start (width_hide_box, true, true);

	global_vpacker.pack_start (*whvbox, false, false);
	global_vpacker.pack_start (button_table, false, false);
	global_vpacker.pack_start (pre_redirect_hpacker, true, true);
	global_vpacker.pack_start (gpm, false, false);
	global_vpacker.pack_start (post_redirect_hpacker, true, true);
	global_vpacker.pack_start (gpm.pan_box, false, false);
	global_vpacker.pack_start (output_button, false, false);
	global_vpacker.pack_start (comment_area, true, true);

	global_frame.add (global_vpacker);
	global_frame.set_shadow_type (GTK_SHADOW_IN);
	global_frame.set_name ("BaseFrame");

	add (global_frame);

	/* force setting of visible selected status */

	_selected = true;
	set_selected (false);

	whvbox->show_all ();
	pre_redirect_scroller.show ();
	pre_redirect_display.show ();
	pre_redirect_hpacker.show ();
	post_redirect_scroller.show ();
	post_redirect_display.show ();
	post_redirect_hpacker.show ();
	name_label.show ();
	group_label.show();
	input_label.show ();
	output_label.show ();
	pre_redirect_hpacker.show_all ();
	post_redirect_hpacker.show ();
	button_table.show ();
	comment_area.show ();
	name_button.show ();
	input_button.show ();
	group_button.show ();
	output_button.show ();
	rec_enable_button.show ();
	solo_button.show ();
	mute_button.show ();
	automation_recording_button.show ();
	automation_mode_button.show ();
	automation_playback_button.show ();
	polarity_button.show ();
	global_vpacker.show ();
	global_frame.show ();

	_packed = false;
	_embedded = false;

	_route.input_changed.connect (slot (*this, &MixerStrip::input_changed));
	_route.input_configuration_changed.connect (slot (*this, &MixerStrip::input_changed));
	_route.output_changed.connect (slot (*this, &MixerStrip::output_changed));
	_route.output_configuration_changed.connect (slot (*this, &MixerStrip::output_changed));
	_route.mute_changed.connect (slot (*this, &RouteUI::mute_changed));
	_route.solo_changed.connect (slot (*this, &RouteUI::solo_changed));
	_route.solo_safe_changed.connect (slot (*this, &RouteUI::solo_changed));
	_route.mix_group_changed.connect (slot (*this, &MixerStrip::mix_group_changed));
	_route.automation_recording_changed.connect (slot (*this, &MixerStrip::automation_recording_changed));
	_route.automation_playback_changed.connect (slot (*this, &MixerStrip::automation_playback_changed));

	if (is_audio_track()) {
		audio_track()->diskstream_changed.connect (slot (*this, &MixerStrip::diskstream_changed));
		get_diskstream()->speed_changed.connect (slot (*this, &MixerStrip::speed_changed));
	}

	_route.redirects_changed.connect (slot (*this, &MixerStrip::redirects_changed));
	_route.name_changed.connect (slot (*this, &RouteUI::name_changed));
	_route.comment_changed.connect (slot (*this, &MixerStrip::comment_changed));
	_route.gui_changed.connect (slot (*this, &MixerStrip::route_gui_changed));

	pre_redirect_display.button_press_event.connect (bind (slot (*this, &MixerStrip::redirect_button), Redirect::PreFader));
	pre_redirect_display.button_release_event.connect (bind (slot (*this, &MixerStrip::redirect_button), Redirect::PreFader));
	post_redirect_display.button_press_event.connect (bind (slot (*this, &MixerStrip::redirect_button), Redirect::PostFader));
	post_redirect_display.button_release_event.connect (bind (slot (*this, &MixerStrip::redirect_button), Redirect::PostFader));

	pre_redirect_display.button_release_event.connect_after (slot (do_not_propagate));
	post_redirect_display.button_release_event.connect_after (slot (do_not_propagate));

	input_button.button_release_event.connect (slot (*this, &MixerStrip::input_press));
	output_button.button_release_event.connect (slot (*this, &MixerStrip::output_press));

	rec_enable_button.button_press_event.connect (slot (*this, &RouteUI::rec_enable_press));
	solo_button.button_press_event.connect (slot (*this, &RouteUI::solo_press));
	solo_button.button_release_event.connect (slot (*this, &RouteUI::solo_release));
	mute_button.button_press_event.connect (slot (*this, &RouteUI::mute_press));
	mute_button.button_release_event.connect (slot (*this, &RouteUI::mute_release));
	automation_recording_button.button_release_event.connect (slot (*this, &MixerStrip::automation_recording_press));
	automation_mode_button.button_release_event.connect (slot (*this, &MixerStrip::automation_mode_press));
	automation_playback_button.button_press_event.connect (slot (*this, &MixerStrip::automation_playback_press));
	polarity_button.toggled.connect (slot (*this, &MixerStrip::polarity_toggled));

	name_button.button_release_event.connect (slot (*this, &MixerStrip::name_button_button_release));

	group_button.button_press_event.connect (slot (*this, &MixerStrip::select_mix_group));

	comment_area.focus_in_event.connect (slot (ARDOUR_UI::generic_focus_in_event));
	comment_area.focus_out_event.connect (slot (ARDOUR_UI::generic_focus_out_event));
	comment_area.changed.connect (slot (*this, &MixerStrip::comment_edited));
	comment_area.key_release_event.connect (slot (*this, &MixerStrip::comment_key_release_handler));
	comment_area.button_release_event.connect_after (slot (do_not_propagate));

	_mixer.plugin_manager().hide.connect(slot(*this,&MixerStrip::disconnect_newplug));

	_width = (Width) -1;
	set_stuff_from_route ();

	/* start off as a passthru strip. we'll correct this, if necessary,
	   in update_diskstream_display().
	*/

	set_name ("AudioTrackStripBase");

	/* now force an update of all the various elements */

	redirects_changed (0);
	mute_changed (0);
	solo_changed (0);
	name_changed (0);
	comment_changed (0);
	mix_group_changed (0);

	if (is_audio_track()) {
		speed_changed ();
	}

	/* XXX hack: no phase invert changed signal */

	polarity_button.set_active (_route.phase_invert());

	update_diskstream_display ();
	update_input_display ();
	update_output_display ();

	add_events (GDK_BUTTON_RELEASE_MASK);
}

MixerStrip::~MixerStrip ()
{
	 GoingAway(); /* EMIT_SIGNAL */

	if (input_selector) {
		delete input_selector;
	}

	if (output_selector) {
		delete output_selector;
	}
}

void
MixerStrip::set_stuff_from_route ()
{
	XMLProperty *prop;
	
	ensure_xml_node ();

	if ((prop = xml_node->property ("strip_width")) != 0) {
		if (prop->value() == "wide") {
			set_width (Wide);
		} else if (prop->value() == "narrow") {
			set_width (Narrow);
		}
		else {
			error << compose(_("unknown strip width \"%1\" in XML GUI information"), prop->value()) << endmsg;
			set_width (Wide);
		}
	}
	else {
		set_width (Wide);
	}

	if ((prop = xml_node->property ("shown_mixer")) != 0) {
		if (prop->value() == "no") {
			_marked_for_display = false;
		} else {
			_marked_for_display = true;
		}
	}
	else {
		/* backwards compatibility */
		_marked_for_display = true;
	}
}

void
MixerStrip::set_width (Width w)
{
	/* always set the gpm width again, things may be hidden */
	gpm.set_width (w);

	if (_width == w) {
		return;
	}

	ensure_xml_node ();
	
	_width = w;

	switch (w) {
	case Wide:
		set_usize (-1, -1);
		xml_node->add_property ("strip_width", "wide");

		static_cast<Gtk::Label*> (rec_enable_button.get_child())->set_text (_("RECORD"));
		static_cast<Gtk::Label*> (mute_button.get_child())->set_text (_("mute"));
		static_cast<Gtk::Label*> (solo_button.get_child())->set_text (_("solo"));
		static_cast<Gtk::Label*> (automation_recording_button.get_child())->set_text (arec_string());
		static_cast<Gtk::Label*> (automation_mode_button.get_child())->set_text (amode_string());
		static_cast<Gtk::Label*> (automation_playback_button.get_child())->set_text (_("aplay"));
		static_cast<Gtk::Label*> (polarity_button.get_child())->set_text (_("polarity"));
		Gtkmmext::set_usize_to_display_given_text (name_button, "long", 2, 2);
		comment_area.set_usize (15, 50);
		break;

	case Narrow:
		set_usize (50, -1);
		xml_node->add_property ("strip_width", "narrow");

		static_cast<Gtk::Label*> (rec_enable_button.get_child())->set_text (_("REC"));
		static_cast<Gtk::Label*> (mute_button.get_child())->set_text (_("m"));
		static_cast<Gtk::Label*> (solo_button.get_child())->set_text (_("s"));
		static_cast<Gtk::Label*> (automation_recording_button.get_child())->set_text (short_arec_string());
		static_cast<Gtk::Label*> (automation_mode_button.get_child())->set_text (short_amode_string());
		static_cast<Gtk::Label*> (automation_playback_button.get_child())->set_text (_("aP"));
		static_cast<Gtk::Label*> (polarity_button.get_child())->set_text (_("pol"));
		Gtkmmext::set_usize_to_display_given_text (name_button, "longest label", 2, 2);
		comment_area.set_usize (5, 50);
		break;
	}

	update_input_display ();
	update_output_display ();
	mix_group_changed (0);
	name_changed (0);
	redirects_changed (0);
}

void
MixerStrip::set_packed (bool yn)
{
	_packed = yn;

	ensure_xml_node ();

	if (_packed) {
		xml_node->add_property ("shown_mixer", "yes");
	} else {
		xml_node->add_property ("shown_mixer", "no");
	}
}

void
MixerStrip::remove_redirect_gui (Redirect *redirect)
{
	Insert *insert = 0;
	Send *send = 0;
	PortInsert *port_insert = 0;

	if ((insert = dynamic_cast<Insert *> (redirect)) != 0) {

		if ((port_insert = dynamic_cast<PortInsert *> (insert)) != 0) {
			PortInsertUI *io_selector = reinterpret_cast<PortInsertUI *> (port_insert->get_gui());
			port_insert->set_gui (0);
			delete io_selector;
		} 

	} else if ((send = dynamic_cast<Send *> (insert)) != 0) {
		SendUIWindow *sui = reinterpret_cast<SendUIWindow*> (send->get_gui());
		send->set_gui (0);
		delete sui;
	}
}

gint
MixerStrip::output_press (GdkEventButton *ev)
{
	using namespace Menu_Helpers;

	MenuList& citems = output_menu.items();
	citems.clear();

	citems.push_back (MenuElem (_("Edit"), slot (*this, &MixerStrip::edit_output_configuration)));
	citems.push_back (SeparatorElem());
	citems.push_back (MenuElem (_("Disconnect"), slot (*(static_cast<RouteUI*>(this)), &RouteUI::disconnect_output)));
	citems.push_back (SeparatorElem());

	_session.foreach_connection (this, &MixerStrip::add_connection_to_output_menu);

	output_menu.popup (1, ev->time);

	return TRUE;
}

void
MixerStrip::edit_output_configuration ()
{
	if (output_selector == 0) {
		output_selector = new IOSelectorWindow (_session, _route.output_io(), false);
	} 

	if (output_selector->is_visible()) {
		output_selector->get_toplevel()->get_window().raise();
	} else {
		output_selector->show_all ();
	}
}

void
MixerStrip::edit_input_configuration ()
{
	if (input_selector == 0) {
		input_selector = new IOSelectorWindow (_session, _route.input_io(), true);
	} 

	if (input_selector->is_visible()) {
		input_selector->get_toplevel()->get_window().raise();
	} else {
		input_selector->show_all ();
	}
}

gint
MixerStrip::input_press (GdkEventButton *ev)
{
	using namespace Menu_Helpers;

	MenuList& citems = input_menu.items();
	citems.clear();

#if ADVANCED_ROUTE_DISKSTREAM_CONNECTIVITY
	if (is_audio_track()) {
		citems.push_back (MenuElem (_("Track"), slot (*this, &MixerStrip::select_stream_input)));
	}
#endif
	citems.push_back (MenuElem (_("Edit"), slot (*this, &MixerStrip::edit_input_configuration)));
	citems.push_back (SeparatorElem());
	citems.push_back (MenuElem (_("Disconnect"), slot (*(static_cast<RouteUI*>(this)), &RouteUI::disconnect_input)));
	citems.push_back (SeparatorElem());

	_session.foreach_connection (this, &MixerStrip::add_connection_to_input_menu);

	input_menu.popup (1, ev->time);

	return TRUE;
}

void
MixerStrip::connection_input_chosen (ARDOUR::Connection *c)
{
	if (!ignore_toggle) {
		_route.input_io().use_input_connection (*c, this);
	}
}

void
MixerStrip::connection_output_chosen (ARDOUR::Connection *c)
{
	if (!ignore_toggle) {
		_route.output_io().use_output_connection (*c, this);
	}
}

void
MixerStrip::add_connection_to_input_menu (ARDOUR::Connection* c)
{
	using namespace Menu_Helpers;

	if (dynamic_cast<InputConnection *> (c) == 0) {
		return;
	}

	MenuList& citems = input_menu.items();
	citems.push_back (CheckMenuElem (c->name(), bind (slot (*this, &MixerStrip::connection_input_chosen), c)));
	
	ARDOUR::Connection *current = _route.input_io().input_connection();

	if (current == c) {
		ignore_toggle = true;
		dynamic_cast<CheckMenuItem *> (citems.back())->set_active (true);
		ignore_toggle = false;
	}
}

void
MixerStrip::add_connection_to_output_menu (ARDOUR::Connection* c)
{
	using namespace Menu_Helpers;

	if (dynamic_cast<OutputConnection *> (c) == 0) {
		return;
	}

	MenuList& citems = output_menu.items();
	citems.push_back (CheckMenuElem (c->name(), bind (slot (*this, &MixerStrip::connection_output_chosen), c)));
	
	ARDOUR::Connection *current = _route.output_connection();

	if (current == c) {
		ignore_toggle = true;
		dynamic_cast<CheckMenuItem *> (citems.back())->set_active (true);
		ignore_toggle = false;
	}
}

void
MixerStrip::select_stream_input ()
{
	using namespace Menu_Helpers;

	Menu *stream_menu = manage (new Menu);
	MenuList& items = stream_menu->items();

	Session::DiskStreamList streams = _session.disk_streams();

	for (Session::DiskStreamList::iterator i = streams.begin(); i != streams.end(); ++i) {

		if (!(*i)->hidden()) {

			items.push_back (CheckMenuElem ((*i)->name(), bind (slot (*this, &MixerStrip::stream_input_chosen), *i)));
			
			if (get_diskstream() == *i) {
				ignore_toggle = true;
				static_cast<CheckMenuItem *> (items.back())->set_active (true);
				ignore_toggle = false;
			} 
		}
	}
	
	stream_menu->popup (1, 0);
}

void
MixerStrip::stream_input_chosen (DiskStream *stream)
{
	if (is_audio_track()) {
		audio_track()->set_diskstream (*stream, this);
	}
}

void
MixerStrip::update_diskstream_display ()
{
	if (is_audio_track()) {

		rec_enable_button.set_sensitive (true);
		update_input_display ();

		if (input_selector) {
			input_selector->hide_all ();
		}

		show_route_color ();

	} else {

		rec_enable_button.set_sensitive (false);
		update_input_display ();
		show_passthru_color ();
	}
}

void
MixerStrip::update_input_display ()
{
	ARDOUR::Connection *c;

	if ((c = _route.input_io().input_connection()) != 0) {
		input_label.set_text (c->name());
	} else {
		switch (_width) {
		case Wide:
			input_label.set_text (_("INPUT"));
			break;
		case Narrow:
			input_label.set_text (_("IN"));
			break;
		}
	}

	gpm.setup_pan ();
}

void
MixerStrip::update_output_display ()
{
	ARDOUR::Connection *c;

	if ((c = _route.output_connection()) != 0) {
		output_label.set_text (c->name());
	} else {
		switch (_width) {
		case Wide:
			output_label.set_text (_("OUTPUT"));
			break;
		case Narrow:
			output_label.set_text (_("OUT"));
			break;
		}
	}

	gpm.setup_meters ();
	gpm.setup_pan ();
}

void
MixerStrip::update ()
{
	gpm.update_meters ();
}

gint
MixerStrip::automation_playback_press (GdkEventButton *ev)
{
	if (!ignore_toggle) {
		_route.set_automation_playback (!_route.automation_playback(), this);
	}

	return stop_signal (automation_playback_button, "button_press_event");
}

gint
MixerStrip::automation_recording_press (GdkEventButton *ev)
{
	arec_menu.popup (1, ev->time);
	return TRUE;
}

gint
MixerStrip::automation_mode_press (GdkEventButton *ev)
{
	amode_menu.popup (1, ev->time);
	return TRUE;
}

string
MixerStrip::arec_string ()
{
	AutomationList::Mode mode = _route.automation_mode ();

	if (mode & AutomationList::Touch) {
		return _("touch");
	} else if (mode & AutomationList::Write) {
		return _("write");
	} else {
		return _("off");
	}
}

string
MixerStrip::short_arec_string ()
{
	AutomationList::Mode mode = _route.automation_mode ();
	if (mode & AutomationList::Touch) {
		return _("T");
	} else if (mode & AutomationList::Write) {
		return _("W");
	} else {
		return _("--");
	}
}

string
MixerStrip::amode_string ()
{
	AutomationList::Style style = _route.automation_style ();

	if (style & AutomationList::Trim) {
		return _("trim");
	} else {
		return _("abs");
	}
}

string
MixerStrip::short_amode_string ()
{
	AutomationList::Style style = _route.automation_style ();

	if (style & AutomationList::Trim) {
		return _("T");
	} else {
		return _("A");
	}
}

void
MixerStrip::diskstream_changed (void *src)
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &MixerStrip::update_diskstream_display));
}	

void
MixerStrip::automation_playback_changed (void *src)

{
	bool x;

	if (automation_playback_button.get_active() != (x = _route.automation_playback())) {
		ignore_toggle = true;
		automation_playback_button.set_active (x);
		ignore_toggle = false;
	}

	set_automated_controls_sensitivity (!x);
	
	/* start watching automation so that things move */
	
	if (x) {
		auto_watching = ARDOUR_UI::RapidScreenUpdate.connect (slot (gpm, &GainPanMeter::update_display));
	} else {
		auto_watching.disconnect();
	}
}

void
MixerStrip::automation_recording_changed (void *src)
{
	switch (_width) {
	case Wide:
		static_cast<Gtk::Label*> (automation_recording_button.get_child())->set_text (arec_string());
		static_cast<Gtk::Label*> (automation_mode_button.get_child())->set_text (amode_string());
		break;
	case Narrow:
		static_cast<Gtk::Label*> (automation_recording_button.get_child())->set_text (short_arec_string());
		static_cast<Gtk::Label*> (automation_mode_button.get_child())->set_text (short_amode_string());
		break;
	}
}

void
MixerStrip::automation_mode_changed (void *src)
{
}

void
MixerStrip::input_changed (void *src)
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &MixerStrip::update_input_display));
}

void
MixerStrip::output_changed (void *src)
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &MixerStrip::update_output_display));
}

void
MixerStrip::build_send_action_menu ()

{
	using namespace Menu_Helpers;

	send_action_menu = new Menu;
	MenuList& items = send_action_menu->items();

	items.push_back (MenuElem (_("New send"), slot (*this, &MixerStrip::new_send)));
	items.push_back (MenuElem (_("Show send controls"), slot (*this, &MixerStrip::show_send_controls)));
}

void
MixerStrip::show_send_controls ()

{
}

void
MixerStrip::new_send ()

{
}

void
MixerStrip::redirect_drag_begin (GdkDragContext *context)
{
	redirect_drag_in_progress = true;
}

void
MixerStrip::redirect_drag_end (GdkDragContext *context)
{
	redirect_drag_in_progress = false;
}



gint
MixerStrip::redirect_button (GdkEventButton *ev, Redirect::Placement which)
{
	gint row, col;
	Redirect *redirect;
	CList *clist = 0;

	switch (which) {
	case Redirect::PreFader:
		clist = &pre_redirect_display;
		break;
	case Redirect::PostFader:
		clist = &post_redirect_display;
		break;
	}

	selected_redirect_placement = which;

	if (clist->get_selection_info ((int)ev->x, (int)ev->y, &row, &col) != 1) {
		redirect = 0;
	} else {
		redirect = reinterpret_cast<Redirect *> (clist->row (row).get_data ());
	}

	switch (ev->type) {
	case GDK_BUTTON_PRESS:
		return FALSE;

	case GDK_2BUTTON_PRESS:
		if (ev->state != 0) {
			return FALSE;
		}
		/* might be edit event, see below */
		break;

	case GDK_BUTTON_RELEASE:
		if (redirect_drag_in_progress) {
			// drag-n-drop reordering 
			return stop_signal (*clist, "button-release-event");
		}
		/* continue on */
		break;

	default:
		/* shouldn't be here, but gcc complains */
		return FALSE;
	}

	if (redirect && Keyboard::is_delete_event (ev)) {
		
		Gtk::Main::idle.connect (bind (slot (*this, &MixerStrip::idle_delete_redirect), redirect));
		return TRUE;

	} else if (redirect && (Keyboard::is_edit_event (ev) || ev->type == GDK_2BUTTON_PRESS)) {
		
		edit_redirect (redirect);

	} else if (Keyboard::is_context_menu_event (ev)) {
		switch (selected_redirect_placement) {
		case Redirect::PreFader:
			if (pre_redirect_menu == 0) {
				pre_redirect_menu = build_redirect_menu (*clist);
			}
			pre_redirect_menu->popup (1, 0);
			break;
		case Redirect::PostFader:
			if (post_redirect_menu == 0) {
				post_redirect_menu = build_redirect_menu (*clist);
			}
			post_redirect_menu->popup (1, 0);
			break;
		}
		return stop_signal (*clist, "button-release-event");

	} else {
		switch (ev->button) {
		case 1:
			return FALSE;
			break;

		case 2:
			if (redirect) {
				redirect->set_active (!redirect->active(), this);
			}
			break;

		case 3:
			
			break;

		default:
			return FALSE;
		}
	}

	return TRUE;
}

Menu *
MixerStrip::build_redirect_menu (CList& clist)
{
	using namespace Menu_Helpers;
	Menu * menu = new Menu;
	MenuList& items = menu->items();

	/* new stuff */
	
	items.push_back (MenuElem (_("New Plugin ..."), slot (*this, &MixerStrip::choose_plugin)));
	items.push_back (MenuElem (_("New Insert"), slot (*this, &MixerStrip::choose_insert)));
	items.push_back (MenuElem (_("New Send ..."), slot (*this, &MixerStrip::choose_send)));
	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Clear"), slot (*this, &MixerStrip::clear_redirects)));
	items.push_back (SeparatorElem());

	/* standard editing stuff */

	items.push_back (MenuElem (_("Cut"), bind (slot (*this, &MixerStrip::cut_redirects), &clist)));
	selection_dependent_items.push_back (items.back());
	items.push_back (MenuElem (_("Copy"), bind (slot (*this, &MixerStrip::copy_redirects), &clist)));
	selection_dependent_items.push_back (items.back());
	items.push_back (MenuElem (_("Paste"), slot (*this, &MixerStrip::paste_redirects)));
	redirect_paste_item = items.back();


	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Select all"), bind (slot (*this, &MixerStrip::select_all_redirects), &clist)));
#if LATER
	Menu *select_sub_menu = manage (new Menu);
	MenuList& sitems = select_sub_menu->items();

	sitems.push_back (MenuElem ("Plugins"));
	sitems.push_back (MenuElem ("Inserts"));
	sitems.push_back (MenuElem ("Sends"));
	sitems.push_back (SeparatorElem());

	items.push_back (MenuElem (_("Select all ..."), *select_sub_menu));
#endif	
	/* activation */
						     
	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Activate"), bind (slot (*this, &MixerStrip::for_selected_redirects),
							&clist, &MixerStrip::activate_redirect)));
	selection_dependent_items.push_back (items.back());
	items.push_back (MenuElem (_("Deactivate"), bind (slot (*this, &MixerStrip::for_selected_redirects),
							  &clist, &MixerStrip::deactivate_redirect)));
	selection_dependent_items.push_back (items.back());
	items.push_back (SeparatorElem());

	items.push_back (MenuElem (_("Activate All"), bind (slot (*this, &MixerStrip::all_redirects_active), true)));
	items.push_back (MenuElem (_("Deactivate All"), bind (slot (*this, &MixerStrip::all_redirects_active), false)));

	/* show editors */

	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Edit"), bind (slot (*this, &MixerStrip::for_selected_redirects),
						    &clist, &MixerStrip::edit_redirect)));
	selection_dependent_items.push_back (items.back());

	menu->map_event.connect (slot (*this, &MixerStrip::redirect_menu_map_handler));

	return menu;
}

gint
MixerStrip::redirect_menu_map_handler (GdkEventAny *ev)
{
	using namespace Menu_Helpers;
	CList* clist = 0;

	switch (selected_redirect_placement) {
	case Redirect::PreFader:
		clist = &pre_redirect_display;
		break;
	case Redirect::PostFader:
		clist = &post_redirect_display;
		break;
	}

	bool sensitive = !clist->selection().empty();

	for (vector<MenuItem*>::iterator i = selection_dependent_items.begin(); i != selection_dependent_items.end(); ++i) {
		(*i)->set_sensitive (sensitive);
	}

	redirect_paste_item->set_sensitive (!_mixer.selection().redirects.empty());
	return FALSE;
}

void
MixerStrip::select_all_redirects (CList *clist)
{
	clist->select_all ();
}

void
MixerStrip::choose_plugin ()
{
	show_plugin_selector();
}

void
MixerStrip::insert_plugin_chosen (LADSPA::Plugin *plugin)
{
	if (plugin) {
		unsigned int count;
		unsigned int rmaxports = max(_route.input_io().n_inputs(), _route.output_io().n_outputs());
		unsigned int pmaxports = max(plugin->get_info().n_inputs, plugin->get_info().n_outputs);
		
		if (rmaxports <= pmaxports) {
			/* always use at least 1 */
			count = 1;
		}
		else {
			/* only replicate plugin such that all ports may be used */
			count = rmaxports / pmaxports;
		}
		
		Redirect *redirect = new PluginInsert (_session, *plugin, selected_redirect_placement, count);
		redirect->active_changed.connect (slot (*this, &MixerStrip::show_redirect_active));
		_route.add_redirect (redirect, this);
	}
}

void
MixerStrip::choose_insert ()
{
	Redirect *redirect = new PortInsert (_session, selected_redirect_placement);
	redirect->active_changed.connect (slot (*this, &MixerStrip::show_redirect_active));
	_route.add_redirect (redirect, this);
}

void
MixerStrip::choose_send ()
{
	Send *send = new Send (_session, selected_redirect_placement);

	/* start with a basic configuration of 1 out. no inputs are needed */
	
	send->add_output_port ("", this);
	
	IOSelectorWindow *ios = new IOSelectorWindow (_session, *send, false);
	
	ios->show_all ();
	ios->selector().Finished.connect (bind (slot (*this, &MixerStrip::send_io_finished), static_cast<Redirect*>(send), ios));
}

void
MixerStrip::send_io_finished (IOSelector::Result r, Redirect* redirect, IOSelectorWindow* ios)
{
	switch (r) {
	case IOSelector::Cancelled:
		delete redirect;
		break;

	case IOSelector::Accepted:
		_route.add_redirect (redirect, this);
		break;
	}

	delete_when_idle (ios);
}

void 
MixerStrip::disconnect_newplug ()
{
    newplug_connection.disconnect();
}
void
MixerStrip::show_plugin_selector ()
{
	newplug_connection =
		_mixer.plugin_manager().PluginCreated.connect (slot (*this,&MixerStrip::insert_plugin_chosen));
	_mixer.plugin_manager().show_all ();
}

void
MixerStrip::redirects_changed (void *src)
{
	pre_redirect_display.freeze ();
	pre_redirect_display.clear ();
	post_redirect_display.freeze ();
	post_redirect_display.clear ();
	redirect_active_connections.clear ();
	_route.foreach_redirect (this, &MixerStrip::add_redirect_to_display);
	build_redirect_tooltip(pre_redirect_display, pre_redirect_eventbox, _("Pre-fader inserts, sends & plugins:"));
	build_redirect_tooltip(post_redirect_display, post_redirect_eventbox, _("Post-fader inserts, sends & plugins:"));
	pre_redirect_display.thaw ();
	post_redirect_display.thaw ();
}

void
MixerStrip::add_redirect_to_display (Redirect *redirect)
{
	const gchar *rowdata[1];
	gint row;
	CList *clist = 0;

	switch (redirect->placement()) {
	case Redirect::PreFader:
		clist = &pre_redirect_display;
		break;
	case Redirect::PostFader:
		clist = &post_redirect_display;
		break;
	}

	rowdata[0] = redirect_name (*redirect).c_str();
	clist->rows().push_back (rowdata);
	row = clist->rows().size() - 1;
	clist->row (row).set_data (redirect);

	show_redirect_active (redirect, this);

	redirect_active_connections.push_back
		(redirect->active_changed.connect (slot (*this, &MixerStrip::show_redirect_active)));
}

string
MixerStrip::redirect_name (Redirect& redirect)
{
	Send *send;
	string name_display;

	if (!redirect.active()) {
		name_display = " (";
	}

	if ((send = dynamic_cast<Send *> (&redirect)) != 0) {

		name_display += '>';

		/* grab the send name out of its overall name */

		string::size_type lbracket, rbracket;
		lbracket = send->name().find ('[');
		rbracket = send->name().find (']');

		switch (_width) {
		case Wide:
			name_display += send->name().substr (lbracket+1, lbracket-rbracket-1);
			break;
		case Narrow:
			name_display += short_version (send->name().substr (lbracket+1, lbracket-rbracket-1), 4);
			break;
		}

	} else {

		switch (_width) {
		case Wide:
			name_display += redirect.name();
			break;
		case Narrow:
			name_display += short_version (redirect.name(), 5);
			break;
		}

	}

	if (!redirect.active()) {
		name_display += ')';
	}

	return name_display;
}

void
MixerStrip::build_redirect_tooltip (CList& clist, EventBox& box, string start)
{
	CList_Helpers::RowIterator ri;
	string tip(start);

	for (ri = clist.rows().begin(); ri != clist.rows().end(); ++ri) {
		tip += '\n';
		tip += clist.cell(ri->get_row_num(), 0).get_text();
	}
	ARDOUR_UI::instance()->tooltips().set_tip (box, tip);
}

void
MixerStrip::show_redirect_active (Redirect *redirect, void *src)
{
	CList_Helpers::RowIterator ri;
	CList *clist;

	if ((ri = pre_redirect_display.rows().find_data (redirect)) == pre_redirect_display.rows().end()) {
		if ((ri = post_redirect_display.rows().find_data (redirect)) == post_redirect_display.rows().end()) {
			return;
		} else {
			clist = &post_redirect_display;
		}
	} else {
		clist = &pre_redirect_display;
	}
	
	clist->cell(ri->get_row_num(), 0).set_text (redirect_name (*redirect));

	if (redirect->active()) {
		// ri->select ();
	} else {
		// ri->unselect ();
	}
}

void
MixerStrip::set_automated_controls_sensitivity (bool yn)

{
	// XXX for various automation modes, we need the fader active

	gpm.set_sensitive (yn);
	mute_button.set_sensitive (yn);
	solo_button.set_sensitive (yn);

	/* XXX need to do this for plugin GUI's as well */
}


void
MixerStrip::comment_changed (void *src)
{
	if (src != this) {
		ignore_comment_edit = true;
		comment_area.freeze ();
		comment_area.delete_text (0, -1);
		comment_area.set_point (0);
		comment_area.insert (_route.comment());
		comment_area.thaw ();
		ignore_comment_edit = false;
	}
}

gint
MixerStrip::comment_key_release_handler (GdkEventKey *ev)
{
	switch (ev->keyval) {
	case GDK_Tab:
		ARDOUR_UI::instance()->allow_focus (false);
		break;
	default:
		break;
	}
	return TRUE;
}

void
MixerStrip::comment_edited ()
{
	if (!ignore_comment_edit) {
		_route.set_comment (comment_area.get_chars(0,-1), this);
	}
}

void
MixerStrip::set_mix_group (RouteGroup *rg)

{
	_route.set_mix_group (rg, this);
	delete group_menu;
	group_menu = 0;
}

void
MixerStrip::add_mix_group_to_menu (RouteGroup *rg)
{
	using namespace Menu_Helpers;

	MenuList& items = group_menu->items();
	items.push_back (MenuElem (rg->name(), bind (slot (*this, &MixerStrip::set_mix_group), rg)));
}

gint
MixerStrip::select_mix_group (GdkEventButton *ev)

{
	using namespace Menu_Helpers;

	group_menu = new Menu;
	MenuList& items = group_menu->items();

	items.push_back (MenuElem (_("no group"), bind (slot (*this, &MixerStrip::set_mix_group), (RouteGroup *) 0)));
	_session.foreach_mix_group (this, &MixerStrip::add_mix_group_to_menu);

	group_menu->popup (ev->button, 0);
	return stop_signal (group_button, "button_press_event");
}	

void
MixerStrip::mix_group_changed (void *ignored)
{
	RouteGroup *rg = _route.mix_group();
	
	if (rg) {
		group_label.set_text (rg->name());
	} else {
		switch (_width) {
		case Wide:
			group_label.set_text (_("no group"));
			break;
		case Narrow:
			group_label.set_text (_("~G"));
			break;
		}
	}
}


void
MixerStrip::polarity_toggled ()
{
	bool x;

	if ((x = polarity_button.get_active()) != _route.phase_invert()) {
		_route.set_phase_invert (x, this);
	}
}


void 
MixerStrip::route_gui_changed (string what_changed, void* ignored)
{
	if (what_changed == "color") {
		if (set_color_from_route () == 0) {
			show_route_color ();
		}
	}
}

void
MixerStrip::show_route_color ()
{
	Gtk::Style *style;

	input_button.ensure_style ();
	style = input_button.get_style()->copy();
	style->set_bg (GTK_STATE_NORMAL, color());
	input_button.set_style (*style);
	style->unref();

	set_name ("AudioTrackStripBase");
	gpm.set_meter_strip_name ("AudioTrackStripBase");
}

void
MixerStrip::show_passthru_color ()
{
	set_name ("AudioBusStripBase");
	gpm.set_meter_strip_name ("AudioBusStripBase");
}

void
MixerStrip::build_route_ops_menu ()
{
	using namespace Menu_Helpers;

	route_ops_menu = new Menu;

	MenuList& items = route_ops_menu->items();
	
	items.push_back (MenuElem (_("Rename"), slot (*this, &RouteUI::route_rename)));
	items.push_back (SeparatorElem());
	items.push_back (CheckMenuElem (_("Active"), slot (*this, &RouteUI::toggle_route_active)));
	route_active_menu_item = dynamic_cast<CheckMenuItem *> (items.back());
	route_active_menu_item->set_active (_route.active());
	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Remove"), slot (*this, &RouteUI::remove_this_route)));
}

gint
MixerStrip::name_button_button_release (GdkEventButton* ev)
{
	if (ev->button == 3) {
		list_route_operations ();
	}
	return FALSE;
}

void
MixerStrip::list_route_operations ()
{
	if (route_ops_menu == 0) {
		build_route_ops_menu ();
	}

	route_ops_menu->popup (1, 0);
}

void
MixerStrip::redirects_reordered (gint src, gint dst)
{
	/* this is called before the reorder has been done, so just queue
	   something for idle time.
	*/

	Gtk::Main::idle.connect (slot (*this, &MixerStrip::compute_redirect_sort_keys));
}

gint
MixerStrip::compute_redirect_sort_keys ()
{
	using Gtk::CList_Helpers;
	CList_Helpers::RowList::iterator i;
	unsigned long sort_key;

	sort_key = 0;

	for (i = pre_redirect_display.rows().begin(); i != pre_redirect_display.rows().end(); ++i) {
		Redirect *redirect = reinterpret_cast<Redirect*> (i->get_data());
		redirect->set_sort_key (sort_key, this);
		sort_key++;
	}

	for (i = post_redirect_display.rows().begin(); i != post_redirect_display.rows().end(); ++i) {
		Redirect *redirect = reinterpret_cast<Redirect*> (i->get_data());
		redirect->set_sort_key (sort_key, this);
		sort_key++;
	}

	_route.sort_redirects ();

	return FALSE;
}

void
MixerStrip::cut_redirects (CList* clist)
{
	vector<Redirect*> to_be_removed;
	
	get_selected_redirects (clist, to_be_removed);

	if (to_be_removed.empty()) {
		return;
	}

	_mixer.selection().clear_redirects ();

	for (vector<Redirect*>::iterator i = to_be_removed.begin(); i != to_be_removed.end(); ++i) {
		cut_redirect (*i);
	}
}

void
MixerStrip::copy_redirects (CList* clist)
{
	_mixer.selection().clear_redirects ();
	for_selected_redirects (clist, &MixerStrip::copy_redirect);
}

gint
MixerStrip::idle_delete_redirect (Redirect *redirect)
{
	/* NOT copied to _mixer.selection() */

	_route.remove_redirect (redirect, this);
	delete redirect;
	return FALSE;
}

void
MixerStrip::cut_redirect (Redirect *redirect)
{
	copy_redirect (redirect);
	_route.remove_redirect (redirect, this);
}

void
MixerStrip::copy_redirect (Redirect *redirect)
{
	Redirect* copy = Redirect::clone (*redirect);
	_mixer.selection().add (copy);
}

void
MixerStrip::paste_redirects ()
{
	if (_mixer.selection().redirects.empty()) {
		return;
	}

	RedirectSelection& sel (_mixer.selection().redirects);

	for (list<Redirect*>::iterator i = sel.begin(); i != sel.end(); ++i) {
		Redirect* copy = Redirect::clone (**i);
		copy->set_placement (selected_redirect_placement, this);
		_route.add_redirect (copy, this);
	}
}

void
MixerStrip::activate_redirect (Redirect *r)
{
	r->set_active (true, 0);
}

void
MixerStrip::deactivate_redirect (Redirect *r)
{
	r->set_active (false, 0);
}

void
MixerStrip::get_selected_redirects (CList* clist, vector<Redirect*>& redirects)
{
	using namespace CList_Helpers;
	SelectionList& sel (clist->selection());

	for (SelectionIterator i = sel.begin(); i != sel.end(); ++i) {
		Redirect* redirect = reinterpret_cast<Redirect *> ((*i).get_data ());
		redirects.push_back (redirect);
	}
}

void
MixerStrip::for_selected_redirects (CList* clist, void (MixerStrip::*pmf)(Redirect*))
{
	using namespace CList_Helpers;
	SelectionList& sel (clist->selection());

	for (SelectionIterator i = sel.begin(); i != sel.end(); ++i) {
		Redirect* redirect = reinterpret_cast<Redirect *> ((*i).get_data ());
		(this->*pmf)(redirect);
	}
}

void
MixerStrip::speed_adjustment_changed ()
{
	/* since there is a usable speed adjustment, there has to be a diskstream */
	if (!ignore_speed_adjustment) {
		get_diskstream()->set_speed (speed_adjustment.get_value());
	}
}

void
MixerStrip::speed_changed ()
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &MixerStrip::update_speed_display));
}

void
MixerStrip::update_speed_display ()
{
	float val;
	
	val = get_diskstream()->speed();

	if (val != 1.0) {
		speed_spinner.set_name ("MixerStripSpeedBaseNotOne");
	} else {
		speed_spinner.set_name ("MixerStripSpeedBase");
	}

	if (speed_adjustment.get_value() != val) {
		ignore_speed_adjustment = true;
		speed_adjustment.set_value (val);
		ignore_speed_adjustment = false;
	}
}			

void
MixerStrip::clone_redirects ()
{
	MixerStripSelection& strips (_mixer.selection().strips);

	if (!strips.empty()) {
		_route.copy_redirects (strips.front()->route(), selected_redirect_placement);
	}
}

void
MixerStrip::all_redirects_active (bool state)
{
	_route.all_redirects_active (state);
}

void
MixerStrip::clear_redirects()
{
	string prompt;
	vector<string> choices;

	if (is_audio_track()) {
		prompt = _("Do you really want to remove all redirects from this track?\n"
			   "(this cannot be undone)");
	} else {
		prompt = _("Do you really want to remove all redirects from this bus?\n"
			   "(this cannot be undone)");
	}

	choices.push_back (_("Yes, remove them all"));
	choices.push_back (_("Cancel"));

	Gtkmmext::Choice prompter (prompt, choices);

	prompter.chosen.connect (Gtk::Main::quit.slot());
	prompter.show_all ();

	Gtk::Main::run ();

	if (prompter.get_choice() == 0) {
		_route.clear_redirects (this);
	}
}

void
MixerStrip::set_selected (bool yn)
{
	AxisView::set_selected (yn);
	if (_selected) {
		global_frame.set_shadow_type (GTK_SHADOW_ETCHED_OUT);
		global_frame.set_name ("MixerStripSelectedFrame");
	} else {
		global_frame.set_shadow_type (GTK_SHADOW_IN);
		global_frame.set_name ("MixerStripFrame");
	}
	global_frame.queue_draw ();
}

void
MixerStrip::name_changed (void *src)
{
	switch (_width) {
	case Wide:
		RouteUI::name_changed (src);
		break;
	case Narrow:
		name_label.set_text (short_version (_route.name(), 5));
		break;
	}
}

void
MixerStrip::width_clicked ()
{
	switch (_width) {
	case Wide:
		set_width (Narrow);
		break;
	case Narrow:
		set_width (Wide);
		break;
	}
}

void
MixerStrip::hide_clicked ()
{
	if (_embedded) {
		 Hiding(); /* EMIT_SIGNAL */
	} else {
		_mixer.unselect_strip_in_display (this);
	}
}

void
MixerStrip::set_embedded (bool yn)
{
	_embedded = yn;
}

void
MixerStrip::edit_redirect (Redirect* redirect)
{
	Insert *insert;
	
	if ((insert = dynamic_cast<Insert *> (redirect)) == 0) {
		
		/* its a send */
		
		Send *send = dynamic_cast<Send*> (redirect);
		
		SendUIWindow *send_ui;
		
		if (send->get_gui() == 0) {
			
			string title;
			title = compose(_("ardour: %1"), send->name());	
			
			send_ui = new SendUIWindow (*send, _session);
			send_ui->set_title (title);
			send->set_gui (send_ui);
			
		} else {
			send_ui = reinterpret_cast<SendUIWindow *> (send->get_gui());
		}
		
		if (send_ui->is_visible()) {
			send_ui->get_window().raise ();
		} else {
			send_ui->show_all ();
		}
		
	} else {
		
		/* its an insert */
		
		PluginInsert *plugin_insert;
		PortInsert *port_insert;
		
		if ((plugin_insert = dynamic_cast<PluginInsert *> (insert)) != 0) {
			
			PluginUIWindow *plugin_ui;
			
			if (plugin_insert->get_gui() == 0) {
				
				string title;
				title = compose(_("ardour: %1: %2"), _route.name(), plugin_insert->name());	
				
				plugin_ui = new PluginUIWindow (_session.engine(), *plugin_insert);
				ARDOUR_UI::instance()->the_editor().ensure_float (*plugin_ui);
				plugin_ui->set_title (title);
				plugin_insert->set_gui (plugin_ui);
				
			} else {
				plugin_ui = reinterpret_cast<PluginUIWindow *> (plugin_insert->get_gui());
			}
			
			if (plugin_ui->is_visible()) {
				plugin_ui->get_window().raise ();
			} else {
				plugin_ui->show_all ();
			}
			
		} else if ((port_insert = dynamic_cast<PortInsert *> (insert)) != 0) {
			
			PortInsertWindow *io_selector;
			
			if (port_insert->get_gui() == 0) {
				io_selector = new PortInsertWindow (_session, *port_insert);
				port_insert->set_gui (io_selector);
				
			} else {
				io_selector = reinterpret_cast<PortInsertWindow *> (port_insert->get_gui());
			}
			
			if (io_selector->is_visible()) {
				io_selector->get_window().raise ();
			} else {
				io_selector->show_all ();
			}
		}
	}
}
