/*
    Copyright (C) 2002 Paul Davis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: panner.cc,v 1.17 2003/06/25 02:28:11 pbd Exp $
*/

#include <cmath>
#include <climits>

#include <pbd/error.h>

#include "panner.h"
#include "keyboard.h"

#include "i18n.h"

using namespace std;
using namespace Gtk;
using namespace SigC;

Panner::Target::Target (float xa, float ya)
	: x (xa), y (ya)
{
}

Panner::Panner (int w, int h)
	: width (w), height (h)
{
	allow_x = false;
	allow_y = false;
	allow_target = false;
	bypassflag = false;
	
	puck = new Target (0.5, 0.5);
	drag_target = 0;

	pair<int,Target *> newpair;
	newpair.first = 0;
	newpair.second = puck;

	targets.insert (newpair);
	puck->visible = true;

	set_events (GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK|GDK_POINTER_MOTION_MASK);
}

Panner::~Panner()
{
	for (Targets::iterator i = targets.begin(); i != targets.end(); ++i) {
		delete i->second;
	}
}

void
Panner::size_allocate_impl (GtkAllocation *alloc)
{
	width = alloc->width;
	height = alloc->height;

	DrawingArea::size_allocate_impl (alloc);
}

int
Panner::add_target (float x, float y)
{
	Target *target = new Target (x, y);

	pair<int,Target *> newpair;
	newpair.first = targets.size();
	newpair.second = target;

	targets.insert (newpair);
	target->visible = true;
	queue_draw ();

	return newpair.first;
}

void
Panner::drop_targets ()
{
	for (Targets::iterator i = targets.begin(); i != targets.end(); ) {

		Targets::iterator tmp;

		tmp = i;
		++tmp;

		if (i->second != puck) {
			delete i->second;
			targets.erase (i);
		}

		i = tmp;
	}

	queue_draw ();
}

void
Panner::remove_target (int which)
{
	Targets::iterator i = targets.find (which);

	if (i != targets.end()) {
		delete i->second;
		targets.erase (i);
		queue_draw ();
	}
}		

void
Panner::move_puck (float x, float y)
{
	move_target (0, x, y);
}

void
Panner::set_bypassed (bool byp)
{
	bypassflag = byp;
	
}

void
Panner::move_target (int which, float x, float y)
{
	Targets::iterator i = targets.find (which);
	Target *target;

	if (which != 0) { // not puck 
		if (!allow_target) {
			return;
		}
	}

	if (i != targets.end()) {
		target = i->second;
		target->x = x;
		target->y = y;
		
		queue_draw ();
	}
}		

void
Panner::show_puck ()
{
	if (!puck->visible) {
		puck->visible = true;
		queue_draw ();
	}
}

void
Panner::hide_puck ()
{
	if (puck->visible) {
		puck->visible = false;
		queue_draw ();
	}
}

void
Panner::show_target (int which)
{
	Targets::iterator i = targets.find (which);
	if (i != targets.end()) {
		if (!i->second->visible) {
			i->second->visible = true;
			queue_draw ();
		}
	}
}

void
Panner::hide_target (int which)
{
	Targets::iterator i = targets.find (which);
	if (i != targets.end()) {
		if (i->second->visible) {
			i->second->visible = false;
			queue_draw ();
		}
	}
}

Panner::Target *
Panner::find_closest_object (gdouble x, gdouble y, int& which) const
{
	gdouble efx, efy;
	Target *closest = 0;
	Target *candidate;
	float distance;
	float best_distance = FLT_MAX;

	efx = x/width;
	efy = y/height;
	which = 0;

	for (Targets::const_iterator i = targets.begin(); i != targets.end(); ++i, ++which) {
		candidate = i->second;

		distance = sqrt ((candidate->x - efx) * (candidate->x - efx) +
				 (candidate->y - efy) * (candidate->y - efy));

		if (distance < best_distance) {
			closest = candidate;
			best_distance = distance;
		}
	}
	
	return closest;
}		

gint
Panner::motion_notify_event_impl (GdkEventMotion *ev)
{
	gint x, y;
	GdkModifierType state;

	if (ev->is_hint) {
		gdk_window_get_pointer (ev->window, &x, &y, &state);
	} else {
		x = (int) floor (ev->x);
		y = (int) floor (ev->y);
		state = (GdkModifierType) ev->state;
	}
	return handle_motion (x, y, state);
}
gint
Panner::handle_motion (gint evx, gint evy, GdkModifierType state)
{
	if (drag_target == 0 || (state & GDK_BUTTON1_MASK) == 0) {
		return FALSE;
	}

	int x, y;
	bool need_move = false;

	if (drag_target != puck && !allow_target) {
		return TRUE;
	}

	if (allow_x || drag_target != puck) {
		float new_x;
		x = min (evx, width - 1);
		x = max (x, 0);
		new_x = (float) x / (width - 1);
		if (new_x != drag_target->x) {
			drag_target->x = new_x;
			need_move = true;
		}
	}

	if (allow_y || drag_target != puck) {
		float new_y;
		y = min (evy, height - 1);
		y = max (y, 0);
		new_y = (float) y / (height - 1);
		if (new_y != drag_target->y) {
			drag_target->y = new_y;
			need_move = true;
		}
	}

	if (need_move) {
		queue_draw ();

		if (drag_target == puck) {
			PuckMoved ();
		} else {
			TargetMoved (drag_index);
		}
	}

	return TRUE;
}

gint
Panner::expose_event_impl (GdkEventExpose *event)
{
	gint x, y;
	float fx, fy;

	/* redraw the background */

	get_window().draw_rectangle (get_style()->get_bg_gc(get_state()),
				     true,
				     event->area.x, event->area.y,
				     event->area.width, event->area.height);
	

	if ( ! bypassflag) {
		if (puck->visible) {
			/* redraw puck */
		
			fx = min (puck->x, 1.0f);
			fx = max (fx, -1.0f);
			x = (gint) floor (width * fx - 4);

			fy = min (puck->y, 1.0f);
			fy = max (fy, -1.0f);
			y = (gint) floor (height * fy - 4);

			get_window().draw_arc (get_style()->get_fg_gc(GTK_STATE_NORMAL),
					       true,
					       x, y,
					       8, 8,
					       0, 360 * 64);
		}

		/* redraw any visible targets */

		for (Targets::iterator i = targets.begin(); i != targets.end(); ++i) {
			Target *target = i->second;

			if (target != puck && target->visible) {

				fx = min (target->x, 1.0f);
				fx = max (fx, -1.0f);
				x = (gint) floor (width * fx - 2);
			
				fy = min (target->y, 1.0f);
				fy = max (fy, -1.0f);
				y = (gint) floor (height * fy - 2);

				get_window().draw_rectangle (get_style()->get_fg_gc(GTK_STATE_ACTIVE),
							     true,
							     x, y,
							     4, 4);
			}
		}
	}

	return TRUE;
}

gint
Panner::button_press_event_impl (GdkEventButton *ev)
{
	if (ev->button == 1) {
		gint x, y;
		GdkModifierType state;

		drag_target = find_closest_object (ev->x, ev->y, drag_index);
		
		x = (int) floor (ev->x);
		y = (int) floor (ev->y);
		state = (GdkModifierType) ev->state;

		return handle_motion (x, y, state);
	}
	
	return FALSE;
}

gint
Panner::button_release_event_impl (GdkEventButton *ev)
{
	if (ev->button == 1) {
		gint x, y;
		int ret;
		GdkModifierType state;

		x = (int) floor (ev->x);
		y = (int) floor (ev->y);
		state = (GdkModifierType) ev->state;

		if ((drag_target == puck) && (Keyboard::modifier_state_contains (state, Keyboard::Shift))) {
		        puck->x = 0.5;
		        puck->y = 0.5;
			queue_draw ();
			PuckMoved ();
		        ret = TRUE;
		}
		else {
		   ret = handle_motion (x, y, state);
		}
		
		drag_target = 0;

		return ret;

	} else if (ev->button == 2) {

		/* toggle bypassed */
		
		set_bypassed(! bypassflag);
		
		queue_draw ();
		BypassToggled();
		
		return TRUE;
	}

	return FALSE;
}

void
Panner::allow_x_motion (bool yn)
{
	allow_x = yn;
}

void
Panner::allow_target_motion (bool yn)
{
	allow_target = yn;
}

void
Panner::allow_y_motion (bool yn)
{
	allow_y = yn;
}

int
Panner::fraction (int target_a, int target_b, float& val)
{
	Target *a = 0;
	Target *b = 0;

	if (target_a == target_b) {
		return -1;
	}

	for (Targets::iterator i = targets.begin(); i != targets.end(); ++i) {
		if (i->first == target_a) {
			a = i->second;
		} else if (i->first == target_b) {
			b = i->second;
		}
	}

	if (!a || !b) {
		error << compose(_("panner: targets %1 and %2 not found!"), target_a, target_b) << endmsg;
		return -1;
	}

	float a_dist, b_dist;
	float tmpx, tmpy;

	tmpx = puck->x - a->x;
	tmpy = puck->y - a->y;
	a_dist = sqrt ((tmpx * tmpx) + (tmpy * tmpy));
	tmpx = puck->x - b->x;
	tmpy = puck->y - b->y;
	b_dist = sqrt ((tmpx * tmpx) + (tmpy * tmpy));
	
	val = a_dist / (a_dist + b_dist);
	return 0;
}

int
Panner::position (int target, float& x, float& y)
{
	if (target == 0) {
		x = puck->x;
		y = puck->y;
		return 0;
	} 
	
	Targets::iterator i;

	if ((i = targets.find (target)) != targets.end()) {
		x = i->second->x;
		y = i->second->y;
		return 0;
	}

	return -1;
}
	
