#include <cmath>

#include <gtk--.h>

#include <gtkmmext/gtk_ui.h>

#include <ardour/audioplaylist.h>
#include <ardour/audioregion.h>
#include <ardour/diskstream.h>
#include <ardour/audio_track.h>
#include <ardour/playlist_templates.h>
#include <ardour/source.h>

#include "streamview.h"
#include "regionview.h"
#include "audio_time_axis.h"
#include "canvas-waveview.h"
#include "canvas-simplerect.h"
#include "region_selection.h"
#include "public_editor.h"
#include "ardour_ui.h"
#include "crossfade_view.h"
#include "rgb_macros.h"

using namespace ARDOUR;

StreamView::StreamView (AudioTimeAxisView& tv)
	: _trackview (tv)
{
	region_color = _trackview.color();

	if (tv.is_audio_track()) {
		/* TRACK */
		//stream_base_color = RGBA_TO_UINT (222,223,218,255);
		stream_base_color = RGBA_TO_UINT (202,205,218,105);
	} else {
		/* BUS */
		//stream_base_color = RGBA_TO_UINT (230,226,238,255);
		stream_base_color = RGBA_TO_UINT (230,210,230,105);
	}

	/* set_position() will position the group */
	
	canvas_group = gtk_canvas_item_new (GTK_CANVAS_GROUP(_trackview.canvas_display),
					    gtk_canvas_group_get_type (),
					    NULL);

	canvas_rect = gtk_canvas_item_new (GTK_CANVAS_GROUP(canvas_group),
					   gtk_canvas_simplerect_get_type(),
					   "x1", 0.0,
					   "y1", 0.0,
					   "x2", 1000000.0,
					   "y2", (double) tv.height,
					   "outline_color_rgba", RGBA_TO_UINT(0,0,0,255),
					   /* outline ends and bottom */
					   "outline_what", (guint32) (0x1|0x2|0x8),
					   "fill_color_rgba", stream_base_color,
					   NULL);

	gtk_signal_connect (GTK_OBJECT(canvas_rect), "event",
			    (GtkSignalFunc) PublicEditor::canvas_stream_view_event, &_trackview);

	_samples_per_unit = _trackview.editor.get_current_zoom();
	_amplitude_above_axis = 1.0;

	if (_trackview.is_audio_track()) {
		_trackview.audio_track()->diskstream_changed.connect (slot (*this, &StreamView::diskstream_changed));
		_trackview.session().TransportStateChange.connect (slot (*this, &StreamView::transport_changed));
		_trackview.get_diskstream()->record_enable_changed.connect (slot (*this, &StreamView::rec_enable_changed));
		_trackview.session().RecordEnabled.connect (slot (*this, &StreamView::sess_rec_enable_changed));
		_trackview.session().RecordDisabled.connect (slot (*this, &StreamView::sess_rec_enable_changed));
	} 

	rec_updating = false;
}

StreamView::~StreamView ()
{
	undisplay_diskstream ();
	gtk_object_destroy (GTK_OBJECT(canvas_group));
}

void
StreamView::attach ()
{
	if (_trackview.is_audio_track()) {
		display_diskstream (_trackview.get_diskstream());
	}
}

int
StreamView::set_position (gdouble x, gdouble y)

{
	gtk_canvas_item_set (canvas_group, "x", x, "y", y, NULL);
	return 0;
}

int
StreamView::set_height (gdouble h)
{
	/* limit the values to something sane-ish */

	if (h < 10.0 || h > 1000.0) {
		return -1;
	}

	gtk_object_set (GTK_OBJECT(canvas_rect), "y2", h, NULL);

	for (AudioRegionViewList::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		(*i)->set_height (h - 2);
	}

	for (CrossfadeViewList::iterator i = crossfade_views.begin(); i != crossfade_views.end(); ++i) {
		(*i)->set_height (h - 2);
	}

	return 0;
}

int 
StreamView::set_samples_per_unit (gdouble spp)
{
	AudioRegionViewList::iterator i;

	if (spp < 1.0) {
		return -1;
	}

	_samples_per_unit = spp;

	for (i = region_views.begin(); i != region_views.end(); ++i) {
		(*i)->set_samples_per_unit (spp);
	}

	for (CrossfadeViewList::iterator xi = crossfade_views.begin(); xi != crossfade_views.end(); ++xi) {
		(*xi)->set_samples_per_unit (spp);
	}

	return 0;
}

int 
StreamView::set_amplitude_above_axis (gdouble app)

{
	AudioRegionViewList::iterator i;

	if (app < 1.0) {
		return -1;
	}

	_amplitude_above_axis = app;

	for (i = region_views.begin(); i != region_views.end(); ++i) {
		(*i)->set_amplitude_above_axis (app);
	}

	return 0;
}

void
StreamView::add_region_view (Region *r)
{
	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::add_region_view), r));
		return;
	}

	AudioRegion* region = dynamic_cast<AudioRegion*> (r);

	if (region == 0) {
		return;
	}

	AudioRegionView *region_view;
	list<AudioRegionView *>::iterator i;

	for (i = region_views.begin(); i != region_views.end(); ++i) {
		if (&(*i)->region == region) {
			
			/* great. we already have a AudioRegionView for this Region. use it again.
			 */

			(*i)->set_valid (true);
			return;
		}
	}

	region_view = new AudioRegionView (GTK_CANVAS_GROUP(canvas_group),
					   _trackview,
					   *region,
					   _samples_per_unit,
					   _amplitude_above_axis, 
					   region_color);

	region_views.push_front (region_view);

	/* catch regionview going away */

	region->GoingAway.connect (slot (*this, &StreamView::remove_region_view));
	
	AudioRegionViewAdded (region_view);
}

void
StreamView::remove_region_view (Region *r)
{
	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::remove_region_view), r));
		return;
	}

	AudioRegion* region = dynamic_cast<AudioRegion*> (r);

	if (region == 0) {
		return;
	}

	for (list<AudioRegionView *>::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		if (&((*i)->region) == region) {
			delete *i;
			region_views.erase (i);
			break;
		}
	}

	for (list<CrossfadeView *>::iterator i = crossfade_views.begin(); i != crossfade_views.end();) {
		list<CrossfadeView*>::iterator tmp;
		
		tmp = i;
		++tmp;
		
		if ((*i)->crossfade.involves (*region)) {
			delete *i;
			crossfade_views.erase (i);
		}
		
		i = tmp;
	}
}

void
StreamView::undisplay_diskstream ()
{
	for (AudioRegionViewList::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		delete *i;
	}

	for (CrossfadeViewList::iterator i = crossfade_views.begin(); i != crossfade_views.end(); ++i) {
		delete *i;
	}

	region_views.clear();
	crossfade_views.clear ();
}

void
StreamView::display_diskstream (DiskStream *ds)
{
	playlist_change_connection.disconnect();
	playlist_changed (ds);
	playlist_change_connection = ds->PlaylistChanged.connect (bind (slot (*this, &StreamView::playlist_changed), ds));
}

void
StreamView::playlist_modified ()
{
}

void
StreamView::playlist_changed (DiskStream *ds)
{
	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::playlist_changed), ds));
		return;
	}

	/* disconnect from old playlist */

	for (vector<SigC::Connection>::iterator i = playlist_connections.begin(); i != playlist_connections.end(); ++i) {
		(*i).disconnect();
	}
	
	playlist_connections.clear();
	undisplay_diskstream ();

	/* draw it */

	ds->playlist()->foreach_region (this, &StreamView::add_region_view);
	ds->playlist()->foreach_crossfade (this, &StreamView::add_crossfade);

	/* catch changes */

	playlist_connections.push_back (ds->playlist()->RegionAdded.connect (slot (*this, &StreamView::add_region_view)));
	playlist_connections.push_back (ds->playlist()->RegionRemoved.connect (slot (*this, &StreamView::remove_region_view)));
	playlist_connections.push_back (ds->playlist()->StateChanged.connect (slot (*this, &StreamView::playlist_state_changed)));
	playlist_connections.push_back (ds->playlist()->Modified.connect (slot (*this, &StreamView::playlist_modified)));
	playlist_connections.push_back (ds->playlist()->NewCrossfade.connect (slot (*this, &StreamView::add_crossfade)));
}

void
StreamView::add_crossfade (Crossfade *crossfade)
{
	AudioRegionView* lview = 0;
	AudioRegionView* rview = 0;

	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::add_crossfade), crossfade));
		return;
	}

	/* first see if we already have a CrossfadeView for this Crossfade */

	for (list<CrossfadeView *>::iterator i = crossfade_views.begin(); i != crossfade_views.end(); ++i) {
		if (&(*i)->crossfade == crossfade) {
			(*i)->set_valid (true);
			return;
		}
	}

	/* create a new one */

	for (list<AudioRegionView *>::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		if (!lview && &((*i)->region) == &crossfade->left()) {
			lview = *i;
		}
		if (!rview && &((*i)->region) == &crossfade->right()) {
			rview = *i;
		}
	}

	CrossfadeView *cv = new CrossfadeView (GTK_CANVAS_GROUP(_trackview.canvas_display),
					       _trackview,
					       *crossfade,
					       _samples_per_unit,
					       region_color,
					       *lview, *rview);

	crossfade->Invalidated.connect (slot (*this, &StreamView::remove_crossfade));
	crossfade_views.push_back (cv);
}

void
StreamView::remove_crossfade (Crossfade *xfade)
{
	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::remove_crossfade), xfade));
		return;
	}

	for (list<CrossfadeView*>::iterator i = crossfade_views.begin(); i != crossfade_views.end(); ++i) {
		if (&(*i)->crossfade == xfade) {
			delete *i;
			crossfade_views.erase (i);
			break;
		}
	}
}

void
StreamView::playlist_state_changed (Change ignored)
{
	if (!Gtkmmext::UI::instance()->caller_is_gui_thread()) {
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::playlist_state_changed), ignored));
		return;
	}

	redisplay_diskstream ();
}

void
StreamView::redisplay_diskstream ()
{
	list<AudioRegionView *>::iterator i, tmp;
	list<CrossfadeView*>::iterator xi, tmpx;

	for (i = region_views.begin(); i != region_views.end(); ++i) {
		(*i)->set_valid (false);
	}

	for (xi = crossfade_views.begin(); xi != crossfade_views.end(); ++xi) {
		(*xi)->set_valid (false);

	}

	if (_trackview.is_audio_track()) {
		_trackview.get_diskstream()->playlist()->foreach_region (this, &StreamView::add_region_view);
		_trackview.get_diskstream()->playlist()->foreach_crossfade (this, &StreamView::add_crossfade);
	}

	for (i = region_views.begin(); i != region_views.end(); ) {
		tmp = i;
		tmp++;

		if (!(*i)->is_valid()) {
			delete *i;
			region_views.erase (i);
		}

		i = tmp;
	}

	for (xi = crossfade_views.begin(); xi != crossfade_views.end();) {
		tmpx = xi;
		tmpx++;

		if (!(*xi)->valid()) {
			delete *xi;
			crossfade_views.erase (xi);
		}

		xi = tmpx;
	}
}

void
StreamView::diskstream_changed (void *src_ignored)
{
	AudioTrack *at;

	if ((at = _trackview.audio_track()) != 0) {
		DiskStream& ds = at->disk_stream();
		/* XXX grrr: when will SigC++ allow me to bind references? */
		Gtkmmext::UI::instance()->call_slot (bind (slot (*this, &StreamView::display_diskstream), &ds));
	} else {
		Gtkmmext::UI::instance()->call_slot (slot (*this, &StreamView::undisplay_diskstream));
	}
}

void
StreamView::apply_color (GdkColor& color, ColorTarget target)

{
	list<AudioRegionView *>::iterator i;

	switch (target) {
	case RegionColor:
		region_color = color;
		for (i = region_views.begin(); i != region_views.end(); ++i) {
			(*i)->set_color (region_color);
		}
		// stream_base_color = RGBA_TO_UINT (color.red/256, color.green/256, color.blue/256, 255);
		// gtk_canvas_item_set (canvas_rect, "fill_color_rgba", stream_base_color, NULL);
		break;
		
	case StreamBaseColor:
		// stream_base_color = RGBA_TO_UINT (color.red/256, color.green/256, color.blue/256, 255);
		// gtk_canvas_item_set (canvas_rect, "fill_color_rgba", stream_base_color, NULL);
		break;
	}
}

void
StreamView::set_show_waveforms (bool yn)
{
	for (list<AudioRegionView *>::iterator i = region_views.begin(); i != region_views.end(); ++i) {
			(*i)->set_waveform_visible (yn);
	}
}

void
StreamView::set_selected_regionviews (AudioRegionSelection& regions)
{
	bool selected;

	for (list<AudioRegionView*>::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		
		selected = false;
		
		for (AudioRegionSelection::iterator ii = regions.begin(); ii != regions.end(); ++ii) {
			if (*i == *ii) {
				selected = true;
			}
		}
		
		(*i)->set_selected (selected, this);
	}
}

void
StreamView::get_selectables (jack_nframes_t start, jack_nframes_t end, list<Selectable*>& results)
{
	for (list<AudioRegionView*>::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		if ((*i)->region.coverage(start, end) != OverlapNone) {
			results.push_back (*i);
		}
	}
}

void
StreamView::set_waveform_shape (WaveformShape shape)
{
	for (AudioRegionViewList::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		(*i)->set_waveform_shape (shape);
	}
}		
		
void
StreamView::region_layered (AudioRegionView* rv)
{
	gtk_canvas_item_lower_to_bottom (rv->get_canvas_group());

	/* don't ever leave it at the bottom, since then it doesn't
	   get events - the parent group does instead ...
	*/

	gtk_canvas_item_raise (rv->get_canvas_group(), rv->region.layer() + 1);
}

void
StreamView::rec_enable_changed (void *src)
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &StreamView::setup_rec_box));
}

void
StreamView::sess_rec_enable_changed ()
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &StreamView::setup_rec_box));
}

void
StreamView::transport_changed()
{
	Gtkmmext::UI::instance()->call_slot (slot (*this, &StreamView::setup_rec_box));
}

void
StreamView::setup_rec_box ()
{
	if (_trackview.session().transport_rolling())
	{
		if (!rec_updating
		    && _trackview.session().record_status() == Session::Recording
		    && _trackview.get_diskstream()->record_enabled())
		{
			/* start a new rec box */
			gdouble xstart = _trackview.editor.frame_to_pixel (_trackview.session().transport_frame());
			gdouble xend = xstart;
			
			GtkCanvasItem * rec_rect = gtk_canvas_item_new (GTK_CANVAS_GROUP(canvas_group),
									gtk_canvas_simplerect_get_type(),
									"x1", xstart,
									"y1", 0.0,
									"x2", xend,
									"y2", (double) _trackview.height - 2,
									"outline_color_rgba", RGBA_TO_UINT(80,40,40,255),
									"fill_color_rgba",  RGBA_TO_UINT(230,200,200,255),
									NULL);
			
			rec_rects.push_back (rec_rect);
			
			screen_update_connection = ARDOUR_UI::instance()->SuperRapidScreenUpdate.connect (slot (*this, &StreamView::update_rec_box));	
			rec_updating = true;
		}
		else if (rec_updating
			 && (_trackview.session().record_status() != Session::Recording
			     || !_trackview.get_diskstream()->record_enabled()))
		{
			/* disconnect rapid update */
			screen_update_connection.disconnect();
			rec_updating = false;
		}
		
	}
	else {
		if (rec_rects.size() > 0) {
			/* disconnect rapid update */
			screen_update_connection.disconnect();
			rec_updating = false;

			/* transport stopped, clear boxes */
			for (vector<GtkCanvasItem*>::iterator iter=rec_rects.begin(); iter != rec_rects.end(); ++iter)
			{
				GtkCanvasItem * rect = (*iter);
				gtk_object_destroy (GTK_OBJECT(rect));
			}

			rec_rects.clear();
		}
	}
}


void
StreamView::update_rec_box ()
{
	/* only update the last box */
	if (rec_rects.size() > 0) {
		GtkCanvasItem * rect = rec_rects.back();
		gdouble xend = _trackview.editor.frame_to_pixel (_trackview.session().transport_frame());
		gtk_canvas_item_set (rect, "x2", xend, NULL);
	}
}

AudioRegionView*
StreamView::find_view (const AudioRegion& region)
{
	for (list<AudioRegionView*>::iterator i = region_views.begin(); i != region_views.end(); ++i) {

		if (&(*i)->region == &region) {
			return *i;
		}
	}
	return 0;
}
	
void
StreamView::foreach_regionview (SigC::Slot1<void,AudioRegionView*> slot)
{
	for (list<AudioRegionView*>::iterator i = region_views.begin(); i != region_views.end(); ++i) {
		slot (*i);
	}
}
