/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: ladspa_plugin.h,v 1.21 2004/02/16 04:26:19 pauld Exp $
*/

#ifndef __ardour_ladspa_h__
#define __ardour_ladspa_h__

#include <list>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <dlfcn.h>

#include <midi++/controllable.h>
#include <sigc++/signal_system.h>

#include <jack/types.h>
#include <ardour/ladspa.h>
#include <ardour/stateful.h>
#include <ardour/plugin_state.h>

using std::string;
using std::vector;
using std::list;
using std::map;

namespace ARDOUR {
	class AudioEngine;
	class Session;
}

namespace LADSPA {

class Info {
  public:
	Info () { };
	Info (const Info &o)
		: name(o.name), uri(o.uri), n_inputs(o.n_inputs), n_outputs(o.n_outputs),
		path (o.path), index(o.index) {}
	~Info () { };
	string name;
	string uri;
	int n_inputs;
	int n_outputs;
	
  private:
	friend class Manager;
	string path;
	unsigned long index;
};

class Plugin : public Stateful, public SigC::Object

{
  public:
	static const string state_node_name;

	Plugin (void *module, ARDOUR::AudioEngine&, unsigned long index, jack_nframes_t sample_rate);
	Plugin (const Plugin &);
	~Plugin ();
	
	unsigned long index() const                      { return _index; }
	unsigned long unique_id() const                  { return descriptor->UniqueID; }
	const char * label() const                       { return descriptor->Label; }
	LADSPA_Properties properties() const             { return descriptor->Properties; }
	const char * name() const                        { return descriptor->Name; }
	const char * maker() const                       { return descriptor->Maker; }
	const char * copyright() const                   { return descriptor->Copyright; }
	unsigned long port_count() const                 { return descriptor->PortCount; }
	const LADSPA_PortDescriptor * port_descriptors() const { return descriptor->PortDescriptors; }
	const char * const * port_names() const          { return descriptor->PortNames; }
	const LADSPA_PortRangeHint * port_range_hints() const { return descriptor->PortRangeHints; }
	float default_value (unsigned long port);

	jack_nframes_t latency() const;

	void set_control_port (unsigned long port, LADSPA_Data val);
	LADSPA_Data get_control_port(unsigned long port) const;

	MIDI::Controllable *get_nth_midi_control (unsigned long);

	unsigned long nth_control_port(unsigned long port, bool& ok) const;

	void activate () { 
		if (descriptor->activate) 
			descriptor->activate (handle);
	}
	void deactivate () {
		if (descriptor->deactivate) 
			descriptor->deactivate (handle);
	}
	void set_gain (LADSPA_Data gain) {
		descriptor->set_run_adding_gain (handle, gain);
	}
	void run (unsigned long nsamples) {
		descriptor->run (handle, nsamples);
	}
	void run_adding (unsigned long nsamples) {
		descriptor->run_adding (handle, nsamples);
	}
	void connect_port (unsigned long port, LADSPA_Data *ptr) {
		descriptor->connect_port (handle, port, ptr);
	}

	SigC::Signal2<void,unsigned long,LADSPA_Data> ControlPortChanged;
	SigC::Signal1<void,Plugin *> GoingAway;

	XMLNode& get_state(void);
	int set_state(const XMLNode& node);
	int set_state(string entry);

	ARDOUR::AudioEngine& engine() const { return _engine; }

	list<string> get_ladspa_settings(void);
	bool save_preset(string name);
	
	string describe_parameter (unsigned long);

	void reset_midi_control (MIDI::Port*, bool);

	Info & get_info() { return _info; }
	void set_info (const Info &inf) { _info = inf; }

	std::set<unsigned long> automatable() const;

	void store_state (ARDOUR::PluginState&);
	void restore_state (ARDOUR::PluginState&);

  private:
	void                    *module;
	ARDOUR::AudioEngine&    _engine;
	const LADSPA_Descriptor *descriptor;
	LADSPA_Handle            handle;
	jack_nframes_t           sample_rate;
	LADSPA_Data             *control_data;
	LADSPA_Data             *latency_control_port;
	unsigned long           _index;
	
	struct MIDIPortControl : public MIDI::Controllable {
	    MIDIPortControl (Plugin&, unsigned long abs_port_id, MIDI::Port *);

	    void set_value (float);

	    Plugin& plugin;
	    unsigned long absolute_port;
	    LADSPA_Data upper;
	    LADSPA_Data lower;
	    LADSPA_Data range;
	    bool  toggled;
	    bool  logarithmic;
	};
	
	void init (void *mod, unsigned long index, jack_nframes_t rate);
	vector<MIDIPortControl*> midi_controls;

	Info _info;
};
 
class Manager {
  public:
	Manager (ARDOUR::AudioEngine&);
	~Manager ();

	list<Info *> &plugin_info () { return _plugin_info; }
	list<Plugin *> &plugins()    { return _plugins; }

	void refresh ();
	int add_directory (string dirpath);

	Plugin *load (ARDOUR::Session& s, Info *info);

  private:
	ARDOUR::AudioEngine&   _engine;
	list<Info *>   _plugin_info;
	list<Plugin *> _plugins;
	map<unsigned long, string> rdf_type;

	string search_path;

	void add_lrdf_data (const string &path);
	void descend_lrdf_classes (string uri, string base, string label);

	int discover_from_path (string path);
	int discover (string path);
};

extern Manager *manager;

Plugin * find_plugin(ARDOUR::Session&, string name);

} /* namespace LADSPA */

#endif /* __ardour_ladspa_h__ */
