/*
    Copyright (C) 2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: automation_event.cc,v 1.15 2004/03/01 02:39:17 pauld Exp $
*/

#include <set>
#include <limits.h>

#include <algorithm>
#include <sigc++/bind.h>
#include <ardour/automation_event.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

#if 1
static void dumpit (AutomationList& al, string prefix = "")
{
	cerr << prefix << &al << endl;
	for (AutomationList::iterator i = al.begin(); i != al.end(); ++i) {
		cerr << prefix << '\t' << (*i)->when << ',' << (*i)->value << endl;
	}
	cerr << "\n";
}
#endif

AutomationList::AutomationList (double defval, bool with_state)
{
	_frozen = false;
	changed_when_thawed = false;
	_mode = Off;
	_style = Absolute;
	_touching = false;
	no_state = with_state;

	max_x = 0;
	min_x = max_frames;
	min_yval = FLT_MIN;
	max_yval = FLT_MAX;
	max_xval = 0; // means "no limit" 
	default_value = defval;
	_dirty = false;
	rt_insertion_point = events.end();
	rt_pos = 0;
	
	if (!no_state) {
		save_state (_("initial"));
	}
}

AutomationList::AutomationList (const AutomationList& other)
{
	min_x = other.min_x;
	max_x = other.max_x;
	min_yval = other.min_yval;
	max_yval = other.max_yval;
	max_xval = other.max_xval;
	default_value = other.default_value;
	_mode = other._mode;
	_touching = other._touching;
	rt_insertion_point = events.end();
	rt_pos = 0;
	no_state = other.no_state;

	for (const_iterator i = other.events.begin(); i != other.events.end(); ++i) {
		/* we have to use other point_factory() because
		   its virtual and we're in a constructor.
		*/
		events.push_back (other.point_factory (**i));
	}

	mark_dirty ();
}

AutomationList::~AutomationList()
{
	std::set<ControlEvent*> all_events;
	AutomationList::State* asp;

	for (StateMap::iterator i = states.begin(); i != states.end(); ++i) {

		if ((asp = dynamic_cast<AutomationList::State*> (*i)) != 0) {
			
			for (AutomationEventList::iterator x = asp->events.begin(); x != asp->events.end(); ++x) {
				all_events.insert (*x);
			}
		}
	}

	for (std::set<ControlEvent*>::iterator i = all_events.begin(); i != all_events.end(); ++i) {
		delete (*i);
	}
}

AutomationList&
AutomationList::operator= (const AutomationList& other)
{
	{
		if (this != &other) {

			events.clear ();

			for (const_iterator i = other.events.begin(); i != other.events.end(); ++i) {
				events.push_back (point_factory (**i));
			}

			mark_dirty ();
			
			min_x = other.min_x;
			max_x = other.max_x;
			min_yval = other.min_yval;
			max_yval = other.max_yval;
			max_xval = other.max_xval;
			default_value = other.default_value;
		}
	}

	maybe_signal_changed ();

	return *this;
}

void
AutomationList::maybe_signal_changed ()
{
	if (_frozen) {
		changed_when_thawed = true;
	} else {
		StateChanged (Change (0));
	}
}

void
AutomationList::set_mode (Mode m)
{
	_mode = m;
}

void
AutomationList::set_style (Style m)
{
	_style = m;
}

void
AutomationList::start_touch ()
{
	_touching = true;
}

void
AutomationList::stop_touch ()
{
	_touching = false;
}

void
AutomationList::clear ()
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		events.clear ();
		if (!no_state) {
			save_state (_("cleared"));
		}
	}

	maybe_signal_changed ();
}

void
AutomationList::reposition_for_rt_add (double when)
{
	rt_insertion_point = events.end();
}

#define last_rt_insertion_point rt_insertion_point

void
AutomationList::rt_add (double when, double value)
{
	/* this is for automation recording */

	if ((_mode & Touch) && !_touching) {
		return;
	}

	{
		LockMonitor lm (lock, __LINE__, __FILE__);

		iterator where;
		TimeComparator cmp;
		ControlEvent cp (when, 0.0);
		bool done = false;

		if (last_rt_insertion_point != events.end()) {

			/* we have a previous insertion point, so we should delete
			   everything between it and the position where we are going
			   to insert this point.
			*/

			iterator after = last_rt_insertion_point;

			if (++after != events.end()) {
				iterator far = after;
				
				while (far != events.end()) {
					if ((*far)->when > when) {
						break;
					}
					++far;
				}

				where = events.erase (after, far);
			}

			if (last_rt_insertion_point != events.begin() && (*last_rt_insertion_point)->value == value) {
				(*last_rt_insertion_point)->when = when;
				done = true;
				
			}
			
		} else {

			where = lower_bound (events.begin(), events.end(), &cp, cmp);

		}
		
		if (!done) {
			last_rt_insertion_point = events.insert (where, point_factory (when, value));
		}

		max_x = max (when, max_x);
		min_x = min (when, min_x);

		mark_dirty ();
	}

	maybe_signal_changed ();
}

#undef last_rt_insertion_point

void
AutomationList::add (double when, double value)
{
	/* this is for graphical editing */

	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		TimeComparator cmp;
		ControlEvent cp (when, 0.0f);
		bool insert = true;
		iterator insertion_point;

		for (insertion_point = lower_bound (events.begin(), events.end(), &cp, cmp); insertion_point != events.end(); ++insertion_point) {

			/* only one point allowed per time point */

			if ((*insertion_point)->when == when) {
				(*insertion_point)->value = value;
				insert = false;
				break;
			} 

			if ((*insertion_point)->when >= when) {
				break;
			}
		}

		if (insert) {

			events.insert (insertion_point, point_factory (when, value));

			/* inserted ev *before* insertion_point, so get an iterator
			   to point to it.
			*/

			--insertion_point;

			reposition_for_rt_add (rt_pos);

		} 

		max_x = max (when, max_x);
		min_x = min (when, min_x);

		mark_dirty ();

		if (!no_state) {
			save_state (_("added event"));
		}
	}

	maybe_signal_changed ();
}

void
AutomationList::erase (AutomationList::iterator i)
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		events.erase (i);
		reposition_for_rt_add (rt_pos);
		if (!no_state) {
			save_state (_("removed event"));
		}
	}
	maybe_signal_changed ();
}

void
AutomationList::erase (AutomationList::iterator start, AutomationList::iterator end)
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		events.erase (start, end);
		reposition_for_rt_add (rt_pos);
		if (!no_state) {
			save_state (_("removed multiple events"));
		}
	}
	maybe_signal_changed ();
}	

void
AutomationList::erase_range (double start, double endt)
{
	bool erased = false;

	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		TimeComparator cmp;
		ControlEvent cp (start, 0.0f);
		iterator s;
		iterator e;

		if ((s = lower_bound (events.begin(), events.end(), &cp, cmp)) != events.end()) {
			cp.when = endt;
			e = upper_bound (events.begin(), events.end(), &cp, cmp);
			events.erase (s, e);
			reposition_for_rt_add (rt_pos);
			erased = true;
			if (!no_state) {
				save_state (_("removed range"));
			}
		}
	}

	if (erased) {
		maybe_signal_changed ();
	}
}

void
AutomationList::move_range (iterator start, iterator end, double xdelta, double ydelta)
{
	/* note: we assume higher level logic is in place to avoid this
	   reordering the time-order of control events in the list. ie. all
	   points after end are later than (end)->when.
	*/

	{
		LockMonitor lm (lock, __LINE__, __FILE__);

		while (start != end) {
			(*start)->when += xdelta;
			(*start)->value += ydelta;
			++start;
		}

		if (!no_state) {
			save_state (_("event range adjusted"));
		}
	}

	maybe_signal_changed ();
}

void
AutomationList::modify (iterator iter, double when, double val)
{
	/* note: we assume higher level logic is in place to avoid this
	   reordering the time-order of control events in the list. ie. all
	   points after *iter are later than when.
	*/

	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		(*iter)->when = when;
		(*iter)->value = val;
		if (!no_state) {
			save_state (_("event adjusted"));
		}
	}
	
	maybe_signal_changed ();
}

std::pair<AutomationList::iterator,AutomationList::iterator>
AutomationList::control_points_adjacent (double xval)
{
	LockMonitor lm (lock, __LINE__, __FILE__);
	iterator i;
	TimeComparator cmp;
	ControlEvent cp (xval, 0.0f);
	std::pair<iterator,iterator> ret;

	ret.first = events.end();
	ret.second = events.end();

	for (i = lower_bound (events.begin(), events.end(), &cp, cmp); i != events.end(); ++i) {
		
		if (ret.first == events.end()) {
			if ((*i)->when >= xval) {
				if (i != events.begin()) {
					ret.first = i;
					--ret.first;
				} else {
					return ret;
				}
			}
		} 
		
		if ((*i)->when > xval) {
			ret.second = i;
			break;
		}
	}

	return ret;
}

void
AutomationList::freeze ()
{
	_frozen = true;
}

void
AutomationList::thaw ()
{
	_frozen = false;
	if (changed_when_thawed) {
		 StateChanged(Change(0)); /* EMIT SIGNAL */
	}
}

StateManager::State*
AutomationList::state_factory (std::string why) const
{
	State* state = new State (why);

	for (AutomationEventList::const_iterator x = events.begin(); x != events.end(); ++x) {
		state->events.push_back (point_factory (**x));
	}

	return state;
}

Change
AutomationList::restore_state (StateManager::State& state) 
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		State* lstate = dynamic_cast<State*> (&state);
		events = lstate->events;
	}

	return Change (0);
}

UndoAction
AutomationList::get_memento () const
{
	return bind (slot (*(const_cast<AutomationList*> (this)), &StateManager::use_state), _current_state_id);
}

void
AutomationList::set_max_xval (double x)
{
	max_xval = x;
}

void 
AutomationList::mark_dirty ()
{
	last_lo = DBL_MAX;
	last_hi = 0;
	_dirty = true;
}

void
AutomationList::truncate_end (double last_coordinate)
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		ControlEvent cp (last_coordinate, 0);
		list<ControlEvent*>::reverse_iterator i;
		double last_val;

		last_val = _eval (last_coordinate);

		if (events.empty()) {
			fatal << _("programming error:")
			      << "AutomationList::truncate_end() called on an empty list"
			      << endmsg;
			/*NOTREACHED*/
			return;
		}

		i = events.rend();
		--i;

		while (i != events.rbegin() && events.size() > 2) {
			list<ControlEvent*>::reverse_iterator tmp;
			
			tmp = i;
			++tmp;

			if ((*i)->when <= last_coordinate) {
				break;
			}

			events.erase (i.base());
			
			i = tmp;
		}

		last_val = max ((double) min_yval, last_val);
		last_val = min ((double) max_yval, last_val);
		
		events.back()->when = last_coordinate;
		events.back()->value = last_val;

		reposition_for_rt_add (rt_pos);

		max_x = last_coordinate;
		
		mark_dirty();
	}

	maybe_signal_changed ();
}

void
AutomationList::truncate_start (double overall_length)
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		AutomationList::iterator i;
		double first_legal_value;
		double first_legal_coordinate;
		
		first_legal_coordinate = events.back()->when - overall_length;
		
		if (first_legal_coordinate == 0) {
			/* no change in overall length */
			return;
		}
		
		if (events.empty()) {
			fatal << _("programming error:")
			      << "AutomationList::truncate_start() called on an empty list"
			      << endmsg;
			/*NOTREACHED*/
			return;
		}

		i = events.begin();

		while (i != events.end() && events.size() > 2) {
			list<ControlEvent*>::iterator tmp;
			
			tmp = i;
			++tmp;

			if ((*i)->when >= first_legal_coordinate) {
				break;
			}

			events.erase (i);
			
			i = tmp;
		}
		
		for (i = events.begin(); i != events.end(); ++i) {
			(*i)->when -= first_legal_coordinate;
		}
		
		first_legal_value = max (min_yval, first_legal_value);
		first_legal_value = min (max_yval, first_legal_value);
		
		events.front()->when = 0;
		events.front()->value = first_legal_value;
		
		reposition_for_rt_add (rt_pos);

		min_x = 0;

		mark_dirty();
	}

	maybe_signal_changed ();
}

double
AutomationList::_eval (double x) 
{
	pair<AutomationList::iterator,AutomationList::iterator> range;
	ControlEvent cp (x, 0);
	TimeComparator cmp;
	double upos, lpos;
	double uval, lval;
	double fraction;

	if (events.empty()) {
		return default_value;
	}

	/* XXX OPTIMIZE ME by caching the range iterators */

	range = equal_range (events.begin(), events.end(), &cp, cmp);

	if (range.first == range.second) {

		/* x does not exist within the list as a control point */

		if (range.first != events.begin()) {
			--range.first;
			lpos = (*range.first)->when;
			lval = (*range.first)->value;
		}  else {
			lpos = 0;
			lval = default_value;
		}
		
		if (range.second == events.end()) {
			return default_value;
		}

		upos = (*range.second)->when;
		uval = (*range.second)->value;

		fraction = (double) (x - lpos) / (double) (upos - lpos);
		return lval + (fraction * (uval - lval));

	} else {
		/* x is a control point in the data */
		return (*range.first)->value;
	}
}

AutomationList*
AutomationList::cut (iterator start, iterator end)
{
	AutomationList* nal = new AutomationList (default_value);

	{
		LockMonitor lm (lock, __LINE__, __FILE__);

		for (iterator x = start; x != end; ) {
			iterator tmp;
			
			tmp = x;
			++tmp;
			
			nal->events.push_back (point_factory (**x));
			events.erase (x);
			
			reposition_for_rt_add (rt_pos);

			x = tmp;
		}
	}

	maybe_signal_changed ();

	return nal;
}

AutomationList*
AutomationList::cut_copy_clear (double start, double end, int op)
{
	AutomationList* nal = new AutomationList (default_value);
	iterator s, e;
	ControlEvent cp (start, 0.0);
	TimeComparator cmp;
	bool changed = false;

	{
		LockMonitor lm (lock, __LINE__, __FILE__);

		if ((s = lower_bound (events.begin(), events.end(), &cp, cmp)) == events.end()) {
			return nal;
		}

		cp.when = end;
		e = upper_bound (events.begin(), events.end(), &cp, cmp);

		if (op != 2 && (*s)->when != start) {
			nal->events.push_back (point_factory (0, _eval (start)));
		}

		for (iterator x = s; x != e; ) {
			iterator tmp;
			
			tmp = x;
			++tmp;

			changed = true;
			
			/* adjust new points to be relative to start, which
			   has been set to zero.
			*/
			
			if (op != 2) {
				nal->events.push_back (point_factory ((*x)->when - start, (*x)->value));
			}

			if (op != 1) {
				events.erase (x);
			}
			
			x = tmp;
		}

		if (op != 2) {
			nal->events.push_back (point_factory (end - start, _eval (end)));
		}

		if (changed) {
			reposition_for_rt_add (rt_pos);
			if (!no_state) {
				save_state (_("cut/copy/clear"));
			}
		}
	}

	maybe_signal_changed ();

	return nal;

}

AutomationList*
AutomationList::copy (iterator start, iterator end)
{
	AutomationList* nal = new AutomationList (default_value);

	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		
		for (iterator x = start; x != end; ) {
			iterator tmp;
			
			tmp = x;
			++tmp;
			
			nal->events.push_back (point_factory (**x));
			
			x = tmp;
		}

		if (!no_state) {
			save_state (_("copy"));
		}
	}

	return nal;
}

AutomationList*
AutomationList::cut (double start, double end)
{
	return cut_copy_clear (start, end, 0);
}

AutomationList*
AutomationList::copy (double start, double end)
{
	return cut_copy_clear (start, end, 1);
}

void
AutomationList::clear (double start, double end)
{
	(void) cut_copy_clear (start, end, 2);
}

bool
AutomationList::paste (AutomationList& alist, double pos, float times)
{
	if (alist.events.empty()) {
		return false;
	}

	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		iterator where;
		iterator prev;
		double shift;
		bool at_start;
		ControlEvent cp (pos, 0.0);
		TimeComparator cmp;

		where = upper_bound (events.begin(), events.end(), &cp, cmp);

		prev = where;
		if (!(at_start = (prev == events.begin()))) {
			--prev;
		} 

		shift = pos;

		events.insert (where, alist.events.begin(), alist.events.end());
		
		/* move to first inserted point */
		
		if (at_start) {
			prev = events.begin();
		} else {
			++prev;
		}

		/* move all inserted points along the timeline by 
		   the correct amount.
		*/

		while (prev != events.end()) {
			(*prev)->when += shift;
			++prev;
		}

		reposition_for_rt_add (rt_pos);

		if (!no_state) {
			save_state (_("paste"));
		}
	}

	maybe_signal_changed ();
	return true;
}

ControlEvent*
AutomationList::point_factory (double when, double val) const
{
	return new ControlEvent (when, val);
}

ControlEvent*
AutomationList::point_factory (const ControlEvent& other) const
{
	return new ControlEvent (other);
}
