/*
 *  Copyright (C) 2002 Steve Harris <steve@plugin.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Id: dither.cc,v 1.3 2004/02/22 23:04:00 taybin Exp $
 */

#include <ardour/dither.h>

/* this monstrosity is necessary to get access to lrintf() and random().
   whoever is writing the glibc headers <cmath> and <cstdlib> should be
   hauled off to a programmer re-education camp. for the rest of
   their natural lives. or longer. <paul@linuxaudiosystems.com>
*/

#define	_ISOC9X_SOURCE	1
#define _ISOC99_SOURCE	1
#include <cmath>
#undef  _ISOC99_SOURCE
#undef  _ISOC9X_SOURCE
#undef  __USE_SVID 
#define __USE_SVID 1
#include <cstdlib>
#undef  __USE_SVID

#include <sys/types.h>

/* Can be overrriden with any code that produces whitenoise between 0.0f and
 * 1.0f */
#ifndef DITHER_NOISE
#define DITHER_NOISE (random() / (float)RAND_MAX)
#endif

/* Lipshitz's minimally audible FIR */
static const float shaped_bs[] = { 2.033f, -2.165f, 1.959f, -1.590f, 0.6149f };

Dither *dither_new (DitherType type, unsigned int channels, unsigned int bit_depth)
{
    Dither *s;

    s = (Dither*) calloc(1, sizeof(Dither));
    s->type = type;
    s->channels = channels;
    s->bit_depth = bit_depth;

    switch (bit_depth) {
    case 8:
	/* Unsigned 8 bit */
	s->scale = 128.0f;
	s->bias = 1.0f;
	s->clamp_u = 255;
	s->clamp_l = 0;
	break;
    case 16:
	/* Signed 16 bit */
	s->scale = 32768.0f;
	s->bias = 0.0f;
	s->clamp_u = 32767;
	s->clamp_l = -32768;
	break;
    case 24:
	/* Signed 24 bit, in upper 24 bits of 32 bit word */
	s->scale = 8388608.0f;
	s->bias = 0.0f;
	s->clamp_u = 8388607;
	s->clamp_l = -8388608;
	break;
    default:
	/* Not a bit depth we can handle */
	free(s);
	return NULL;
	break;
    }

    switch (type) {
    case DitherNone:
    case DitherRect:
	/* No state */
	break;
    case DitherTri:
	/* The last whitenoise sample */
	s->tri_state = (float*) calloc(channels, sizeof(float));
	break;
    case DitherShaped:
	/* The error from the last few samples encoded */
	s->shaped_state = (DitherShapedState*) calloc(channels, sizeof(DitherShapedState));
	break;
    }

    return s;
}

void dither_free(Dither * s)
{
    if (s) {
	free(s->tri_state);
	free(s->shaped_state);
	free(s);
    }
}

void dither_run(Dither * s, unsigned int channel, unsigned int length, float *x, void *y)
{
    unsigned int pos, i;
    u_int8_t *o8 = (u_int8_t*) y;
    int16_t *o16 = (int16_t*) y;
    int32_t *o32 = (int32_t*) y;
    float tmp, r, ideal;
    int64_t clamped;
    DitherShapedState *ss = NULL;

    if (!s || channel >= s->channels) {
	return;
    }

    if (s->shaped_state) {
	ss = s->shaped_state + channel;
    }

    for (pos = 0; pos < length; pos++) {
	i = channel + (pos * s->channels);
	tmp = (x[i] + s->bias) * s->scale;

	switch (s->type) {
	case DitherNone:
	    break;
	case DitherRect:
	    tmp -= DITHER_NOISE;
	    break;
	case DitherTri:
	    r = DITHER_NOISE * 2.0f - 1.0f;
	    tmp += r - s->tri_state[channel];
	    s->tri_state[channel] = r;
	    break;
	case DitherShaped:
	    /* Save raw value for error calculations */
	    ideal = tmp;

	    /* Run FIR and add white noise */
	    tmp += ss->buffer[ss->phase] * shaped_bs[0]
		   + ss->buffer[(ss->phase - 1) & DITHER_SH_BUF_MASK]
		     * shaped_bs[1]
		   + ss->buffer[(ss->phase - 2) & DITHER_SH_BUF_MASK]
		     * shaped_bs[2]
		   + ss->buffer[(ss->phase - 3) & DITHER_SH_BUF_MASK]
		     * shaped_bs[3]
		   + ss->buffer[(ss->phase - 4) & DITHER_SH_BUF_MASK]
		     * shaped_bs[4]
		   + DITHER_NOISE;

	    /* Roll buffer and store last error */
	    ss->phase = (ss->phase + 1) & DITHER_SH_BUF_MASK;
	    ss->buffer[ss->phase] = (float)lrintf(tmp) - ideal;
	    break;
	}
	
	clamped = lrintf(tmp);
	if (clamped > s->clamp_u) {
		clamped = s->clamp_u;
	} else if (clamped < s->clamp_l) {
		clamped = s->clamp_l;
	}

	switch (s->bit_depth) {
	case 8:
	    o8[i] = (u_int8_t) clamped;
	    break;
	case 16:
	    o16[i] = (int16_t) clamped;
	    break;
	case 24:
	    o32[i] = (int32_t) (clamped * 256);
	    break;
	}
    }
}

