/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: globals.cc,v 1.27 2003/11/21 04:47:44 pbd Exp $
*/

#include <cstdio> // Needed so that libraptor (included in lrdf) won't complain
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <locale.h>

#include <lrdf.h>

#include <pbd/error.h>

#include <midi++/port.h>
#include <midi++/port_request.h>
#include <midi++/manager.h>
#include <midi++/mmc.h>

#include <ardour/ardour.h>
#include <ardour/audio_library.h>
#include <ardour/configuration.h>
#include <ardour/ladspa_plugin.h>
#include <ardour/version.h>
#include <ardour/source.h>
#include <ardour/cycles.h>
#include <ardour/utils.h>

#include "i18n.h"

ARDOUR::Configuration *ARDOUR::Config = 0;
ARDOUR::AudioLibrary *ARDOUR::Library = 0;
LADSPA::Manager *LADSPA::manager = 0;

using namespace ARDOUR;
using namespace std;

MIDI::Port *default_mmc_port = 0;
MIDI::Port *default_mtc_port = 0;

Change ARDOUR::StartChanged = ARDOUR::new_change ();
Change ARDOUR::LengthChanged = ARDOUR::new_change ();
Change ARDOUR::PositionChanged = ARDOUR::new_change ();
Change ARDOUR::NameChanged = ARDOUR::new_change ();
Change ARDOUR::BoundsChanged = Change (0); // see init(), below

static int 
setup_midi ()
{
	Configuration::MidiPortDescriptor *mtc_port_descriptor = 0;
	Configuration::MidiPortDescriptor *mmc_port_descriptor = 0;
	int nports;

	if ((nports = Config->midi_ports.size()) == 0) {
		warning << _("no MIDI ports specified: no MMC or MTC control possible") << endmsg;
		return 0;
	}

	if (nports > 1) {

		/* More than one port, so try using specific names for each port */

		map<string,Configuration::MidiPortDescriptor *>::iterator i;

		if (Config->get_mmc_port_name() != N_("default")) {
			if ((i = Config->midi_ports.find (Config->get_mmc_port_name())) == Config->midi_ports.end()) {
				error << compose(_("MMC MIDI port \"%1\" is not defined in your Ardour rc file. Using first port instead."), Config->get_mmc_port_name()) << endmsg;
			} else {
				mmc_port_descriptor = i->second;
			}
		} 

		if (Config->get_mtc_port_name() != N_("default")) {
			if ((i = Config->midi_ports.find (Config->get_mtc_port_name())) == Config->midi_ports.end()) {
				error << compose(_("MTC MIDI port \"%1\" is not defined in your Ardour rc file. Using first port instead."), Config->get_mtc_port_name()) << endmsg;
			} else {
				mtc_port_descriptor = i->second;
			}
		} 

		/* If that didn't work, just use the first listed port */

		if (mtc_port_descriptor == 0) {
			mtc_port_descriptor = Config->midi_ports.begin()->second;
		}

		if (mmc_port_descriptor == 0) {
			mmc_port_descriptor = Config->midi_ports.begin()->second;
		}



	} else {

		/* Only one port described, so use it for both MTC and MMC */

		mtc_port_descriptor = Config->midi_ports.begin()->second;
		mmc_port_descriptor = Config->midi_ports.begin()->second;
	}


	cerr << "MIDI: MTC on port " << mtc_port_descriptor->tag
	     << " MMC on port " << mmc_port_descriptor->tag
	     << endl;

	/* Create MMC port and MMC object */
	
	MIDI::PortRequest request (mmc_port_descriptor->device, 
				   mmc_port_descriptor->tag, 
				   mmc_port_descriptor->mode, 
				   mmc_port_descriptor->type);
	
	if (request.status != MIDI::PortRequest::OK) {
		error << compose(_("MIDI port specifications for \"%1\" are not understandable."), mmc_port_descriptor->tag) << endmsg;
		return -1;
	}
		
	default_mmc_port = MIDI::Manager::instance()->add_port (request);
	
	if (default_mmc_port == 0) {
		warning << compose(_("MMC MIDI port \"%1\" not available: no MMC control possible"), mmc_port_descriptor->tag) << endmsg;
		return 0;
	} 

	if (mtc_port_descriptor != mmc_port_descriptor) {

		MIDI::PortRequest mtc_request (mtc_port_descriptor->device, 
					       mtc_port_descriptor->tag, 
					       mtc_port_descriptor->mode, 
					       mtc_port_descriptor->type);
		
		if (request.status != MIDI::PortRequest::OK) {
			error << compose(_("MIDI port specifications for \"%1\" are not understandable."), mtc_port_descriptor->tag) << endmsg;
			return -1;
		}
		
		default_mtc_port = MIDI::Manager::instance()->add_port (mtc_request);
		
		if (default_mtc_port == 0) {
			warning << compose(_("MMC MIDI port \"%1\" not available: no MTC control possible"), mtc_port_descriptor->tag) << endmsg;
			return 0;
		} 

	} else {
		default_mtc_port = default_mmc_port;
	}

	return 0;
}

int
ARDOUR::init (AudioEngine& engine)
{
	bindtextdomain(PACKAGE, LOCALEDIR);
	
	Config = new Configuration;
	
	if (Config->load_state ()) {
		return -1;
	}

	if (setup_midi ()) {
		return -1;
	}

	lrdf_init();
	Library = new AudioLibrary;

	LADSPA::manager = new LADSPA::Manager (engine);

	BoundsChanged = Change (StartChanged|PositionChanged|LengthChanged);

	return 0;
}

int
ARDOUR::cleanup ()
{
	delete Library;
	lrdf_cleanup ();
	return 0;
}

ARDOUR::id_t
ARDOUR::new_id ()
{
	/* XXX this is just a hack. Do something better */

	id_t now;
	now = get_cycles();
	return now;
}

ARDOUR::Change
ARDOUR::new_change ()
{
	Change c;
	static unsigned long change_bit = 1;

	/* XXX catch out-of-range */

	c = Change (change_bit);
	change_bit <<= 1;

	return c;
}

static string
find_file (string name, string dir)
{
	string path;

	/* stop A: ~/.ardour/... */

	path = getenv ("HOME");

	if (path.length()) {
		
		path += "/.ardour/";

		/* try to ensure that the directory exists.
		   failure doesn't mean much here.
		*/

		mkdir (path.c_str(), 0755);

		path += name;

		if (access (path.c_str(), R_OK) == 0) {
			return path;
		}
	}

	/* stop B: dir/... */

	path = dir;
	path += "/ardour/";
	path += name;
	
	if (access (path.c_str(), R_OK) == 0) {
		return path;
	}

	return "";
}

string
ARDOUR::find_config_file (string name)
{
	return find_file (name, CONFIG_DIR);
}

string
ARDOUR::find_data_file (string name)
{
	return find_file (name, DATA_DIR);
}
