/*
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1988-1990 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


#ifndef ArgusUtil
#define ArgusUtil
#endif

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>

#include <netinet/in.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>

#include <compat.h>
#include <pcap.h>

#include <interface.h>

#include <argus_util.h>
#include <argus_out.h>
#include <argus_parse.h>
#include <argus_filter.h>
#include <argus_client.h>


int target_flags = 0;
extern void ArgusLog (int, char *, ...);

void
ArgusAdjustGlobalTime (struct timeval *now)
{
   struct timeval ArgusTimeDelta;

   ArgusTimeDelta.tv_sec  = ArgusRealTime.tv_sec  - ArgusGlobalTime.tv_sec;
   ArgusTimeDelta.tv_usec = ArgusRealTime.tv_usec - ArgusGlobalTime.tv_usec;

   ArgusGlobalTime.tv_sec  = now->tv_sec  - ArgusTimeDelta.tv_sec;
   ArgusGlobalTime.tv_usec = now->tv_usec - ArgusTimeDelta.tv_usec;

   if (ArgusGlobalTime.tv_usec < 0) {
      ArgusGlobalTime.tv_sec--;
      ArgusGlobalTime.tv_usec += 1000000;
   } else {
      if (ArgusGlobalTime.tv_usec > 1000000) {
         ArgusGlobalTime.tv_sec++;
            ArgusGlobalTime.tv_usec -= 1000000;
      }
   }

   ArgusRealTime = *now;
}


struct ArgusRecord *ArgusSubtractRecord (struct ArgusRecord *, struct ArgusRecord *);

struct ArgusRecord *
ArgusSubtractRecord (struct ArgusRecord *arg1, struct ArgusRecord *arg2)
{
   struct ArgusRecord *retn = arg1;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   unsigned int status, status1 = 0, status2 = 0;
   int i;

   status1 = ArgusIndexRecord (arg1, a1farhdr);
   status2 = ArgusIndexRecord (arg2, a2farhdr);

   status = (status1 & status2);

   for (i = 0; i < 32; i++) {
      if (status & 0x01) {
         switch (i) {
            case ARGUS_MAC_DSR_INDEX:
            case ARGUS_VLAN_DSR_INDEX:
            case ARGUS_MPLS_DSR_INDEX:
            case ARGUS_ICMP_DSR_INDEX:
            case ARGUS_IGMP_DSR_INDEX:
            case ARGUS_ARP_DSR_INDEX:
            case ARGUS_FRG_DSR_INDEX:
            case ARGUS_SRCUSRDATA_DSR_INDEX:
            case ARGUS_DSTUSRDATA_DSR_INDEX:
            case ARGUS_ESP_DSR_INDEX:
               break;

            case ARGUS_TCP_DSR_INDEX: {
               struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
               struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

               tcp1->src.ackbytes -= tcp2->src.ackbytes;
               tcp1->src.bytes    -= tcp2->src.bytes;
               tcp1->src.rpkts    -= tcp2->src.rpkts;

               tcp1->dst.ackbytes -= tcp2->dst.ackbytes;
               tcp1->dst.bytes    -= tcp2->dst.bytes;
               tcp1->dst.rpkts    -= tcp2->dst.rpkts;

               break;
            }

            case ARGUS_FAR_DSR_INDEX: {
               struct ArgusFarStruct *far1 = (struct ArgusFarStruct *) a1farhdr[i];
               struct ArgusFarStruct *far2 = (struct ArgusFarStruct *) a2farhdr[i];

               far1->time.start.tv_sec  -= far2->time.start.tv_sec;
               far1->time.start.tv_usec -= far2->time.start.tv_usec;

               if (far1->time.start.tv_usec < 0) {
                  far1->time.start.tv_sec--;
                  far1->time.start.tv_usec += 1000000;
               }

               far1->time.last.tv_sec   -= far2->time.last.tv_sec;
               far1->time.last.tv_usec  -= far2->time.last.tv_usec;

               if (far1->time.last.tv_usec < 0) {
                  far1->time.last.tv_sec--;
                  far1->time.last.tv_usec += 1000000;
               }

               far1->src.count    -= far2->src.count;
               far1->src.bytes    -= far2->src.bytes;
               far1->src.appbytes -= far2->src.appbytes;

               far1->dst.count    -= far2->dst.count;
               far1->dst.bytes    -= far2->dst.bytes;
               far1->dst.appbytes -= far2->dst.appbytes;

               break;
            }

            case ARGUS_RTP_DSR_INDEX: {
               struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
               struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];

               rtp1->sdrop -= rtp2->sdrop;
               rtp1->ddrop -= rtp2->ddrop;
               rtp1->ssdev -= rtp2->ssdev;

               break;
            }

            case ARGUS_TIME_DSR_INDEX: {
/*
               struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
               struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];
*/
               break;
            }

            case ARGUS_AGR_DSR_INDEX: {
/*
               struct ArgusAGRStruct *agr1 = (struct ArgusAGRStruct *) a1farhdr[i];
               struct ArgusAGRStruct *agr2 = (struct ArgusAGRStruct *) a2farhdr[i];
*/
               break;
            }
         }
      }

      status >>= 1;
   }

   return (retn);
}


void
ArgusZeroRecord (struct ArgusRecord *argus)
{
   struct ArgusFarHeaderStruct *a1farhdr[32];
   unsigned int status = 0;
   int i;

   status = ArgusIndexRecord (argus, a1farhdr);

   for (i = 0; i < 32; i++) {
      if (status & 0x01) {
         switch (i) {
            case ARGUS_MAC_DSR_INDEX:
            case ARGUS_VLAN_DSR_INDEX:
            case ARGUS_MPLS_DSR_INDEX:
            case ARGUS_ICMP_DSR_INDEX:
            case ARGUS_IGMP_DSR_INDEX:
            case ARGUS_ARP_DSR_INDEX:
            case ARGUS_FRG_DSR_INDEX:
            case ARGUS_SRCUSRDATA_DSR_INDEX:
            case ARGUS_DSTUSRDATA_DSR_INDEX:
            case ARGUS_ESP_DSR_INDEX:
            case ARGUS_AGR_DSR_INDEX:
               break;

            case ARGUS_TCP_DSR_INDEX: {
               struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) a1farhdr[i];

               bzero ((char *)&tcp->src, sizeof(struct ArgusTCPObjectMetrics));
               bzero ((char *)&tcp->dst, sizeof(struct ArgusTCPObjectMetrics));
               break;
            }

            case ARGUS_FAR_DSR_INDEX: {
               struct ArgusFarStruct *far = (struct ArgusFarStruct *) a1farhdr[i];

               bzero ((char *)&far->src, sizeof(struct ArgusMeter));
               bzero ((char *)&far->dst, sizeof(struct ArgusMeter));
               break;
            }

            case ARGUS_RTP_DSR_INDEX: {
               struct ArgusRTPObject *rtp = (struct ArgusRTPObject *) a1farhdr[i];

               rtp->sdrop = 0;
               rtp->ssdev = 0;
               break;
            }

            case ARGUS_TIME_DSR_INDEX: {
               struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) a1farhdr[i];

               bzero ((char *)&time->src, sizeof(struct ArgusTimeEntity));
               bzero ((char *)&time->dst, sizeof(struct ArgusTimeEntity));
               break;
            }
         }
      }

      status >>= 1;
   }
}


void
ArgusReverseRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *tmp = RaCopyArgusRecord(argus);
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0, status;
   int i;

   a1DSRStatus = ArgusIndexRecord (argus, a1farhdr);
   a2DSRStatus = ArgusIndexRecord (tmp,   a2farhdr);

   status = a1DSRStatus;
 
   for (i = 0; i < 32; i++) {
      if (status & 0x01) {
         switch (i) {
            case ARGUS_FAR_DSR_INDEX: {
               struct ArgusFarStruct *far1 = (struct ArgusFarStruct *) a1farhdr[i];
               struct ArgusFarStruct *far2 = (struct ArgusFarStruct *) a2farhdr[i];

               if ((argus->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                  far2->flow.ip_flow.ip_src = far1->flow.ip_flow.ip_dst;
                  far2->flow.ip_flow.ip_dst = far1->flow.ip_flow.ip_src;
                  switch (far1->flow.ip_flow.ip_p) {
                     case IPPROTO_TCP:
                     case IPPROTO_UDP:
                        far2->flow.ip_flow.sport = far1->flow.ip_flow.dport;
                        far2->flow.ip_flow.dport = far1->flow.ip_flow.sport;
                        break;

                     case IPPROTO_ESP:
                     case IPPROTO_ICMP:
                     case IPPROTO_IGMP:
                        break;
                  }

                  far2->attr_ip.soptions = far1->attr_ip.doptions;
                  far2->attr_ip.doptions = far1->attr_ip.soptions;
                  far2->attr_ip.sttl     = far1->attr_ip.dttl;
                  far2->attr_ip.dttl     = far1->attr_ip.sttl;
                  far2->attr_ip.stos     = far1->attr_ip.dtos;
                  far2->attr_ip.dtos     = far1->attr_ip.stos;
               }

               bcopy ((char *)&far1->src, (char *)&far2->dst, sizeof(struct ArgusMeter));
               bcopy ((char *)&far1->dst, (char *)&far2->src, sizeof(struct ArgusMeter));
               break;
            }

            case ARGUS_MAC_DSR_INDEX: {
               struct ArgusMacStruct *mac1 = (struct ArgusMacStruct *) a1farhdr[i];
               struct ArgusMacStruct *mac2 = (struct ArgusMacStruct *) a2farhdr[i];

               bcopy ((char *)mac1->ether_mac.ethersrc, (char *)mac2->ether_mac.etherdst, 6);
               bcopy ((char *)mac1->ether_mac.etherdst, (char *)mac2->ether_mac.ethersrc, 6);

               break;
            }

            case ARGUS_VLAN_DSR_INDEX: {
               struct ArgusVlanStruct *vlan1 = (struct ArgusVlanStruct *) a1farhdr[i];
               struct ArgusVlanStruct *vlan2 = (struct ArgusVlanStruct *) a2farhdr[i];

               vlan2->sid = vlan1->did;
               vlan2->did = vlan1->sid;

               if (vlan1->status & ARGUS_SRC_VLAN) {
                  vlan2->status &= ~ARGUS_SRC_VLAN;
                  vlan2->status |=  ARGUS_DST_VLAN;
               }
               if (vlan1->status & ARGUS_SRC_CHANGED) {
                  vlan2->status &= ~ARGUS_SRC_CHANGED;
                  vlan2->status |=  ARGUS_DST_CHANGED;
               }
               if (vlan1->status & ARGUS_DST_VLAN) {
                  vlan2->status &= ~ARGUS_DST_VLAN;
                  vlan2->status |=  ARGUS_SRC_VLAN;
               }
               if (vlan1->status & ARGUS_DST_CHANGED) {
                  vlan2->status &= ~ARGUS_DST_CHANGED;
                  vlan2->status |=  ARGUS_SRC_CHANGED;
               }

               break;
            }

            case ARGUS_MPLS_DSR_INDEX: {
               struct ArgusMplsStruct *mpls1 = (struct ArgusMplsStruct *) a1farhdr[i];
               struct ArgusMplsStruct *mpls2 = (struct ArgusMplsStruct *) a2farhdr[i];

               mpls2->slabel = mpls1->dlabel;
               mpls2->dlabel = mpls1->slabel;

               if (mpls1->status & ARGUS_SRC_MPLS) {
                  mpls2->status &= ~ARGUS_SRC_MPLS;
                  mpls2->status |=  ARGUS_DST_MPLS;
               }
               if (mpls1->status & ARGUS_SRC_CHANGED) {
                  mpls2->status &= ~ARGUS_SRC_CHANGED;
                  mpls2->status |=  ARGUS_DST_CHANGED;
               }
               if (mpls1->status & ARGUS_DST_MPLS) {
                  mpls2->status &= ~ARGUS_DST_MPLS; 
                  mpls2->status |=  ARGUS_SRC_MPLS; 
               }
               if (mpls1->status & ARGUS_DST_CHANGED) {
                  mpls2->status &= ~ARGUS_DST_CHANGED;
                  mpls2->status |=  ARGUS_SRC_CHANGED;
               }

               break;
            }

            case ARGUS_AGR_DSR_INDEX: {
               break;
            }

            case ARGUS_TIME_DSR_INDEX: {
               struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
               struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];

               bcopy ((char *)&time1->src, (char *)&time2->dst, sizeof(struct ArgusTimeEntity));
               bcopy ((char *)&time1->dst, (char *)&time2->src, sizeof(struct ArgusTimeEntity));

               break;
            }

            case ARGUS_SRCUSRDATA_DSR_INDEX:
            case ARGUS_DSTUSRDATA_DSR_INDEX: {
               struct ArgusUserStruct *user2 = (struct ArgusUserStruct *) a2farhdr[i];
               if (user2->type == ARGUS_SRCUSRDATA_DSR)
                  user2->type = ARGUS_DSTUSRDATA_DSR;
               else
                  user2->type = ARGUS_SRCUSRDATA_DSR;
               break;
            }

            case ARGUS_TCP_DSR_INDEX: {
               struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
               struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

               bcopy ((char *)&tcp1->src, (char *)&tcp2->dst, sizeof(struct ArgusTCPObjectMetrics));
               bcopy ((char *)&tcp1->dst, (char *)&tcp2->src, sizeof(struct ArgusTCPObjectMetrics));

               break;
            }

            case ARGUS_ICMP_DSR_INDEX: {
               break;
            }


            case ARGUS_RTP_DSR_INDEX: {
               struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
               struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];

               bcopy ((char *)&rtp1->src, (char *)&rtp2->dst, sizeof(struct rtphdr));
               bcopy ((char *)&rtp1->dst, (char *)&rtp2->src, sizeof(struct rtphdr));
               rtp2->sdrop = rtp2->ddrop;
               rtp2->ddrop = rtp2->sdrop;
               rtp2->ssdev = rtp2->dsdev;
               rtp2->dsdev = rtp2->ssdev;

               break;
            }

            case ARGUS_IGMP_DSR_INDEX: {
               break;
            }

            case ARGUS_ARP_DSR_INDEX:
               break;

            case ARGUS_FRG_DSR_INDEX:
               break;

            case ARGUS_ESP_DSR_INDEX: {
               break;
            }
         }
      }

      status >>= 1;
   }

   bcopy ((char *)tmp, (char *)argus, argus->ahdr.length);
   ArgusFree(tmp);
}


unsigned int
ArgusIndexRecord (struct ArgusRecord *argus, struct ArgusFarHeaderStruct **hdrs)
{
   unsigned int retn = 0;
   struct ArgusFarHeaderStruct *far = (struct ArgusFarHeaderStruct *) &argus->argus_far;
   unsigned int length = argus->ahdr.length - sizeof(argus->ahdr);
   unsigned int farlen;
 
   bzero ((char *) hdrs, 32 * sizeof(struct ArgusFarHeaderStruct *));

   if (argus->ahdr.type & ARGUS_FAR) {
      while ((length > 0) && (length >= far->length)) {
         switch (far->type) {
            case ARGUS_FAR:
               if (retn & ARGUS_FAR_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_FAR_DSR_STATUS;
               hdrs[ARGUS_FAR_DSR_INDEX] = far;
               break;
            case ARGUS_MAC_DSR:    
               if (retn & ARGUS_MAC_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_MAC_DSR_STATUS;
               hdrs[ARGUS_MAC_DSR_INDEX] = far;
               break;
            case ARGUS_VLAN_DSR:    
               if (retn & ARGUS_VLAN_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_VLAN_DSR_STATUS;
               hdrs[ARGUS_VLAN_DSR_INDEX] = far;
               break;
            case ARGUS_MPLS_DSR:    
               if (retn & ARGUS_MPLS_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_MPLS_DSR_STATUS;
               hdrs[ARGUS_MPLS_DSR_INDEX] = far;
               break;
            case ARGUS_AGR_DSR:    
               if (retn & ARGUS_AGR_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_AGR_DSR_STATUS;
               hdrs[ARGUS_AGR_DSR_INDEX] = far;
               break;
            case ARGUS_TIME_DSR: 
               if (retn & ARGUS_TIME_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_TIME_DSR_STATUS;
               hdrs[ARGUS_TIME_DSR_INDEX] = far;
               break;
            case ARGUS_SRCUSRDATA_DSR:
               if (retn & ARGUS_SRCUSRDATA_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_SRCUSRDATA_DSR_STATUS;
               hdrs[ARGUS_SRCUSRDATA_DSR_INDEX] = far;
               break;
            case ARGUS_DSTUSRDATA_DSR:
               if (retn & ARGUS_DSTUSRDATA_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_DSTUSRDATA_DSR_STATUS;
               hdrs[ARGUS_DSTUSRDATA_DSR_INDEX] = far;
               break;
            case ARGUS_TCP_DSR:    
               if (retn & ARGUS_TCP_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_TCP_DSR_STATUS;
               hdrs[ARGUS_TCP_DSR_INDEX] = far;
               break;
            case ARGUS_ICMP_DSR:   
               if (retn & ARGUS_ICMP_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_ICMP_DSR_STATUS;
               hdrs[ARGUS_ICMP_DSR_INDEX] = far;
               break;
            case ARGUS_RTP_DSR:    
               if (retn & ARGUS_RTP_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_RTP_DSR_STATUS;
               hdrs[ARGUS_RTP_DSR_INDEX] = far;
               break;
            case ARGUS_IGMP_DSR:   
               if (retn & ARGUS_IGMP_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_IGMP_DSR_STATUS;
               hdrs[ARGUS_IGMP_DSR_INDEX] = far;
               break;
            case ARGUS_ARP_DSR:    
               if (retn & ARGUS_ARP_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_ARP_DSR_STATUS;
               hdrs[ARGUS_ARP_DSR_INDEX] = far;
               break;
            case ARGUS_FRG_DSR:    
               if (retn & ARGUS_FRG_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_FRG_DSR_STATUS;
               hdrs[ARGUS_FRG_DSR_INDEX] = far;
               break;
            case ARGUS_ESP_DSR:    
               if (retn & ARGUS_ESP_DSR_STATUS)
                  return (retn);
               retn |= ARGUS_ESP_DSR_STATUS;
               hdrs[ARGUS_ESP_DSR_INDEX] = far;
               break;
         }
   
         if ((farlen = far->length) == 0)
            break;

         if ((far->type == ARGUS_SRCUSRDATA_DSR) ||
             (far->type == ARGUS_DSTUSRDATA_DSR))
            farlen = farlen * 4;

         length -= farlen;
         far = (struct ArgusFarHeaderStruct *)((char *)far + farlen);
      }
   }

   return (retn);
}

char *ArgusVersionStr =  "Argus Version ";

int
ArgusConvertInitialWriteStruct (struct WriteStruct *ws, struct ArgusRecord *argus)
{
   int retn = 0;
   char *ptr;
   
   if (ws && argus) {
      bzero ((char *) argus, sizeof (*argus));
      if ((ptr = strstr (ws->ws_init.initString, ArgusVersionStr)) != NULL) {
         ptr = &ptr[strlen(ArgusVersionStr)];
         if (sscanf (ptr, "%d.%d", &major_version, &minor_version)) {
            argus->ahdr.type                 = ARGUS_MAR | ARGUS_WRITESTRUCT;
            argus->ahdr.length               = sizeof (*argus);
            argus->ahdr.length               = htons(argus->ahdr.length);
            argus->ahdr.cause                = ARGUS_START;
            argus->ahdr.status               = htonl(ARGUS_VERSION);
            argus->ahdr.argusid              = htonl(ARGUS_COOKIE);
            argus->argus_mar.startime        = ws->ws_init.startime;
            argus->argus_mar.now             = ws->ws_init.now;
            argus->argus_mar.major_version   = major_version;
            argus->argus_mar.minor_version   = minor_version;
            argus->argus_mar.interfaceType   = ws->ws_init.interfaceType;
            argus->argus_mar.interfaceStatus = ws->ws_init.interfaceStatus;

            argus->argus_mar.reportInterval  = ws->ws_init.reportInterval;
            argus->argus_mar.argusMrInterval = ws->ws_init.dflagInterval;
            argus->argus_mar.record_len      = -1;
            retn = 1;
         }
      }
   }

   return (retn);
}

#include <cons_def.h>

#if !defined(__OpenBSD__)
#include <netinet/ip_icmp.h>
#endif

extern int ArgusTotalBytes;
extern int ArgusTotalCount;

extern long long ntohll (long long);

int
ArgusConvertWriteStruct (struct WriteStruct *ws, struct ArgusRecord *argus)
{
   int retn = 0;
   unsigned int status;

   if (ws && argus) {
      bzero ((char *) argus, sizeof (*argus));
      status = ntohl(ws->status);

      if (status & ARGUSCONTROL) {
         argus->ahdr.type                 = ARGUS_MAR | ARGUS_WRITESTRUCT;
         argus->ahdr.length               = sizeof (*argus);
         argus->ahdr.argusid              = 0;
         argus->ahdr.status               = ARGUS_VERSION;

         argus->argus_mar.major_version   = VERSION_MAJOR;
         argus->argus_mar.minor_version   = VERSION_MINOR;
         argus->argus_mar.interfaceType   = ws->ws_stat.interfaceType;
         argus->argus_mar.interfaceStatus = ws->ws_stat.interfaceStatus;
         argus->argus_mar.reportInterval  = ntohs(ws->ws_stat.reportInterval);
         argus->argus_mar.argusMrInterval = ntohs(ws->ws_stat.dflagInterval);

         if (status & CLOSE) {
            argus->ahdr.cause                 = ARGUS_STOP;
            argus->argus_mar.startime.tv_sec  = (ws->ws_stat.startime.tv_sec);
            argus->argus_mar.startime.tv_usec = (ws->ws_stat.startime.tv_usec);
            argus->argus_mar.now.tv_sec       = (ws->ws_stat.now.tv_sec);
            argus->argus_mar.now.tv_usec      = (ws->ws_stat.now.tv_usec);
            argus->argus_mar.pktsRcvd         = ArgusTotalCount;
            argus->argus_mar.bytesRcvd        = ArgusTotalBytes;
            argus->argus_mar.pktsDrop         = ntohl(ws->ws_stat.pktsDrop);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actTCPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actUDPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actIPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloTCPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloUDPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloIPcons);
            argus->argus_mar.actICMPcons      = ntohl(ws->ws_stat.actICMPcons);
            argus->argus_mar.cloICMPcons      = ntohl(ws->ws_stat.cloICMPcons);

         } else {
            argus->ahdr.cause                 = ARGUS_STATUS;
            argus->argus_mar.startime.tv_sec  = ntohl(ws->ws_stat.startime.tv_sec);
            argus->argus_mar.startime.tv_usec = ntohl(ws->ws_stat.startime.tv_usec);
            argus->argus_mar.now.tv_sec       = ntohl(ws->ws_stat.now.tv_sec);
            argus->argus_mar.now.tv_usec      = ntohl(ws->ws_stat.now.tv_usec);

            argus->argus_mar.pktsRcvd         = ntohl(ws->ws_stat.pktsRcvd);
            argus->argus_mar.bytesRcvd        = ntohl(ws->ws_stat.bytesRcvd);

            argus->argus_mar.pktsDrop         = ntohl(ws->ws_stat.pktsDrop);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actTCPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actUDPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actIPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloTCPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloUDPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloIPcons);
            argus->argus_mar.actICMPcons      = ntohl(ws->ws_stat.actICMPcons);
            argus->argus_mar.cloICMPcons      = ntohl(ws->ws_stat.cloICMPcons);
         }

         argus->argus_mar.record_len = -1;
         retn = 1;

      } else {
         struct ArgusMacStruct macbuffer, *mac = &macbuffer;
         unsigned int lasttime = 0;
         unsigned short difftime = 0, ttl = 0;

         argus->ahdr.type                     = ARGUS_FAR | ARGUS_WRITESTRUCT;
         argus->ahdr.length                   = sizeof (argus->ahdr) + sizeof (argus->argus_far);
         argus->ahdr.cause                    = ARGUS_STATUS;
         argus->ahdr.status                   = ARGUS_VERSION;
         argus->ahdr.status                  |= ETHERTYPE_IP;
         argus->ahdr.argusid                  = 0;
         argus->argus_far.type                = ARGUS_FAR;
         argus->argus_far.length              = sizeof(argus->argus_far);
         argus->argus_far.status              = 0;
         argus->argus_far.ArgusTransRefNum    = 0;

         argus->argus_far.flow.ip_flow.ip_src = ntohl(ws->ws_ip.src.s_addr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(ws->ws_ip.dst.s_addr);

         if (!(status & (ICMPPROTO | TCPPROTO | UDPPROTO)) && (status & FRAG_ONLY)) {
            struct ArgusFragObject fragbuf, *frag = &fragbuf;
 
            bzero ((char *) frag, sizeof (*frag));
            frag->type   = ARGUS_FRG_DSR;
            frag->length = sizeof(*frag);
            frag->status = ntohs(ws->ws_ip_frag.status);
 
            argus->argus_far.flow.ip_flow.sport  = ntohs(ws->ws_ip.sport);
            argus->argus_far.flow.ip_flow.dport  = ntohs(ws->ws_ip.dport);
            argus->argus_far.flow.ip_flow.tp_p = ARGUS_FRAG_FLOWTAG;
 
            frag->fragnum = ntohl(ws->ws_ip_frag.fragnum);
            frag->frag_id = ntohl(ws->ws_ip_frag.frag_id);
            frag->totlen  = ntohs(ws->ws_ip_frag.totlen);
            frag->currlen = ntohs(ws->ws_ip_frag.currlen);
            frag->maxfraglen = ntohs(ws->ws_ip_frag.maxfraglen);
 
            argus->argus_far.status |= ARGUS_FRAGMENTS;
            argus->argus_far.src.count = frag->fragnum;
            argus->argus_far.src.bytes = frag->currlen;

            bcopy ((char *)frag, &((char *)argus)[argus->ahdr.length], sizeof(*frag));
            argus->ahdr.length += sizeof(*frag);
 
         } else {
            switch (status & (ICMPPROTO | TCPPROTO | UDPPROTO)) {
               case ICMPPROTO: {
                  struct icmpWriteStruct *icmp = &ws->ws_ip_icmp;
                  struct ArgusICMPFlow *icmpFlow = &argus->argus_far.flow.icmp_flow;

                  argus->argus_far.flow.ip_flow.ip_p = IPPROTO_ICMP;

                  argus->argus_far.src.count = ntohl(ws->ws_ip_udp.src_count);
                  argus->argus_far.src.bytes = ntohl(ws->ws_ip_udp.src_bytes);
                  argus->argus_far.dst.count = ntohl(ws->ws_ip_udp.dst_count);
                  argus->argus_far.dst.bytes = ntohl(ws->ws_ip_udp.dst_bytes);

                  if (!(status & (CON_ESTABLISHED | TIMED_OUT))) {
                     icmpFlow->type   = icmp->type;
                     icmpFlow->code   = icmp->code;
                     icmpFlow->id     = icmp->data;
                     icmpFlow->id     = ntohs(icmpFlow->id);

                  } else {
                     icmpFlow->type   = ((unsigned char *) &ws->ws_ip_udp.src_bytes)[0];
                     icmpFlow->code   = ((unsigned char *) &ws->ws_ip_udp.src_bytes)[1];
                     icmpFlow->id     = ((unsigned short *)&ws->ws_ip_udp.src_bytes)[1];
                     icmpFlow->id     = ntohs(icmpFlow->id);

                     argus->argus_far.src.bytes = 0;
                     argus->argus_far.dst.bytes = 0;
                  }

                  if ((icmpFlow->type == ICMP_UNREACH) && (icmpFlow->code == ICMP_UNREACH_PORT))
                     argus->argus_far.flow.ip_flow.tp_p = ((char *)&ws->ws_ip_icmp.gwaddr.s_addr)[3];

                  break;
               }

               case TCPPROTO: {
                  struct ArgusTCPObject tcpbuffer, *tcp = &tcpbuffer;

                  bzero ((char *) tcp, sizeof (*tcp));
                  tcp->type   = ARGUS_TCP_DSR;
                  tcp->length = sizeof(*tcp);

                  if (status & SAW_SYN)
                     tcp->state |= ARGUS_SAW_SYN;
                  if (status & SAW_SYN_SENT)
                     tcp->state |= ARGUS_SAW_SYN_SENT;
                  if (status & CON_ESTABLISHED)
                     tcp->state |= ARGUS_CON_ESTABLISHED;
                  if (status & SRC_RESET)
                     tcp->state |= ARGUS_SRC_RESET;
                  if (status & DST_RESET)
                     tcp->state |= ARGUS_DST_RESET;
                  if (status & CLOSE_WAITING)
                     tcp->state |= ARGUS_FIN;
                  if (status & NORMAL_CLOSE)
                     tcp->state |= ARGUS_NORMAL_CLOSE;

                  if (status & PKTS_RETRANS) {
                     if (status & SRC_PKTS_RETRANS)
                        tcp->state |= ARGUS_SRC_PKTS_RETRANS;
                     if (status & DST_PKTS_RETRANS)
                        tcp->state |= ARGUS_DST_PKTS_RETRANS;
                  }

                  argus->argus_far.src.count = ntohl(ws->ws_ip_tcp.src_count);
                  argus->argus_far.dst.count = ntohl(ws->ws_ip_tcp.dst_count);

                  if ((status & SAW_SYN) &&
                           !(status & (SAW_SYN_SENT|CON_ESTABLISHED|RESET))) {
                     tcp->src.seqbase = ntohl(ws->ws_ip_inittcp.seq);
                  } else {
                     if ((status & (SAW_SYN | SAW_SYN_SENT)) &&
                                        !(status & (CON_ESTABLISHED))) {
                        tcp->dst.seqbase = ntohl(ws->ws_ip_inittcp.seq);
                     } else {
                        argus->argus_far.src.bytes = ntohl(ws->ws_ip_tcp.src_bytes);
                        argus->argus_far.dst.bytes = ntohl(ws->ws_ip_tcp.dst_bytes);
                     }
                  }

                  bcopy ((char *)tcp, &((char *)argus)[argus->ahdr.length], sizeof(*tcp));
                  argus->ahdr.length += sizeof(*tcp);

                  argus->argus_far.flow.ip_flow.ip_p = IPPROTO_TCP;
                  argus->argus_far.flow.ip_flow.sport  = ntohs(ws->ws_ip.sport);
                  argus->argus_far.flow.ip_flow.dport  = ntohs(ws->ws_ip.dport);
                  break;
               }

               case UDPPROTO:
                  argus->argus_far.flow.ip_flow.ip_p = IPPROTO_UDP;
                  argus->argus_far.flow.ip_flow.sport  = ntohs(ws->ws_ip.sport);
                  argus->argus_far.flow.ip_flow.dport  = ntohs(ws->ws_ip.dport);

               default:
                  argus->argus_far.src.count = ntohl(ws->ws_ip_udp.src_count);
                  argus->argus_far.src.bytes = ntohl(ws->ws_ip_udp.src_bytes);
                  argus->argus_far.dst.count = ntohl(ws->ws_ip_udp.dst_count);
                  argus->argus_far.dst.bytes = ntohl(ws->ws_ip_udp.dst_bytes);
                  break;
            }
         }

         if (status & TIMED_OUT)
            argus->ahdr.cause |= ARGUS_TIMEOUT;

         if (status & FRAGMENTS)
            argus->argus_far.status |= ARGUS_FRAGMENTS;

         if (status & IPOPTIONMASK) {
            argus->argus_far.status |= ARGUS_IPOPTIONS;
            if (status & SSRCROUTE) {
               argus->argus_far.attr_ip.soptions |= ARGUS_SSRCROUTE;
               argus->argus_far.attr_ip.doptions |= ARGUS_SSRCROUTE;
            }
            if (status & LSRCROUTE) {
               argus->argus_far.attr_ip.soptions |= ARGUS_LSRCROUTE;
               argus->argus_far.attr_ip.doptions |= ARGUS_LSRCROUTE;
            }
            if (status & TIMESTAMP) {
               argus->argus_far.attr_ip.soptions |= ARGUS_TIMESTAMP;
               argus->argus_far.attr_ip.doptions |= ARGUS_TIMESTAMP;
            }
            if (status & SECURITY) {
               argus->argus_far.attr_ip.soptions |= ARGUS_SECURITY;
               argus->argus_far.attr_ip.doptions |= ARGUS_SECURITY;
            }
            if (status & RECORDROUTE) {
               argus->argus_far.attr_ip.soptions |= ARGUS_RECORDROUTE;
               argus->argus_far.attr_ip.doptions |= ARGUS_RECORDROUTE;
            }
            if (status & RTRALERT) {
               argus->argus_far.attr_ip.soptions |= ARGUS_RTRALERT;
               argus->argus_far.attr_ip.doptions |= ARGUS_RTRALERT;
            }
         }

         if (status & CON_ESTABLISHED)
            argus->ahdr.status |= ARGUS_CONNECTED;

         argus->argus_far.time.start.tv_sec       = ntohl(ws->ws_ip.startime.tv_sec);
         argus->argus_far.time.start.tv_usec      = ntohl(ws->ws_ip.startime.tv_usec);

         if ((major_version > 1) || (minor_version > 6)) {
            difftime = ntohs(((u_short *)&ws->ws_ip.lasttime.tv_sec)[1]);
            lasttime = ntohl(((arg_uint32)ws->ws_ip.startime.tv_sec)) + difftime;
            ttl = ((u_short *)&ws->ws_ip.lasttime.tv_sec)[0];
            ws->ws_ip.lasttime.tv_sec = lasttime;
         }

         argus->argus_far.time.last.tv_sec    = ws->ws_ip.lasttime.tv_sec;
         argus->argus_far.time.last.tv_usec   = ntohl(ws->ws_ip.lasttime.tv_usec);
         mac->type                            = ARGUS_MAC_DSR;
         mac->length                          = sizeof(*mac);
         bcopy((char *)&ws->ws_ip.ws_phys.ethersrc, (char *)&mac->phys_union.ether.ethersrc, 6);
         bcopy((char *)&ws->ws_ip.ws_phys.etherdst, (char *)&mac->phys_union.ether.etherdst, 6);
 
         bcopy ((char *)mac, &((char *)argus)[argus->ahdr.length], sizeof(*mac));
         argus->ahdr.length += sizeof(*mac);
         retn = 1;
      }

#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif
   }

   return (retn);
}


extern char *ArgusProgramName;

void
ArgusPrintStartDate (char *date, struct ArgusRecord *ptr)
{
   char delim = ' ';
   struct timeval *tvp = NULL;

   if (ptr->ahdr.type & ARGUS_MAR)
      tvp = &ptr->argus_mar.startime;
   else
      tvp = &ptr->argus_far.time.start;

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   sprintf (&date[strlen(date)], "%s%c", print_time(tvp), delim);
}

void
ArgusPrintLastDate (char *date, struct ArgusRecord *ptr)
{
   char delim = ' ';
   struct timeval *tvp = NULL;

   if (ptr->ahdr.type & ARGUS_MAR)
      tvp = &ptr->argus_mar.now;
   else 
      tvp = &ptr->argus_far.time.last;
      
   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   sprintf (&date[strlen(date)], "%s%c", print_time(tvp), delim);
}

void
ArgusPrintDate (char *date, struct ArgusRecord *ptr)
{
   char delim = ' ';

   struct timeval *start = NULL, *last = NULL;

   if (ptr->ahdr.type & ARGUS_MAR) {
      start = &ptr->argus_mar.startime;
      last  = &ptr->argus_mar.now;

   } else {
      start = &ptr->argus_far.time.start;
      last  = &ptr->argus_far.time.last;
   }
   
   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   if (RaPrintStartTime)
      sprintf (&date[strlen(date)], "%s%c", print_time(start), delim);

   if (!((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')))
      if (RaPrintStartTime && RaPrintLastTime)
         sprintf (&date[strlen(date)], " ");

   if (RaPrintLastTime)
      sprintf (&date[strlen(date)], "%s%c", print_time(last), delim);
}

void ArgusPrintSrcRate (char *, struct ArgusRecord *);
void ArgusPrintDstRate (char *, struct ArgusRecord *);
void ArgusPrintRate (char *, struct ArgusRecord *);
void ArgusPrintLoss (char *, struct ArgusRecord *);
void ArgusPrintSrcLoad (char *, struct ArgusRecord *);
void ArgusPrintDstLoad (char *, struct ArgusRecord *);
void ArgusPrintLoad (char *, struct ArgusRecord *);
void ArgusPrintSrcTTL (char *, struct ArgusRecord *);
void ArgusPrintDstTTL (char *, struct ArgusRecord *);
void ArgusPrintSrcTOS (char *, struct ArgusRecord *);
void ArgusPrintDstTOS (char *, struct ArgusRecord *);
void ArgusPrintSrcVLAN (char *, struct ArgusRecord *);
void ArgusPrintDstMPLS (char *, struct ArgusRecord *);
void ArgusPrintWindow (char *, struct ArgusRecord *);
void ArgusPrintDuration (char *, struct ArgusRecord *);
void ArgusPrintAvgDuration (char *, struct ArgusRecord *);
void ArgusPrintTransactions (char *, struct ArgusRecord *);

void
ArgusPrintTransactions (char *buf, struct ArgusRecord *ptr)
{
   struct ArgusAGRStruct *ArgusThisAgr = NULL;
   int hits = 1;

   if (ptr->ahdr.type & ARGUS_MAR) {
      sprintf(&buf[strlen(buf)], "      ");

   } else {
      ArgusThisAgr = (struct ArgusAGRStruct *)ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX];

      if (ArgusThisAgr && (ArgusThisAgr->type == ARGUS_AGR_DSR))
         hits = ArgusThisAgr->count;

      sprintf(&buf[strlen(buf)], "%5d ", hits);
   }
}

void ArgusPrintAverageDuration (char *, struct ArgusRecord *);

void
ArgusPrintAvgDuration (char *date, struct ArgusRecord *ptr)
{
   struct ArgusAGRStruct *ArgusThisAgr = NULL;
   char *sptr, *iptr, delim = ' ';
   struct timeval *start = NULL, *last = NULL;
   struct timeval  buf,  *time = &buf;

   if (ptr->ahdr.type & ARGUS_MAR) {
      start = &ptr->argus_mar.startime;
      last  = &ptr->argus_mar.now;

   } else {
      start = &ptr->argus_far.time.start;
      last  = &ptr->argus_far.time.last;
   }

   ArgusThisAgr = (struct ArgusAGRStruct *)ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX];
   if ((ArgusThisAgr && (ArgusThisAgr->type == ARGUS_AGR_DSR))) {
      int ArgusThisMultiplier = 1000;

      if (ArgusThisAgr->status & ARGUS_AGR_USECACTTIME)
         ArgusThisMultiplier = 1000000;

      time->tv_sec  = ArgusThisAgr->act.meanval / ArgusThisMultiplier;
      time->tv_usec = ArgusThisAgr->act.meanval % ArgusThisMultiplier;

   } else {
      *time = *last;
      time->tv_sec  -= start->tv_sec;
      time->tv_usec -= start->tv_usec;

      if (time->tv_usec < 0) {
         time->tv_sec--;
         time->tv_usec += 1000000;
      }

      if (time->tv_usec >= 1000000) {
         time->tv_sec++;
         time->tv_usec -= 1000000;
      }
   }

   iptr = &date[strlen(date)];
   sptr = &date[strlen(date)];

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      sprintf(sptr, "%u", (unsigned int) time->tv_sec);
   else
      sprintf(sptr, "%8u", (unsigned int) time->tv_sec);

   if (pflag) {
      sptr = &date[strlen(date)];
      sprintf(sptr, ".%06d", (int) time->tv_usec);
      sptr[pflag + 1] = '\0';
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      sprintf(&date[strlen(date)], "%c", delim);
   else
      sprintf(&date[strlen(date)], "%c", ' ');
}

void
ArgusPrintDuration (char *date, struct ArgusRecord *ptr)
{
   char *sptr, *iptr, delim = ' ';
   struct timeval *start = NULL, *last = NULL;
   struct timeval  buf,  *time = &buf;

   if (ptr->ahdr.type & ARGUS_MAR) {
      start = &ptr->argus_mar.startime;
      last  = &ptr->argus_mar.now;

   } else {
      start = &ptr->argus_far.time.start;
      last  = &ptr->argus_far.time.last;
   }

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   iptr = &date[strlen(date)];
   sptr = &date[strlen(date)];

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      sprintf(sptr, "%u", (unsigned int) time->tv_sec);
   else
      sprintf(sptr, "%8u", (unsigned int) time->tv_sec);

   if (pflag) {
      sptr = &date[strlen(date)];
      sprintf(sptr, ".%06d", (int) time->tv_usec);
      sptr[pflag + 1] = '\0';
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      sprintf(&date[strlen(date)], "%c", delim);
   else
      sprintf(&date[strlen(date)], "%c", ' ');
}

void ArgusGetIndicatorString (struct ArgusRecord *, char *);


void
ArgusGetIndicatorString (struct ArgusRecord *argus, char *buf)
{
   char *ptr = buf;

   bzero (buf, 16);

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      if (argus->ahdr.type & ARGUS_MAR) {

      } else {
         if (argus->ahdr.status & ARGUS_MPLS)
            *ptr++ = 'm';

         if (argus->ahdr.status & ARGUS_PPPoE)
            *ptr++ = 'p';

         if (argus->ahdr.status & ARGUS_VLAN)
            *ptr++ = 'q';

         if ((argus->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
            if (argus->argus_far.status & ARGUS_ICMP_MAPPED)
               *ptr++ = 'I';

            if ((argus->argus_far.attr_ip.soptions & ARGUS_FRAGMENTS) ||
                (argus->argus_far.attr_ip.doptions & ARGUS_FRAGMENTS)) {

               if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG)
                  *ptr++ = 'f';
               else
                  *ptr++ = 'F';

               if (argus->argus_far.attr_ip.soptions & ARGUS_FRAGOVERLAP)
                  *ptr++ = 'V';
            }

            if (argus->ahdr.status & ARGUS_MULTIADDR)
               *ptr++ = 'M';

            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
               struct ArgusTCPObject *tcp = NULL;
               unsigned int status;

               tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];
               if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                  if (status) {
                     if (status & ARGUS_PKTS_RETRANS) {
                        if ((status & ARGUS_SRC_PKTS_RETRANS) && (status & ARGUS_DST_PKTS_RETRANS))
                           *ptr++ = '*';
                        else {
                           if (status & ARGUS_SRC_PKTS_RETRANS)
                              *ptr++ = 's';
                           if (status & ARGUS_DST_PKTS_RETRANS)
                              *ptr++ = 'd';
                        }
                     }
                     if (status & ARGUS_WINDOW_SHUT) {
                        if ((status & ARGUS_SRC_WINDOW_SHUT) && (status & ARGUS_DST_WINDOW_SHUT))
                           *ptr++ = '@';
                        else {
                           if (status & ARGUS_SRC_WINDOW_SHUT)
                              *ptr++ = 'S';
                           if (status & ARGUS_DST_WINDOW_SHUT)
                              *ptr++ = 'D';
                        }
                     }

                     if (status & ARGUS_ECN_CONGESTED)
                        *ptr++ = 'E';
                  }
               }
            }

            if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS) {
               struct ArgusESPStruct *esp = NULL;

               if ((esp = (struct ArgusESPStruct *)ArgusThisFarHdrs[ARGUS_ESP_DSR_INDEX]) != NULL) {
                  if ((esp->src.lostseq > 0) && (esp->dst.lostseq > 0))
                     *ptr++ = '*';
                  else {
                     if (esp->src.lostseq > 0)
                        *ptr++ = 's';
                     if (esp->dst.lostseq > 0)
                        *ptr++ = 'd';
                  }
               }
            }
         }
      }

      *ptr = RaFieldDelimiter;

   } else {
      int encdone = 0;

      bcopy ("          ", buf, 9);
      if (argus->ahdr.type & ARGUS_MAR) {
      } else {
         if (argus->ahdr.status & ARGUS_MPLS) {
            buf[0] = 'm';
            encdone++;
         }

         if (argus->ahdr.status & ARGUS_PPPoE) {
            buf[0] = 'p';
            encdone++;
         }

         if (argus->ahdr.status & ARGUS_VLAN) {
            buf[0] = 'q';
            encdone++;
         }

         if (encdone > 1)
            buf[0] = 'E';

         if ((argus->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
               struct ArgusTCPObject *tcp = NULL;
               unsigned int status;

               tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

               if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                  if (status) {
                     if (status & ARGUS_PKTS_RETRANS) {
                        if ((status & ARGUS_SRC_PKTS_RETRANS) && (status & ARGUS_DST_PKTS_RETRANS))
                           buf[1] =  '*';
                        else {
                           if (status & ARGUS_SRC_PKTS_RETRANS)
                              buf[1] = 's';
                           if (status & ARGUS_DST_PKTS_RETRANS)
                              buf[1] = 'd';
                        }
                     }
                     if (status & ARGUS_WINDOW_SHUT) {
                        if ((status & ARGUS_SRC_WINDOW_SHUT) && (status & ARGUS_DST_WINDOW_SHUT))
                           buf[2] = '@';
                        else {
                           if (status & ARGUS_SRC_WINDOW_SHUT)
                              buf[2] = 'S';
                           if (status & ARGUS_DST_WINDOW_SHUT)
                              buf[2] = 'D';
                        }
                     }
                     if (status & ARGUS_ECN_CONGESTED) {
                        if ((status & ARGUS_SRC_CONGESTED) && (status & ARGUS_DST_CONGESTED))
                           buf[2] = 'E';
                        else {
                           if (status & ARGUS_SRC_CONGESTED)
                              buf[2] = 'x';
                           if (status & ARGUS_DST_CONGESTED)
                              buf[2] = 't';
                        }
                     }
                  }
               }
            }

            if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS) {
               struct ArgusESPStruct *esp = NULL;

               if ((esp = (struct ArgusESPStruct *)ArgusThisFarHdrs[ARGUS_ESP_DSR_INDEX]) != NULL) {
                  if ((esp->src.lostseq > 0) && (esp->dst.lostseq > 0))
                     buf[1] = '*';
                  else {
                     if (esp->src.lostseq > 0)
                        buf[1] = 's';
                     if (esp->dst.lostseq > 0)
                        buf[1] = 'd';
                  }
               }
            }

            if (argus->ahdr.status & ARGUS_MULTIADDR)
               buf[3] = 'M';

            if (argus->argus_far.status & ARGUS_ICMP_MAPPED)
               buf[4] = 'I';

            if ((argus->argus_far.attr_ip.soptions & ARGUS_FRAGMENTS) ||
                (argus->argus_far.attr_ip.doptions & ARGUS_FRAGMENTS)) {

               if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG)
                  buf[5] = 'f';
               else
                  buf[5] = 'F';

               if ((argus->argus_far.attr_ip.soptions & ARGUS_FRAGOVERLAP) ||
                    (argus->argus_far.attr_ip.doptions & ARGUS_FRAGOVERLAP)) {
                  buf[5] = 'V';
               }
            }

            if ((argus->argus_far.attr_ip.soptions & ARGUS_IPOPTIONS) ||
                (argus->argus_far.attr_ip.doptions & ARGUS_IPOPTIONS)) {
               switch ((argus->argus_far.attr_ip.soptions | argus->argus_far.attr_ip.doptions)
                                    & ARGUS_IPOPTIONS) {
                  case ARGUS_SSRCROUTE:    buf[6] = 'S'; break;
                  case ARGUS_LSRCROUTE:    buf[6] = 'L'; break;
                  case ARGUS_TIMESTAMP:    buf[6] = 'T'; break;
                  case ARGUS_SECURITY:     buf[6] = '+'; break;
                  case ARGUS_RECORDROUTE:  buf[6] = 'R'; break;
                  case ARGUS_RTRALERT:     buf[6] = 'A'; break;
                  case ARGUS_IPOPTIONS:    buf[6] = 'E'; break;
                  default:                 buf[6] = 'O'; break;
               }
            }
         }
      }
   }

   return;
}


struct ArgusInterfaceStruct interfacetypes [] = {
{  0, "DLT_NULL", "no link-layer encapsulation"},
{  1, "DLT_EN10MB", "Ethernet (10Mb)"},
{  2, "DLT_EN3MB", "Experimental Ethernet (3Mb)"},
{  3, "DLT_AX25", "Amateur Radio AX.25"},
{  4, "DLT_PRONET", "Proteon ProNET Token Ring"},
{  5, "DLT_CHAOS", "Chaos"},
{  6, "DLT_IEEE802", "IEEE 802 Networks"},
{  7, "DLT_ARCNET", "ARCNET"},
{  8, "DLT_SLIP", "Serial Line IP"},
{  9, "DLT_PPP",  "Point-to-point Protocol"},
{ 10,"DLT_FDDI", "FDDI"},
{ 11, "DLT_ATM_RFC1483", "LLC/SNAP encapsulated atm"},
{ 12, "DLT_LOOP", "loopback"},
{100, "DLT_ATM_RFC1483", "LLC/SNAP encapsulated atm"},
{101, "DLT_RAW", "raw IP"},
{102, "DLT_SLIP_BSDOS", "BSD/OS Serial Line IP"},
{103, "DLT_PPP_BSDOS", "BSD/OS Point-to-point Protocol"},
{104, "DLT_CHDLC", "Cisco HDLC"},
{-1, "Undefined", "Undefined"},
};


char argus_strbuf[MAXSTRLEN];
char * get_argus_string (struct ArgusRecord *);

char *
get_man_string (struct ArgusRecord *argus)
{
   return (get_argus_string(argus));
}

unsigned short ArgusThisProto;

void
ArgusPrintSourceID (char *buf, struct ArgusRecord *argus) {
   char argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;

   bzero (argusIDStrBuf, 32);

   if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_ID_IS_IPADDR)
      argusIDStr = ipaddr_string (&argus->ahdr.argusid);
   else
      sprintf (argusIDStr, "%u", argus->ahdr.argusid);

   sprintf(buf, "%-15.15s ", argusIDStr);
}

void ArgusPrintBinNumber (char *, struct ArgusRecord *);
void ArgusPrintBins (char *, struct ArgusRecord *);

void
ArgusPrintBinNumber (char *buf, struct ArgusRecord *argus) {
}


void
ArgusPrintBins (char *buf, struct ArgusRecord *argus) {
}


void
ArgusPrintSequenceNumber (char *buf, struct ArgusRecord *argus) {
   sprintf(buf, "%d ", argus->ahdr.seqNumber);
}


void
ArgusPrintIndicator (char *buf, struct ArgusRecord *argus) {
   ArgusGetIndicatorString (argus, buf);
}

void
ArgusPrintMACAddress (char *buf, struct ArgusRecord *argus) {
   char *blankStr = " ", *esrcString, *edstString;

   ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);
   if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
      struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];
 
      esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
      edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);
 
      sprintf (buf, "%17.17s %17.17s ", esrcString, edstString);

   } else
      sprintf (buf, "%17.17s %17.17s ", blankStr, blankStr);
}

void
ArgusPrintProto (char *buf, struct ArgusRecord *argus)
{
   char protoStrBuf[16], *protoStr = NULL;
   u_short eproto;
   u_char proto;

   bzero (protoStrBuf, 16);

   if (argus->ahdr.type & ARGUS_MAR) {
      sprintf (protoStrBuf, " man ");
       protoStr = protoStrBuf;

   } else {
      switch (eproto = argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP: {
            proto = argus->argus_far.flow.ip_flow.ip_p;
            protoStr = protoStrBuf;
            sprintf (protoStr, "%u", proto);
            protoStr = (nflag > 2) ? protoStrBuf :
                       (proto >= IPPROTOSTR) ? "unas" : ip_proto_string[proto];

            if ((proto == IPPROTO_UDP) && (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG))
               protoStr = "rtp";

            if ((proto == IPPROTO_UDP) && (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTCP_FLOWTAG))
               protoStr = "rtcp";

            break;
         }

         default:
            protoStr = etherproto_string(eproto);
            break;
      }
   }

   if (protoStr != NULL)
      sprintf (buf, "%4.4s ", protoStr);
}

int ArgusPrintNet = 0;

void
ArgusPrintSrcNet (char *buf, struct ArgusRecord *argus)
{
   ArgusPrintNet++;
   ArgusPrintSrcAddr (buf, argus);
   ArgusPrintNet--;
}


void
ArgusPrintSrcAddr (char *buf, struct ArgusRecord *argus)
{
   void *addr = NULL;
   struct ArgusFlow *flow;

   if (argus->ahdr.type & ARGUS_MAR) {
      ArgusThisProto = ETHERTYPE_IP;
      addr = &argus->argus_mar.argusid;

   } else {
      flow = &argus->argus_far.flow;
      ArgusThisProto = argus->ahdr.status & 0xFFFF;

      switch (ArgusThisProto) {
         case ETHERTYPE_IP:
            addr = &flow->ip_flow.ip_src;
            break;

         case ETHERTYPE_REVARP:
         case ETHERTYPE_ARP:
            ArgusThisProto = ETHERTYPE_IP;
            if (Rflag)
               addr = &flow->arp_flow.arp_tpa;
            else 
               addr = &flow->arp_flow.arp_spa;
            break;

         default:
            addr = &flow->mac_flow.ehdr.ether_shost;
            break;
      }
   }

   ArgusPrintAddr (&buf[strlen(buf)], ArgusThisProto, addr);
}



void
ArgusPrintDstNet (char *buf, struct ArgusRecord *argus)
{
   ArgusPrintNet++;
   ArgusPrintDstAddr (buf, argus);
   ArgusPrintNet--;
}


void
ArgusPrintDstAddr (char *buf, struct ArgusRecord *argus)
{
   void *addr = NULL;
   struct ArgusFlow *flow;

   if (argus->ahdr.type & ARGUS_MAR) {
      sprintf (&buf[strlen(buf)], "%*d ", hfield, argus->argus_mar.nextMrSequenceNum);

   } else {
      flow = &argus->argus_far.flow;
      ArgusThisProto = argus->ahdr.status & 0xFFFF;

      switch (ArgusThisProto) {
         case ETHERTYPE_IP:
            addr = &flow->ip_flow.ip_dst;
            break;

         case ETHERTYPE_REVARP:
            ArgusThisProto = ETHERTYPE_IP;
            break;

         case ETHERTYPE_ARP:
            if (Rflag) {
               ArgusThisProto = ARGUS_ETHERTYPE;
               addr = argus->argus_far.attr_arp.response;

            } else {
               ArgusThisProto = ETHERTYPE_IP;
               addr = &flow->arp_flow.arp_tpa;
            }
            break;

         default:
            addr = &flow->mac_flow.ehdr.ether_dhost;
            break;
      }

      ArgusPrintAddr (&buf[strlen(buf)], ArgusThisProto, addr);
   }
}


void
ArgusPrintAddr (char *buf, unsigned short proto, void *addr)
{
   char *addrstr = NULL;
   int thishfield = hfield;

   switch (proto) {
      case ETHERTYPE_IP:
         if (ArgusPrintNet) {
            unsigned int naddr = (*(unsigned int *)addr & ipaddrtonetmask(*(unsigned int *)addr));
            addrstr = ipaddr_string (&naddr);
         } else
            addrstr = ipaddr_string (addr);
         break;

      default:
      case ARGUS_ETHERTYPE:
         addrstr = etheraddr_string ((unsigned char *) addr);
         break;
   }

   if (addr != NULL) 
      sprintf (buf, "%*.*s ", thishfield, thishfield, addrstr);
   else
      sprintf (buf, "%*.*s ", thishfield, thishfield, "                  ");
}


void
ArgusPrintSrcPort (char *buf, struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   ArgusThisProto = argus->ahdr.status & 0xFFFF;

   if (argus->ahdr.type & ARGUS_MAR) {
      int thispfield = pfield;
      char marVersionBuf[32];

      sprintf (marVersionBuf, " v%d.%d", argus->argus_mar.major_version,
                                         argus->argus_mar.minor_version);
      sprintf (&buf[strlen(buf)], "%-*.*s ", thispfield, thispfield, marVersionBuf);

   } else {
      switch (ArgusThisProto) {
         case ETHERTYPE_IP: {
            int adjnflag = 0;
            if (flow->ip_flow.sport != 20) {
               adjnflag = 2;
               nflag += adjnflag;
            }
            ArgusPrintPort (&buf[strlen(buf)], argus, flow->ip_flow.sport);
            if (adjnflag) { nflag -= adjnflag;}
            break;
         }

         case ETHERTYPE_REVARP:
         case ETHERTYPE_ARP:
            sprintf (&buf[strlen(buf)], "%*s", pfield - 1, " ");
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
               sprintf (buf, "%c", RaFieldDelimiter);
            break;

         default:
            ArgusPrintPort (&buf[strlen(buf)], argus, flow->mac_flow.ssap);
            break;
      }
   }
}


int RaPrintIndex = 0;


void
ArgusPrintDstPort (char *buf, struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR) {
      sprintf (&buf[strlen(buf)], "%-*d", pfield + 1, argus->argus_mar.flows);

   } else {
      struct ArgusFlow *flow = &argus->argus_far.flow;
      ArgusThisProto = argus->ahdr.status & 0xFFFF;

      switch (ArgusThisProto) {
         case ETHERTYPE_IP: {
            ArgusPrintPort (&buf[strlen(buf)], argus, flow->ip_flow.dport);
            break;
         }

         case ETHERTYPE_REVARP:
         case ETHERTYPE_ARP:
            if (RaPrintIndex > 0) {
               if ((RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintSrcAddr) ||
                   (RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintDstAddr))
                  sprintf (&buf[strlen(buf)], "%*s", pfield - 1, " ");
               else
                  sprintf (&buf[strlen(buf)], "%*s", pfield + 1, " ");
            } else
               sprintf (&buf[strlen(buf)], "%*s", pfield - 1, " ");

            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
               sprintf (buf, "%c", RaFieldDelimiter);

            break;

         default:
            ArgusPrintPort (&buf[strlen(buf)], argus, flow->mac_flow.dsap);
            break;
      }
   }
}


void
ArgusPrintPort (char *buf, struct ArgusRecord *argus, unsigned short port)
{
   int thispfield = pfield;
   unsigned char proto = argus->argus_far.flow.ip_flow.ip_p;

   switch (ArgusThisProto) {
      case ETHERTYPE_IP: {
         if (port != 0xFFFF) {
            if (nflag > 1) {
               switch (proto) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                     if (RaPrintIndex > 0)
                        if ((RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintSrcAddr) ||
                             (RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintDstAddr))
                           if (RaFieldDelimiter == '\0')
                              if (buf[strlen(buf) - 1] == ' ')
                                 buf[strlen(buf) - 1] = '.';

                     sprintf (buf, "%-*d ", thispfield, port);
                     break;

                  default:
                     if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
                        sprintf (&buf[strlen(buf)], "%c", RaFieldDelimiter);
                     else
                        sprintf (buf, "%-*.*s ", thispfield, thispfield, " ");
                     break;
               }
            } else {
               switch (proto) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                     if (RaPrintIndex > 0)
                        if ((RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintSrcAddr) ||
                             (RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintDstAddr))
                           if (RaFieldDelimiter == '\0')
                              if (buf[strlen(buf) - 1] == ' ')
                                 buf[strlen(buf) - 1] = '.';

                     if (proto == IPPROTO_TCP)
                        sprintf (buf, "%-*.*s ", thispfield, thispfield, tcpport_string(port));
                     else
                        sprintf (buf, "%-*.*s ", thispfield, thispfield, udpport_string(port));
                     break;

                  default:
                     if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
                        sprintf (&buf[strlen(buf)], "%c", RaFieldDelimiter);
                     else
                        sprintf (buf, "%-*.*s ", thispfield, thispfield, " ");
                     break;
               }
            }

         } else {
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
               sprintf (buf, "%c", RaFieldDelimiter);
            } else {
               switch (proto) {
                  default:
                     sprintf (buf, "%-*.*s ", thispfield, thispfield, " ");
                     break;

                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                     if (RaPrintIndex > 0)
                        if ((RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintSrcAddr) ||
                             (RaPrintAlgorithms[RaPrintIndex - 1] == ArgusPrintDstAddr))
                           if (RaFieldDelimiter == '\0')
                              if (buf[strlen(buf) - 1] == ' ')
                                 buf[strlen(buf) - 1] = '.';

                     if (argus->ahdr.status & ARGUS_MERGED)
                        sprintf (buf, "%-*.*s ", thispfield, thispfield, "*");
                     else 
                        sprintf (buf, "%-*d ", thispfield, port);
                     break;
               }
            }
         }

         break;
      }

      default:
         thispfield -= 2;

         if ((ArgusThisProto == 0) && (proto != 0)) {
            sprintf (buf, "%-*.*s ", thispfield, thispfield, llcsap_string((unsigned char) port));
         } else {

            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
               sprintf (buf, "%c", RaFieldDelimiter);
            } else {
               sprintf (buf, "%-*.*s ", thispfield, thispfield, " ");
            }
         }
   }

}


void
ArgusPrintIpId (char *buf, struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   ArgusThisProto = argus->ahdr.status & 0xFFFF;

   if (argus->ahdr.type & ARGUS_MAR) {
      sprintf (&buf[strlen(buf)], "        ");

   } else {
      switch (ArgusThisProto) {
         case ETHERTYPE_IP: {
            sprintf (&buf[strlen(buf)], "0x%04hx ", flow->ip_flow.ip_id);
            break;
         }

         default:
            sprintf (&buf[strlen(buf)], "        ");
            break;
      }
   }
}

char *ArgusProcessStr = NULL;

void
ArgusPrintDir (char *buf, struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;
   unsigned int status;
   int src_count = 0, dst_count = 0;
   char dirStr[8];

   bzero (dirStr, 8);

   if (argus->ahdr.type & ARGUS_MAR) {
      sprintf (dirStr, "   ");

   } else {
      ArgusThisProto = argus->ahdr.status & 0xFFFF;

      switch (ArgusThisProto) {
         case ETHERTYPE_IP: {
            strcpy (dirStr, "<->");
         
            src_count = argus->argus_far.src.count;
            dst_count = argus->argus_far.dst.count;
         
            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
               tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];
         
            if ((tcp != NULL) && ((status = tcp->state) != 0)) {
               if (!(status & (ARGUS_SAW_SYN | ARGUS_SAW_SYN_SENT))) {
                  dirStr[1] = '?';
                  if (src_count) dirStr[2] = '>';
                  if (dst_count) dirStr[0] = '<';
               }


               if (Rflag && (status & ARGUS_RESET)) {
                     dirStr[0] = ' ';
                     dirStr[2] = ' ';
                  if (status & ARGUS_DST_RESET) {
                     dirStr[0] = '<';
                  }
                  if (status & ARGUS_SRC_RESET) {
                     dirStr[2] = '>';
                  }
                  dirStr[1] = '|';
                  ArgusProcessStr = process_state_strings[RA_RST_STATE];
         
               } else 
               if (status & ARGUS_RESET) {
                  dirStr[0] = ' ';
                  dirStr[2] = '>';
                  ArgusProcessStr = process_state_strings[RA_RST_STATE];
               } else
               if (status & ARGUS_NORMAL_CLOSE) {
                  dirStr[0] = ' ';
                  dirStr[2] = '>';
                  ArgusProcessStr = process_state_strings[RA_CLO_STATE];
               } else
               if (status & (ARGUS_FIN | ARGUS_FIN_ACK)) {
                  dirStr[0] = ' ';
                  dirStr[2] = '>';
                  ArgusProcessStr = process_state_strings[RA_FIN_STATE];
               } else
               if (argus->ahdr.cause & ARGUS_TIMEOUT) {
                  if (src_count)
                     dirStr[2] = '>';
                  if (dst_count)
                     dirStr[0] = '<';
                  ArgusProcessStr = process_state_strings[RA_TIM_STATE];
               } else

               if (status & ARGUS_CON_ESTABLISHED) {
                  dirStr[0] = ' ';
                  dirStr[2] = '>';
                  if ((argus->argus_far.src.bytes == 0) && (argus->argus_far.dst.bytes == 0))
                     ArgusProcessStr = process_state_strings[RA_STA_STATE];
                  else
                     ArgusProcessStr = process_state_strings[RA_CON_STATE];
               } else
               if (status & ARGUS_SAW_SYN_SENT) {
                  dirStr[0] = '<';
                  dirStr[2] = ' ';
                  ArgusProcessStr = process_state_strings[RA_ACC_STATE];
               } else
               if (status & ARGUS_SAW_SYN) {
                  dirStr[0] = ' ';
                  dirStr[2] = '>';
                  ArgusProcessStr = process_state_strings[RA_REQ_STATE];
               }
         
            } else {
               if (argus->argus_far.src.count == 0)
                  dirStr[2] = ' ';
         
               if (argus->argus_far.dst.count == 0)
                  dirStr[0] = ' ';
         
               switch (argus->argus_far.flow.ip_flow.ip_p) {
                  case IPPROTO_TCP:
                     dirStr[1] = '?';
                     if (argus->ahdr.cause & ARGUS_START)
                        ArgusProcessStr = "REQ";
                     if (argus->ahdr.cause & ARGUS_STATUS)
                        ArgusProcessStr = "CON";
                     if (argus->ahdr.cause & ARGUS_STOP)
                        ArgusProcessStr = "CLO";
                     if (argus->ahdr.cause & ARGUS_TIMEOUT)
                        ArgusProcessStr = "TIM";
                     if (argus->ahdr.cause & ARGUS_ERROR)
                        ArgusProcessStr = "ERR";
                     break;
         
                  default:
                     if ((argus->ahdr.cause & ARGUS_TIMEOUT))
                        ArgusProcessStr = "TIM";
                     else
                        if (src_count && dst_count) {
                           if ((src_count == 1) && (dst_count == 1))
                              ArgusProcessStr = "ACC";
                           else
                              ArgusProcessStr = "CON";
                        } else
                           if (argus->ahdr.cause & ARGUS_START)
                              ArgusProcessStr = "INT";
                  
                     break;
               }
            }
            break;
         }

         case ETHERTYPE_REVARP:
         case ETHERTYPE_ARP: {
            if (Rflag)
               sprintf (dirStr, " is-at ");
            else
               sprintf (dirStr, "who-has");
            break;
         }

         default:
            src_count = argus->argus_far.src.count;
            dst_count = argus->argus_far.dst.count;
         
            if (src_count == 0)
               sprintf (dirStr, "  <-   ");
            else
            if (dst_count == 0)
               sprintf (dirStr, "   ->  ");
            else
               sprintf (dirStr, "  <->  ");
            break;
      }
   }

   sprintf (buf, "%s ", dirStr);
}

void
ArgusPrintCounts (char *buf, struct ArgusRecord *argus)
{
   int src_count = 0, dst_count = 0;

   if (argus->ahdr.type & ARGUS_MAR) {
      src_count = argus->argus_mar.pktsRcvd;
      dst_count = argus->argus_mar.pktsDrop;
      
   } else {
      src_count = argus->argus_far.src.count;
      dst_count = argus->argus_far.dst.count;
   }

   sprintf (buf, "%-8u %-8u  ", src_count, dst_count);
}

void
ArgusPrintSrcPackets (char *buf, struct ArgusRecord *argus)
{
   int src_count = 0;
 
   if (argus->ahdr.type & ARGUS_MAR) {
      src_count = argus->argus_mar.pktsRcvd;
       
   } else {
      src_count = argus->argus_far.src.count;
   }
 
   sprintf (buf, "%-8u ", src_count);
}

void
ArgusPrintDstPackets (char *buf, struct ArgusRecord *argus)
{
   int dst_count = 0;
 
   if (argus->ahdr.type & ARGUS_MAR) {
      dst_count = argus->argus_mar.pktsDrop;
       
   } else {
      dst_count = argus->argus_far.dst.count;
   }
 
   sprintf (buf, "%-8u ", dst_count);
}

void
ArgusPrintBytes (char *buf, struct ArgusRecord *argus)
{
   int src_bytes = 0, dst_bytes = 0;

   if (argus->ahdr.type & ARGUS_MAR) {
      src_bytes = argus->argus_mar.bytesRcvd;
      dst_bytes = argus->argus_mar.flowsClosed;

   } else {
      if (Aflag) {
         src_bytes = argus->argus_far.src.appbytes;
         dst_bytes = argus->argus_far.dst.appbytes;
      } else {
         src_bytes = argus->argus_far.src.bytes;
         dst_bytes = argus->argus_far.dst.bytes;
      }
   }

   sprintf (buf, "%-12u %-12u", src_bytes, dst_bytes);
}

void
ArgusPrintSrcBytes (char *buf, struct ArgusRecord *argus)
{
   int src_bytes = 0;

   if (argus->ahdr.type & ARGUS_MAR)
      src_bytes = argus->argus_mar.bytesRcvd;

   else {
      if (Aflag) {
         src_bytes = argus->argus_far.src.appbytes;
      } else {
         src_bytes = argus->argus_far.src.bytes;
      }
   }

   sprintf (buf, "%-12u", src_bytes);
}

void
ArgusPrintDstBytes (char *buf, struct ArgusRecord *argus)
{
   int dst_bytes = 0;

   if (argus->ahdr.type & ARGUS_MAR)
      dst_bytes = argus->argus_mar.flowsClosed;
   else 
      if (Aflag) {
         dst_bytes = argus->argus_far.dst.appbytes;
      } else {
         dst_bytes = argus->argus_far.dst.bytes;
      }

   sprintf (buf, "%-12u", dst_bytes);
}



#include <math.h>

void
ArgusPrintSrcJitter (char *buf, struct ArgusRecord *argus)
{
   RaPrintSrcTime = 1;
   RaPrintDstTime = 0;
   ArgusPrintJitter (buf, argus);
}

void
ArgusPrintDstJitter (char *buf, struct ArgusRecord *argus)
{
   RaPrintSrcTime = 1;
   RaPrintDstTime = 0;
   ArgusPrintJitter (buf, argus);
}

void
ArgusPrintJitter (char *buf, struct ArgusRecord *argus)
{
   struct ArgusTimeStruct *time = NULL;
   char stdstr[128], jitterstr[128];
   int maxval = 0, minval = 0x7FFFFFFF, n, meanval = 0;
   long long sumtime = 0, tmpsumtime = 0;
   double sumsqrd = 0.0, stdev = 0.0;

   ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);
   time = (struct ArgusTimeStruct *) ArgusThisFarHdrs[ARGUS_TIME_DSR_INDEX];

   if (RaPrintSrcTime) {
      bzero (stdstr, 128);
      bzero (jitterstr, 128);
      maxval = 0;
      minval = 0x7FFFFFFF;
      n = 0;

      if (time != NULL) {
         if (RaPrintActTime && time->src.act.n) {
            n += time->src.act.n;
            tmpsumtime = (time->src.act.meanval * time->src.act.n);
            sumtime += tmpsumtime;
            sumsqrd += (time->src.act.n * pow(time->src.act.stdev, 2.0)) + pow(tmpsumtime, 2.0)/time->src.act.n;
            maxval = (maxval < time->src.act.maxval) ? time->src.act.maxval : maxval;
            minval = (minval > time->src.act.minval) ? time->src.act.minval : minval;
         }

         if (RaPrintIdleTime && time->src.idle.n) {
            n += time->src.idle.n;
            tmpsumtime = (time->src.idle.meanval * time->src.idle.n);
            sumtime += tmpsumtime;
            sumsqrd += (time->src.idle.n * pow(time->src.idle.stdev, 2.0)) + pow(tmpsumtime, 2.0)/time->src.idle.n;
            maxval = (maxval < time->src.idle.maxval) ? time->src.idle.maxval : maxval;
            minval = (minval > time->src.idle.minval) ? time->src.idle.minval : minval;
         }
      }

      if (n > 0) {
         meanval = sumtime/n;
/*
         stdev = sqrt (sumsqrd/n - pow (meanval, 2.0));
         istdev = (stdev) * 1;

         sprintf (stdstr, "%d.%03d", istdev/1000, istdev%1000);
         stdstr[6] = '\0';
         sprintf (jitterstr, "%6d.%03d (%6s)", meanval/1000, meanval%1000, stdstr);
*/
         sprintf (jitterstr, "%6d.%03d", meanval/1000, meanval%1000);
         sprintf (&buf[strlen(buf)], "%-12s ", jitterstr);

      } else {
         sprintf (jitterstr, "%6d.%03d", meanval/1000, meanval%1000);
         sprintf (&buf[strlen(buf)], "%-12s ", jitterstr);
      }
   }

   if (RaPrintDstTime) {
      bzero (stdstr, 128);
      bzero (jitterstr, 128);
      sumtime = 0;
      sumsqrd = 0.0;
      meanval = 0;
      stdev = 0.0;
      maxval = 0;
      minval = 0x7FFFFFFF;
      n = 0;

      if (time != NULL) {
         if (RaPrintActTime && time->dst.act.n) {
            n += time->dst.act.n;
            tmpsumtime = (time->dst.act.meanval * time->dst.act.n);
            sumtime += tmpsumtime;
            sumsqrd += (time->dst.act.n * pow(time->dst.act.stdev, 2.0)) + pow(tmpsumtime, 2.0)/time->dst.act.n;
            maxval = (maxval < time->dst.act.maxval) ? time->dst.act.maxval : maxval;
            minval = (minval > time->dst.act.minval) ? time->dst.act.minval : minval;
         }

         if (RaPrintIdleTime && time->dst.idle.n) {
            n += time->dst.idle.n;
            tmpsumtime = (time->dst.idle.meanval * time->dst.idle.n);
            sumtime += tmpsumtime;
            sumsqrd += (time->dst.idle.n * pow(time->dst.idle.stdev, 2.0)) + pow(tmpsumtime, 2.0)/time->dst.idle.n;
            maxval = (maxval < time->dst.idle.maxval) ? time->dst.idle.maxval : maxval;
            minval = (minval > time->dst.idle.minval) ? time->dst.idle.minval : minval;
         }
      }

      if (n > 0) {
         meanval = sumtime/n;
/*
         stdev = sqrt (sumsqrd/n - pow (meanval, 2.0));
         istdev = (stdev)*1;

         sprintf (stdstr, "%d.%03d", istdev/1000, istdev%1000);
         stdstr[6] = '\0';
         sprintf (jitterstr, "%6d.%03d (%s)", meanval/1000, meanval%1000, stdstr);
*/
         sprintf (jitterstr, "%6d.%03d", meanval/1000, meanval%1000);
         sprintf (&buf[strlen(buf)], "%-12s", jitterstr);

      } else {
         sprintf (jitterstr, "%6d.%03d", meanval/1000, meanval%1000);
         sprintf (&buf[strlen(buf)], "%-12s", jitterstr);
      }
   }
}

void
ArgusPrintSrcRate (char *buf, struct ArgusRecord *argus)
{
   int src_count = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds;

   start = &argus->argus_far.time.start;
   last  = &argus->argus_far.time.last;

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   src_count = argus->argus_far.src.count;

   if (src_count > 1)
      sprintf (ptr, "%.2f", ((double)(src_count)/seconds));
   else
      sprintf (ptr, "%.2f", 0.0);

   sprintf (&buf[strlen(buf)], "%10s  ", ptr);
}

void
ArgusPrintDstRate (char *buf, struct ArgusRecord *argus)
{
   int dst_count = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds;

   start = &argus->argus_far.time.start;
   last  = &argus->argus_far.time.last;

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   dst_count = argus->argus_far.dst.count;

   if (dst_count > 1)
      sprintf (ptr, "%.2f", ((double)(dst_count)/seconds));
   else
      sprintf (ptr, "%.2f", 0.0);

   sprintf (&buf[strlen(buf)], "%10s  ", ptr);
}

void
ArgusPrintRate (char *buf, struct ArgusRecord *argus)
{
   int src_count = 0, dst_count = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds;

   start = &argus->argus_far.time.start;
   last  = &argus->argus_far.time.last;

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   src_count = argus->argus_far.src.count;
   dst_count = argus->argus_far.dst.count;

   if (src_count > 1)
      sprintf (ptr, "%.2f", ((double)(src_count)/seconds));
   else
      sprintf (ptr, "%.2f", 0.0);

   sprintf (buf, "%10s", ptr);

   if (dst_count > 1)
      sprintf (ptr, "%.2f", ((double)(dst_count)/seconds));
   else
      sprintf (ptr, "%.2f", 0.0);

   sprintf (&buf[strlen(buf)], "%10s  ", ptr);
}

/*

struct ArgusTCPObjectMetrics {
   unsigned int seqbase, ackbytes;
   unsigned int bytes, rpkts;
   unsigned short win;
   unsigned char flags, pad;
};

struct ArgusTCPObject {
   unsigned char type, length;
   unsigned short status;
   unsigned int state;
   unsigned int options;
   unsigned int synAckuSecs, ackDatauSecs;
   struct ArgusTCPObjectMetrics src, dst;
};

*/

void
ArgusPrintLoss (char *buf, struct ArgusRecord *argus)
{
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds, srcLossPkts = 0.0, dstLossPkts = 0.0;

   if (argus->ahdr.type & ARGUS_MAR) {
      sprintf (&buf[strlen(buf)], "%22s", " ");
      
   } else {
      switch (argus->argus_far.flow.ip_flow.ip_p) {
         case IPPROTO_UDP: {
            if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG) {
               if (ArgusThisFarStatus & ARGUS_RTP_DSR_STATUS) {
                  struct ArgusRTPObject *rtp = (void *)ArgusThisFarHdrs[ARGUS_RTP_DSR_INDEX];
                  srcLossPkts = rtp->sdrop;
                  dstLossPkts = rtp->ddrop;
               }
            }
            break;
         }

         case IPPROTO_ICMP: {
            struct ArgusICMPFlow *icmpFlow = &argus->argus_far.flow.icmp_flow;
            struct ArgusICMPObject *icmp = NULL;

            unsigned char ra_icmp_type = 0, ArgusRevIndicator = 0;

            if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS) {
               icmp = (struct ArgusICMPObject *)ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];
               ra_icmp_type = icmp->icmp_type;
            } else
               ra_icmp_type = icmpFlow->type;

            switch (ra_icmp_type) {
               case ICMP_ECHOREPLY:
                  ArgusRevIndicator = 1;
               case ICMP_ECHO:
                  if (ArgusRevIndicator) {
                     if (argus->argus_far.src.count > 0)
                        srcLossPkts = (((argus->argus_far.src.count - argus->argus_far.dst.count) * 100.0) /
                                       ((argus->argus_far.src.count) * 1.0));
                  } else {
                     if (argus->argus_far.src.count > 0)
                        dstLossPkts = (((argus->argus_far.src.count - argus->argus_far.dst.count) * 100.0) /
                                       ((argus->argus_far.src.count) * 1.0));
                  }
                  break;
            }

            break;
         }

         case IPPROTO_TCP: {
            struct ArgusTCPObject *tcp = NULL;
            unsigned int status;

            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
               tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

            if ((tcp != NULL) && ((status = tcp->state) != 0)) {
               if (argus->argus_far.src.count)
                  srcLossPkts = (tcp->src.rpkts * 100.0)/(argus->argus_far.src.count * 1.0);
               if (argus->argus_far.dst.count)
                  dstLossPkts = (tcp->dst.rpkts * 100.0)/(argus->argus_far.dst.count * 1.0);
            }

            break;
         }
      }

      start = &argus->argus_far.time.start;
      last  = &argus->argus_far.time.last;

      *time = *last;
      time->tv_sec  -= start->tv_sec;
      time->tv_usec -= start->tv_usec;

      if (time->tv_usec < 0) {
         time->tv_sec--;
         time->tv_usec += 1000000;
      }

      if (time->tv_usec >= 1000000) {
         time->tv_sec++;
         time->tv_usec -= 1000000;
      }

      seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

      if (seconds) {
         sprintf (ptr, "%.4f", ((double)srcLossPkts));
         sprintf (buf, "%10s", ptr);
         sprintf (ptr, "%.4f", ((double)dstLossPkts));
         sprintf (&buf[strlen(buf)], "%10s  ", ptr);
      } else {
         sprintf (ptr, "%.4f", 0.0);
         sprintf (buf, "%10s", ptr);
         sprintf (ptr, "%.4f", 0.0);
         sprintf (&buf[strlen(buf)], "%10s  ", ptr);
      }
   }
}

void
ArgusPrintSrcLoad (char *buf, struct ArgusRecord *argus)
{
   int src_count = 0, src_bytes = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds;

   start = &argus->argus_far.time.start;
   last  = &argus->argus_far.time.last;

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   src_count = argus->argus_far.src.count;

   if (Aflag) {
      src_bytes = argus->argus_far.src.appbytes;
   } else {
      src_bytes = argus->argus_far.src.bytes;
   }

   if (seconds) {
      sprintf (ptr, "%.2f", ((double)(src_bytes * 8.0))/seconds);
      sprintf (&buf[strlen(buf)], "%11s ", ptr);
   } else {
      sprintf (ptr, "%.2f", 0.0);
      sprintf (&buf[strlen(buf)], "%11s ", ptr);
   }
}


void
ArgusPrintDstLoad (char *buf, struct ArgusRecord *argus)
{
   int dst_count = 0, dst_bytes = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds;

   start = &argus->argus_far.time.start;
   last  = &argus->argus_far.time.last;

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   dst_count = argus->argus_far.dst.count;

   if (Aflag) {
      dst_bytes = argus->argus_far.dst.appbytes;
   } else {
      dst_bytes = argus->argus_far.dst.bytes;
   }

   if (seconds) {
      sprintf (ptr, "%.2f", ((double)(dst_bytes * 8.0))/seconds);
      sprintf (&buf[strlen(buf)], "%13s  ", ptr);
   } else {
      sprintf (ptr, "%.2f", 0.0);
      sprintf (&buf[strlen(buf)], "%13s  ", ptr);
   }
}

void
ArgusPrintLoad (char *buf, struct ArgusRecord *argus)
{
   int src_count = 0, dst_count = 0, src_bytes = 0, dst_bytes = 0;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  timebuf,  *time = &timebuf;
   char tmpbuf[128], *ptr = tmpbuf;
   double seconds;

   start = &argus->argus_far.time.start;
   last  = &argus->argus_far.time.last;

   *time = *last;
   time->tv_sec  -= start->tv_sec;
   time->tv_usec -= start->tv_usec;

   if (time->tv_usec < 0) {
      time->tv_sec--;
      time->tv_usec += 1000000;
   }

   if (time->tv_usec >= 1000000) {
      time->tv_sec++;
      time->tv_usec -= 1000000;
   }

   seconds = (double)(time->tv_sec * 1.0) + (double)(time->tv_usec/1000000.0);

   src_count = argus->argus_far.src.count;
   dst_count = argus->argus_far.dst.count;

   if (Aflag) {
      src_bytes = argus->argus_far.src.appbytes;
      dst_bytes = argus->argus_far.dst.appbytes;
   } else {
      src_bytes = argus->argus_far.src.bytes;
      dst_bytes = argus->argus_far.dst.bytes;
   }

   if (seconds) {
      sprintf (ptr, "%.2f", ((double)(src_bytes * 8.0))/seconds);
      sprintf (buf, "%13s", ptr);
      sprintf (ptr, "%.2f", ((double)(dst_bytes * 8.0))/seconds);
      sprintf (&buf[strlen(buf)], "%13s  ", ptr);
   } else {
      sprintf (ptr, "%.2f", 0.0);
      sprintf (buf, "%13s", ptr);
      sprintf (ptr, "%.2f", 0.0);
      sprintf (&buf[strlen(buf)], "%13s  ", ptr);
   }
}

void
ArgusPrintSrcTTL (char *buf, struct ArgusRecord *argus)
{
   int i;
   if (RaPrintDistance) {
      int sdis = -1;
      for (i = 1; i < 257; i <<= 1) {
         if ((sdis < 0) && (argus->argus_far.attr_ip.sttl <= i)) {
            sdis = i - argus->argus_far.attr_ip.sttl;
         }
      }
      sprintf (&buf[strlen(buf)], "  %3d  ", sdis);
   } else
      sprintf (&buf[strlen(buf)], "  %3d  ", argus->argus_far.attr_ip.sttl);
}

void
ArgusPrintDstTTL (char *buf, struct ArgusRecord *argus)
{
   int i;
   if (RaPrintDistance) {
      int ddis = -1;
      for (i = 1; i < 257; i <<= 1) {
         if ((ddis < 0) && (argus->argus_far.attr_ip.dttl <= i)) {
            ddis = i - argus->argus_far.attr_ip.dttl;
         }
      }
      sprintf (&buf[strlen(buf)], "  %3d  ", ddis);
 
   } else
      sprintf (&buf[strlen(buf)], "  %3d  ", argus->argus_far.attr_ip.dttl);
}

void
ArgusPrintVLAN (char *buf, struct ArgusRecord *argus)
{
   if (ArgusThisFarStatus & ARGUS_VLAN_DSR_STATUS) {
      struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) ArgusThisFarHdrs[ARGUS_VLAN_DSR_INDEX];
 
      if (vlan->status & ARGUS_SRC_VLAN)
         sprintf (&buf[strlen(buf)], "0x%04x ", vlan->sid);
      else
         sprintf(&buf[strlen(buf)], "       ");
      if (vlan->status & ARGUS_DST_VLAN)
         sprintf (&buf[strlen(buf)], "0x%04x ", vlan->did);
      else
         sprintf(&buf[strlen(buf)], "       ");
   } else
      sprintf (&buf[strlen(buf)], "              ");
}

void
ArgusPrintMPLS (char *buf, struct ArgusRecord *argus)
{
   if (ArgusThisFarStatus & ARGUS_MPLS_DSR_STATUS) {
      struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) ArgusThisFarHdrs[ARGUS_MPLS_DSR_INDEX];
 
      if (mpls->status & ARGUS_SRC_MPLS)
         sprintf (&buf[strlen(buf)], " %5x ", mpls->slabel);
      else
         sprintf(&buf[strlen(buf)], "       ");

      if (mpls->status & ARGUS_DST_MPLS)
         sprintf (&buf[strlen(buf)], " %5x ", mpls->dlabel);
      else
         sprintf(&buf[strlen(buf)], "       ");
   } else 
      sprintf (&buf[strlen(buf)], "              ");
}

void
ArgusPrintSrcTOS (char *buf, struct ArgusRecord *argus)
{
   sprintf (&buf[strlen(buf)], "  %3d  ", argus->argus_far.attr_ip.stos);
}

void
ArgusPrintDstTOS (char *buf, struct ArgusRecord *argus)
{
   sprintf (&buf[strlen(buf)], "  %3d  ", argus->argus_far.attr_ip.dtos);
}


void
ArgusPrintWindow (char *buf, struct ArgusRecord *argus)
{
   switch (argus->argus_far.flow.ip_flow.ip_p) {
      case IPPROTO_TCP: {
         unsigned short srcwin, dstwin;
         struct ArgusTCPObject *tcp = NULL;

         if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
            tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

         if (tcp != NULL) {
            srcwin = tcp->src.win;
            dstwin = tcp->dst.win;
            sprintf (&buf[strlen(buf)], "%-5d  %-5d  ", srcwin, dstwin);
         } else {
            sprintf (&buf[strlen(buf)], "%14s", " ");
         }
         break;
      }

      default:
         sprintf (&buf[strlen(buf)], "%14s", " ");
   }
}

void
ArgusPrintTCPExtensions (char *buf, struct ArgusRecord *argus)
{
}


char *ArgusTCPFlags [] = {
   "F", "S", "R", "P", "A", "U", "E", "C"
};


void
ArgusPrintStatus (char *buf, struct ArgusRecord *argus)
{
   if (ArgusProcessStr == NULL)
      ArgusProcessStr = "UNK";

   if (zflag)
      sprintf (buf, "       ");

   if (Zflag)
      sprintf (buf, "              ");

   if (argus->ahdr.type & ARGUS_MAR) {
      ArgusProcessStr = ArgusGetManStatus (argus);

   } else {
      switch (argus->argus_far.flow.ip_flow.ip_p) {
         case IPPROTO_TCP: {
            struct ArgusTCPObject *tcp = NULL;
            char statusbuf[64], *TCPStatusString = statusbuf;
            char SrcTCPFlagsStr[16], DstTCPFlagsStr[16];
            unsigned int status;

            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
               tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

            if ((tcp != NULL) && ((status = tcp->state) != 0)) {
               if (zflag || Zflag) {
                  if (zflag) {
                     bzero ((char *)TCPStatusString, sizeof(statusbuf));
                     if (status & ARGUS_SAW_SYN)         strcat (TCPStatusString, "s");
                     if (status & ARGUS_SAW_SYN_SENT)    strcat (TCPStatusString, "S");
                     if (status & ARGUS_CON_ESTABLISHED) strcat (TCPStatusString, "E");
                     if (status & ARGUS_FIN)             strcat (TCPStatusString, "f");
                     if (status & ARGUS_FIN_ACK)         strcat (TCPStatusString, "F");
                     if (status & ARGUS_NORMAL_CLOSE)    strcat (TCPStatusString, "C");
                     if (status & ARGUS_RESET)           strcat (TCPStatusString, "R");

                  } else {
                     if (Zflag) {
                        int i, index;
                        bzero(SrcTCPFlagsStr, sizeof(SrcTCPFlagsStr));
                        bzero(DstTCPFlagsStr, sizeof(DstTCPFlagsStr));

                        for (i = 0, index = 1; i < 8; i++) {
                           if (tcp->src.flags & index) {
                              strcat (SrcTCPFlagsStr, ArgusTCPFlags[i]);
                           }
                           if (tcp->dst.flags & index) {
                              strcat (DstTCPFlagsStr, ArgusTCPFlags[i]);
                           }
                           index <<= 1;
                        }
                        switch (Zflag) {
                           case 'b':
                              sprintf(TCPStatusString, "%s_%s", SrcTCPFlagsStr, DstTCPFlagsStr);
                              break;
                           case 's':
                              sprintf(TCPStatusString, "%s", SrcTCPFlagsStr);
                              break;
                           case 'd':
                              sprintf(TCPStatusString, "%s", DstTCPFlagsStr);
                              break;
                        }
                     }
                  }

                  ArgusProcessStr = TCPStatusString;
               }
            }

            break;
         }

         case IPPROTO_ICMP: {
            ArgusProcessStr = ArgusGetICMPStatus (argus);
            break;
         }

         default:
            ArgusProcessStr = ArgusGetIPStatus (argus);
            break;
      }
   }

   sprintf (buf, "%s ", ArgusProcessStr);
}


char *
get_tcp_string (struct ArgusRecord *argus)
{
   return (get_argus_string(argus));
}

char *
get_udp_string (struct ArgusRecord *argus)
{
   return (get_argus_string(argus));
}

char *
get_icmp_string (struct ArgusRecord *argus)
{
   return (get_argus_string(argus));
}

char *
get_ip_string (struct ArgusRecord *argus)
{
   return (get_argus_string(argus));
}

char *
get_nonip_string (struct ArgusRecord *argus)
{
   return (get_argus_string(argus));
}

char *
get_argus_string (struct ArgusRecord *argus)
{
   char tmpbuf[MAXSTRLEN], *ptr = tmpbuf;
   char *str = argus_strbuf, lastchr = ' ';
   char date[128];
   int len = 0;

   ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

   bzero (argus_strbuf, MAXSTRLEN);
   bzero (tmpbuf, MAXSTRLEN);

   for (RaPrintIndex = 0; RaPrintIndex < MAX_PRINT_ALG_TYPES; RaPrintIndex++) {
      if (RaPrintAlgorithms[RaPrintIndex] != NULL)
         RaPrintAlgorithms[RaPrintIndex](&argus_strbuf[strlen(argus_strbuf)], argus);
      else
         break;
   }

   while (isspace((int)(argus_strbuf[strlen(argus_strbuf) - 1])))
      argus_strbuf[strlen(argus_strbuf) - 1] = '\0';
   

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      if (RaPrintDate) {
         bzero (date, 128);
         bzero (tmpbuf, MAXSTRLEN);

         ArgusPrintDate (date, argus);

         if (strlen(date) > 0) {
            len = strlen(date) - 1;
            lastchr = date[len];
         }

         bcopy (str, ptr, len);
         str += len;
         ptr += len;
         if (len > 1) {
           if ((ptr[-1] == ' ') && (ptr > tmpbuf))
              *--ptr = '\0';
         }
      }

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }

      if (tmpbuf[strlen(tmpbuf) - 1] == RaFieldDelimiter)
         tmpbuf[strlen(tmpbuf) - 1] = '\0';
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}

char ArgusIPStatus[32];

char *
ArgusGetIPStatus (struct ArgusRecord *argus)
{
   if (argus->argus_far.src.count && argus->argus_far.dst.count) {
      strcpy (ArgusIPStatus, "CON");
   } else {
      if (IN_MULTICAST(argus->argus_far.flow.ip_flow.ip_dst)) 
         strcpy (ArgusIPStatus, "CON");
      else
         strcpy (ArgusIPStatus, "INT");
   }

   return (ArgusIPStatus);
}


char ArgusManStatus[32];

char *
ArgusGetManStatus (struct ArgusRecord *argus)
{
   bzero (ArgusManStatus, 32);

   if (argus->ahdr.cause & ARGUS_START)              sprintf (ArgusManStatus, "STA");
   else if (argus->ahdr.cause & ARGUS_STATUS)        sprintf (ArgusManStatus, "CON");
   else if (argus->ahdr.cause & ARGUS_STOP)          sprintf (ArgusManStatus, "STP");
   else if (argus->ahdr.cause & ARGUS_SHUTDOWN)      sprintf (ArgusManStatus, "SHT");
   else if (argus->ahdr.cause & ARGUS_ERROR)         sprintf (ArgusManStatus, "ERR");
   else if (argus->ahdr.cause & ARGUS_MAXLISTENEXCD) sprintf (ArgusManStatus, "MAX");

   return(ArgusManStatus);
}

int ArgusRevIndicator = 0;

char ArgusICMPStatus[32];

char *
ArgusGetICMPStatus (struct ArgusRecord *argus)
{
   struct ArgusICMPFlow *icmpFlow = &argus->argus_far.flow.icmp_flow;
   struct ArgusICMPObject *icmp = NULL;
   char ArgusResponseString[32];
   char icmptype[32];

   unsigned char ra_icmp_type = 0, ra_icmp_code = 0;
   unsigned short ra_icmp_data = 0;
   unsigned int  ra_src_addr = 0, ra_dst_addr = 0, ra_gw_addr = 0;
      
   bzero (ArgusICMPStatus, 32);
   bzero (ArgusResponseString, 32);
   bzero (icmptype, 32);

   if (argus) {
      if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS) {
         icmp = (struct ArgusICMPObject *)ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];
         if (icmp->type != ARGUS_ICMP_DSR) {
            icmp = NULL;
         } else {
            ra_src_addr  = icmp->isrcaddr;
            ra_dst_addr  = icmp->idstaddr;
            ra_gw_addr   = icmp->igwaddr;
            ra_icmp_type = icmp->icmp_type;
            ra_icmp_code = icmpFlow->code;
         }
      } else {
         ra_icmp_type = icmpFlow->type;
         ra_icmp_code = icmpFlow->code;
      }

      ra_icmp_data = icmpFlow->id;
      
      if (ra_icmp_type < (unsigned char) (ICMP_MAXTYPE + 1))
         strcpy (icmptype, icmptypestr[ra_icmp_type]);
      else
         strcpy (icmptype, "UNK");

      switch (ra_icmp_type) {
         case ICMP_UNREACH:
            switch (ra_icmp_code) {
               case ICMP_UNREACH_NET:
                  strcat (icmptype, "N");
                  if (ra_dst_addr) {
                     u_long addr = ra_dst_addr;
                     sprintf (ArgusResponseString, "net %s", 
                              ipaddr_string (&addr));
                  }
                  break;
               case ICMP_UNREACH_HOST:
                  strcat (icmptype, "H");

                  if (ra_dst_addr)
                     sprintf (ArgusResponseString, "host %s", 
                           ipaddr_string (&ra_dst_addr));
                  break;

               case ICMP_UNREACH_PROTOCOL:
                  strcat (icmptype, "O");
                  if (ra_icmp_data && (ra_icmp_data < IPPROTOSTR))
                     sprintf (ArgusResponseString,"proto %s",
                        ip_proto_string[ra_icmp_data]);
                  break;

               case ICMP_UNREACH_PORT: {
                  int index = icmpFlow->tp_p;
                  strcat (icmptype, "P"); 

                  if ((ra_icmp_data && ((index < IPPROTOSTR)) && (index > 0))) {
                     sprintf (ArgusResponseString, "%s_port     %d", ip_proto_string[index], ra_icmp_data);

                  } else if (ra_icmp_data)
                     sprintf (ArgusResponseString, "port     %d", ra_icmp_data);
                  break;
               }
               case ICMP_UNREACH_NEEDFRAG:
                  strcat (icmptype, "F"); break;
               case ICMP_UNREACH_SRCFAIL:
                  strcat (icmptype, "S"); break;

#ifndef ICMP_UNREACH_NET_UNKNOWN
#define ICMP_UNREACH_NET_UNKNOWN        6
#endif
               case ICMP_UNREACH_NET_UNKNOWN:
                  strcat (icmptype, "NU"); 
                  sprintf (ArgusResponseString, "dst_net unknown"); break;
               
#ifndef ICMP_UNREACH_HOST_UNKNOWN
#define ICMP_UNREACH_HOST_UNKNOWN       7
#endif
               case ICMP_UNREACH_HOST_UNKNOWN:
                  strcat (icmptype, "HU"); 
                  sprintf (ArgusResponseString, "dst_host unknown"); break;

#ifndef ICMP_UNREACH_ISOLATED
#define ICMP_UNREACH_ISOLATED           8
#endif
               case ICMP_UNREACH_ISOLATED:
                  strcat (icmptype, "ISO"); 
                  sprintf (ArgusResponseString, "src_host isolated"); break;

#ifndef ICMP_UNREACH_NET_PROHIB
#define ICMP_UNREACH_NET_PROHIB         9
#endif
               case ICMP_UNREACH_NET_PROHIB:
                  strcat (icmptype, "NPRO"); 
                  sprintf (ArgusResponseString, "admin_net prohib"); break;

#ifndef ICMP_UNREACH_HOST_PROHIB
#define ICMP_UNREACH_HOST_PROHIB        10
#endif
               case ICMP_UNREACH_HOST_PROHIB:
                  strcat (icmptype, "HPRO"); 
                  sprintf (ArgusResponseString, "admin_host prohib"); break;

#ifndef ICMP_UNREACH_TOSNET
#define ICMP_UNREACH_TOSNET             11
#endif
               case ICMP_UNREACH_TOSNET:
                  strcat (icmptype, "NTOS"); 
                  sprintf (ArgusResponseString, "tos_net prohib"); break;

#ifndef ICMP_UNREACH_TOSHOST
#define ICMP_UNREACH_TOSHOST            12
#endif
               case ICMP_UNREACH_TOSHOST:
                  strcat (icmptype, "HTOS"); 
                  sprintf (ArgusResponseString, "tos_host prohib"); break;
 
#ifndef ICMP_UNREACH_FILTER_PROHIB
#define ICMP_UNREACH_FILTER_PROHIB      13
#endif
               case ICMP_UNREACH_FILTER_PROHIB:
                  strcat (icmptype, "FIL"); 
                  sprintf (ArgusResponseString, "admin_filter prohib"); break;

#ifndef ICMP_UNREACH_HOST_PRECEDENCE
#define ICMP_UNREACH_HOST_PRECEDENCE    14
#endif
               case ICMP_UNREACH_HOST_PRECEDENCE:
                  strcat (icmptype, "PRE"); 
                  sprintf (ArgusResponseString, "precedence violation"); break;

#ifndef ICMP_UNREACH_PRECEDENCE_CUTOFF
#define ICMP_UNREACH_PRECEDENCE_CUTOFF  15
#endif
               case ICMP_UNREACH_PRECEDENCE_CUTOFF:
                  strcat (icmptype, "CUT"); 
                  sprintf (ArgusResponseString, "precedence cutoff"); break;

            }
            break;

         case ICMP_MASKREPLY:
            ArgusRevIndicator = 1;
            if (ra_src_addr)
               sprintf (ArgusResponseString, "mask 0x%08x", ra_src_addr);
            break;

         case ICMP_REDIRECT:
            switch (ra_icmp_code) {
            case ICMP_REDIRECT_NET:
               (void) sprintf (ArgusResponseString, "net %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_HOST:
               (void) sprintf (ArgusResponseString, "host %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_TOSNET:
               (void) sprintf (ArgusResponseString, "tosN %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_TOSHOST:
               (void) sprintf (ArgusResponseString, "tosH %s",
                         ipaddr_string (&ra_gw_addr));
               break;
            }
            break;

#ifndef ICMP_ROUTERADVERT
#define ICMP_ROUTERADVERT               9       
#endif
         case ICMP_ROUTERADVERT:
            sprintf (ArgusResponseString, "router advertisement"); break;

#ifndef ICMP_ROUTERSOLICIT
#define ICMP_ROUTERSOLICIT              10     
#endif
         case ICMP_ROUTERSOLICIT:
            sprintf (ArgusResponseString, "router solicitation"); break;


         case ICMP_ECHOREPLY:
         case ICMP_TSTAMPREPLY:
         case ICMP_IREQREPLY:
            ArgusRevIndicator = 1;
            sprintf (ArgusResponseString, "%-6d      %-6d",
                           argus->argus_far.src.bytes, argus->argus_far.dst.bytes);
            break;

         case ICMP_TIMXCEED:
               (void) sprintf (ArgusResponseString, "timexceed %s",
                         ra_icmp_code ? "reassembly" : "in-transit");
               break;

         case ICMP_PARAMPROB:
         case ICMP_SOURCEQUENCH:
         case ICMP_ECHO:
         case ICMP_TSTAMP:
         case ICMP_IREQ:
         case ICMP_MASKREQ:
            sprintf (ArgusResponseString, "%-6d       %-6d",
                           argus->argus_far.src.bytes, argus->argus_far.dst.bytes);
         default:
            sprintf (ArgusResponseString, "%-6d       %-6d",
                           argus->argus_far.src.bytes, argus->argus_far.dst.bytes);
            break;
      }

      if (!(Rflag)) {
         sprintf (ArgusResponseString, "%-6d       %-6d",
            argus->argus_far.src.bytes, argus->argus_far.dst.bytes);
      }

      strcpy (ArgusICMPStatus, icmptype);
   }

   return (ArgusICMPStatus);
}

char *
get_arp_string (argus)
struct ArgusRecord *argus;
{
   return (get_argus_string(argus));
}


#ifdef NOVFPRINTF
/*
 * Stock 4.3 doesn't have vfprintf. 
 * This routine is due to Chris Torek.
 */
vfprintf(f, fmt, args)
   FILE *f;
   char *fmt;
   va_list args;
{
   int ret;

   if ((f->_flag & _IOWRT) == 0) {
      if (f->_flag & _IORW)
         f->_flag |= _IOWRT;
      else
         return EOF;
   }
   ret = _doprnt(fmt, args, f);
   return ferror(f) ? EOF : ret;
}
#endif


extern int ArgusSrcUserDataLen;
extern int ArgusDstUserDataLen;

int ArgusEncode (const char *, const char *, int, char *, int);
int ArgusEncode32 (const char *, int, char *, int);
int ArgusEncode64 (const char *, int, char *, int);
int ArgusEncodeAscii (const char *, int, char *, int);

void
ArgusPrintUserData (char *RaUserDataStr, struct ArgusRecord *argus)
{
   char strbuf[MAXSTRLEN], *str = strbuf;
   char conbuf[MAXSTRLEN], *con = conbuf;
   int len = 0, exlen = 0;
   char delim = ' ';

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   if ((ArgusSrcUserDataLen > 0) || (ArgusDstUserDataLen)) {
      if (ArgusSrcUserDataLen > 0) {
         switch (eflag) {
            case ARGUS_ENCODE_ASCII:
               exlen = ArgusSrcUserDataLen;
               break;

            case ARGUS_ENCODE_32:
            case ARGUS_ENCODE_64:
               exlen = ArgusSrcUserDataLen;
               exlen *= 2;
               break;
         }
         exlen += 8;
         if (ArgusSrcUserDataLen > 10) exlen++;

         if (ArgusThisFarStatus & ARGUS_SRCUSRDATA_DSR_STATUS) {
            struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_SRCUSRDATA_DSR_INDEX];

            len = (user->length - 1) * 4;
            len = (len < argus->argus_far.src.appbytes) ? len : argus->argus_far.src.appbytes;
            len = len > ArgusSrcUserDataLen ? ArgusSrcUserDataLen : len;
     
            bzero (strbuf, sizeof(strbuf));
            bzero (conbuf, sizeof(conbuf));
            if ((len = ArgusEncode (&user->data, NULL, len, str, sizeof(strbuf))) > 0)
               sprintf (con, "%cs[%d]=%s", delim, len, str);
            else
               sprintf (con, " ");

            if (delim == ' ')
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%-*s ", exlen, con);
            else
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%-s ", con);

         } else {
            if (delim != ' ')
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%c", delim);
            else
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%-*s ", exlen, " ");
         }
      }

      if (ArgusDstUserDataLen > 0) {
         switch (eflag) {
            case ARGUS_ENCODE_ASCII:
               exlen = ArgusDstUserDataLen;
               break;

            case ARGUS_ENCODE_32:
            case ARGUS_ENCODE_64:
               exlen = ArgusDstUserDataLen;
               exlen *= 2;
               break;
         }

         exlen += 8;
         if (ArgusDstUserDataLen > 10) exlen++;

         if (ArgusThisFarStatus & ARGUS_DSTUSRDATA_DSR_STATUS) {
            struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_DSTUSRDATA_DSR_INDEX];

            len = (user->length - 1) * 4;
            len = (len < argus->argus_far.dst.appbytes) ? len : argus->argus_far.dst.appbytes;
            len = len > ArgusDstUserDataLen ? ArgusDstUserDataLen : len;
      
            bzero (strbuf, sizeof(strbuf));
            bzero (conbuf, sizeof(conbuf));
            if ((len = ArgusEncode (&user->data, NULL, len, str, sizeof(strbuf))) > 0)
               sprintf (con, "%cd[%d]=%s", delim, len, str);
            else
               sprintf (con, " ");

            if (delim == ' ')
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%-*s ", exlen, con);
            else
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%-s ", con);

         } else
            if (delim != ' ')
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%c", delim);
            else
               sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%-*s ", exlen, " ");
      }

      sprintf (&RaUserDataStr[strlen(RaUserDataStr)], " ");
   }
}


static char basis_64[] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????";


int
ArgusEncode (const char *ptr, const char *mask, int len, char *str, int slen)
{
   int retn = 0, i;

   switch (eflag) {
      case ARGUS_ENCODE_32:
         sprintf (str, "\"");
         retn = ArgusEncode32(ptr, len, &str[strlen(str)], slen - strlen(str));
         strcat (str, "\"");
         if (mask != NULL) {
            for (i = 0; i < len; i++) {
               if ((mask[i/8] & (0x80 >> (i % 8)))) {
                  str[(1+(2*i))]     = ' ';
                  str[(1+(2*i)) + 1] = ' ';
               }
            }
         }
         break;

      case ARGUS_ENCODE_64:
         sprintf (str, "\"");
         retn = ArgusEncode64(ptr, len, &str[strlen(str)], slen - strlen(str));
         strcat (str, "\"");
         break;

      case ARGUS_ENCODE_ASCII:
         sprintf (str, "\"");
         retn = ArgusEncodeAscii(ptr, len, &str[strlen(str)], slen - strlen(str));
         strcat (str, "\"");
         if (mask != NULL) {
            for (i = 0; i < len; i++)
               if ((mask[i/8] & (0x80 >> (i % 8))))
                  str[(1+i)] = ' ';
         }
         break;

      default:
         ArgusLog (LOG_ERR, "ArgusEncode: error encode method %d unknown", eflag);
         break;
   }


   return (retn);
}

static char basis_16[] = "0123456789ABCDEF";

int
ArgusEncode32 (const char *ptr, int len, char *str, int slen)
{
   int retn = 0, i;
   unsigned char *buf = (unsigned char *) str;
   unsigned newlen;

   if (ptr && ((newlen = (((len + 1) & ~0x01) * 2)) < slen)) {
      for (i = 0; i < len; i++) {
         *buf++ = basis_16[((ptr[i] & 0xF0) >> 4)];
         *buf++ = basis_16[((ptr[i] & 0x0F))];
      }

      retn = newlen;
   }
   
   return (retn);
}


int
ArgusEncode64 (const char *ptr, int len, char *str, int slen)
{
   int retn = 0;
   const unsigned char *in = (const unsigned char *)ptr;
   unsigned char *buf = (unsigned char *) str;
   unsigned char oval;
   unsigned newlen;

   if (ptr && ((newlen = (len + 2) / 3 * 4) < slen)) {
      while (len >= 3) {
          *buf++ = basis_64[in[0] >> 2];
          *buf++ = basis_64[((in[0] << 4) & 0x30) | (in[1] >> 4)];
          *buf++ = basis_64[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
          *buf++ = basis_64[in[2] & 0x3f];
          in += 3;
          len -= 3;
      }
      if (len > 0) {
          *buf++ = basis_64[in[0] >> 2];
          oval = (in[0] << 4) & 0x30;
          if (len > 1) oval |= in[1] >> 4;
          *buf++ = basis_64[oval];
          *buf++ = (len < 2) ? '=' : basis_64[(in[1] << 2) & 0x3c];
          *buf++ = '=';
      }

      retn = newlen;
   }
   
   return (retn);
}

#include <ctype.h>

int
ArgusEncodeAscii (const char *ptr, int len, char *str, int slen)
{
   int retn = 0, newlen = len;
   unsigned char *buf = (unsigned char *) str;

   if (ptr && (len < slen)) {
      while (len > 0) {
         if (isascii((int)*ptr) && isprint((int)*ptr))
            *buf = *ptr;
         else
            *buf = '.';
         buf++;
         ptr++;
         len--;
      }

      if (!(Pflag)) {
         if ((buf = (unsigned char *) strstr (str, "PASS")) != NULL) {
            buf += 5;
            while (((void *)(str + newlen) > (void *) buf) && ((*buf != ' ') && (*buf != '.')))
               *buf++ = 'x';
         }
      }

      retn = newlen;
   }
   
   return (retn);
}
