/* $Id: ArkSystem.cpp,v 1.42 2003/03/26 22:07:08 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/Ark.h>
#include <Ark/ArkSystem.h>

#include <Ark/ArkMath.h>
#include <Ark/ArkConfig.h>
#include <Ark/ArkLoader.h>
#include <Ark/ArkBuffer.h>
#include <Ark/ArkFactory.h>

#ifdef WIN32
#include <windows.h>
#endif

#include <sstream>


#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

/* ERRORS */
#define ERR_CONFIG_FILE(x)	"Unable to find the file %s.", x
#define HOME_NOT_FOUND		"The variable HOME was not found in your environement variables\nHOME is considered as '/'"

const char *Ark_Copyright_Informations =
PACKAGE " " ARK_VERSION
"  Copyright (C) 1999, 2000 The Contributors of the Ark Project.\n"
"\n"
"  This program is free software; you can redistribute it and/or modify\n"
"  it under the terms of the GNU General Public License as published by\n"
"  the Free Software Foundation; either version 2 of the License, or\n"
"  (at your option) any later version.\n"
"\n"
"  This program is distributed in the hope that it will be useful,\n"
"  but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"  GNU General Public License for more details.\n";

namespace Ark
{

   static System *g_System = NULL;

   System *Sys ()
   {
      assert (g_System != NULL);
      return g_System;
   }
   
// ======
// Config callbacks
// ======
   static bool cfgMount (System *sys, Config *self, int nargs, String *args)
   {
      if (nargs != 2)
	 return false;
      
      return sys->FS()->Mount (args[0], args[1]);
   }

   static bool cfgUnMount (System *sys, Config *self, int nargs, String *args)
   {
      if (nargs != 1)
	 return false;

      return sys->FS()->UnMount (args[0]);
   }

   static bool cfgMessage (System *sys, Config *self, int nargs, String *args)
   {
      for (int i = 0; i < nargs; i++)
	 sys->Log ("%s\n", args[i].c_str());

      return true;
   }

// ===========================================================================
// Class used to bind function to exit's() and signal's()
// ===========================================================================
   class ExitCbs
   {
      private:
	 enum
	 {
	    ATEXIT_MAX = 10,
	 };

	 typedef void (*AtExitFunc)();
	 AtExitFunc m_AtExits [ATEXIT_MAX];
	 size_t m_nAtExits;

	 static ExitCbs *m_AtExit;

      protected:
	 void Call ()
	 {
	    for (size_t i = 0; i < m_nAtExits; i++)
	    {
	       if (m_AtExits[i])
		  m_AtExits[i]();
	    }
	 }

#ifdef HAVE_SIGNAL_H
	 /* Called whenever a signal is caught */
	 static RETSIGTYPE OnSignalCb (int sig)
	 {
#ifdef HAVE_PSIGNAL
	    psignal (sig, "Signal caught");
#else
	    fprintf (stderr, "Signal caught: %d\n", sig);
	    fflush (stderr);
#endif

	    if (m_AtExit)
	       m_AtExit->Call();

	    m_AtExit = NULL;
	    abort();
	 }
#endif

	 static void AtExitCb ()
	 {
	    if (m_AtExit)
	       m_AtExit->Call();

	    m_AtExit = NULL;
	 }

      public:
	 ExitCbs ()
	 {
	    m_nAtExits = 0;
	    m_AtExit = this;
	    atexit (AtExitCb);

#ifdef HAVE_SIGNAL_H
	    signal (SIGSEGV, OnSignalCb);
	    signal (SIGINT,  OnSignalCb);
#ifdef SIGHUP
	    signal (SIGHUP,  OnSignalCb);
#endif
#endif
	 }

	 void Add (void (*handler)())
	 {
	    assert (m_nAtExits < ATEXIT_MAX);
	    assert (handler != NULL);

	    m_AtExits[m_nAtExits++] = handler;
	 }
   };

   ExitCbs *ExitCbs::m_AtExit;


// ===========================================================================
// System class implementation
// ===========================================================================
   class SystemImpl : public System
   {
	 Loaders m_Loaders;
	 FileSystem *m_FS;
	 Config m_Config;
	 ExitCbs m_AtExit;
	 bool m_Verbose;
	 FactoryList *m_Factories;
      
      public:
	 SystemImpl ();
      
	 virtual bool IsVerbose () const 
	 { return m_Verbose; }
      
	 virtual void AtExit (void (*cb) ())
	 {m_AtExit.Add (cb);}
      
	 virtual Config *Cfg ()
	 {return &m_Config;}
      
	 virtual FileSystem *FS ()               
	 {return m_FS;}

	 virtual FactoryList *Factories ()               
	 {return m_Factories;}
      
	 virtual Loaders *GetLoaders ()
	 {return &m_Loaders;}
      
	 void ParseArgs (int *argc, char ***argv);
	 void Init (int *argc, char ***argv);
   };


   extern void ark_AddLibLoaders (Loaders *loaders);

   // Initialises the system : create sig handlers, etc..
   SystemImpl::SystemImpl () :
      System (),
      m_FS (0),
      m_Verbose (0)
   {
      g_System = this;
      m_FS = new FileSystem;

      ark_AddLibLoaders (&m_Loaders);
      m_Config.AddFunction
	 ("system::Mount", 2, (Config::CfgCb) &cfgMount, g_System);
      m_Config.AddFunction
	 ("system::UnMount", 1, (Config::CfgCb) &cfgUnMount, g_System);
      m_Config.AddFunction
	 ("system::Message", -1, (Config::CfgCb) &cfgMessage, g_System);
   }

   // Parse arguments & remove them of the command line
   void
   SystemImpl::ParseArgs (int *argc, char ***argv)
   {
      int i, j, k;

      for (i = 0; i < (*argc); i++)
      {
	 if ((*argv)[i][0] == '+')
	 {
	    String str = (const char*)((*argv)[i] + 1);
	    str += ";";

	    std::ostringstream name;
	    name << "(program argument " << i << ")";

	    std::istringstream stream( str );
	    Cfg()->Load(name.str(), stream);

	    (*argv)[i] = 0;
	 }
      }

      for (i = 1; i < *argc; i++)
      {
	 for (k = i; k < *argc; k++)
	 {
	    if ((*argv)[k] != NULL)
	       break;
	 }
	
	 if (k > i)
	 {
	    k -= i;

	    for (j = i + k; j < *argc; j++)
	       (*argv)[j-k] = (*argv)[j];

	    *argc -= k;
	 }
      }
   }

// Return an environment string.
   String System::GetEnv (const String &name)
   {
#ifdef HAVE_GETENV
      const char *env;

      if ((env = getenv (name.c_str())) != NULL)
	 return env;
#endif
      return "";
   }

   // Set an environment string.
   bool System::SetEnv (const String &name, const String &val)
   {
#ifdef HAVE_GETENV

      if (setenv (name.c_str(),val.c_str(), 1) == 0)
	 return true;
#endif
      return false;
   }


   void SystemImpl::Init (int *argc, char ***argv)
   {
      // Init math lookup tables..
      Ark::Math::Init();

      std::cout << "Ark roleplaying kernel version " << ARK_VERSION
		<< "\nDistributed under the terms of the GNU GPL.\n\n";

      String home = GetEnv ("HOME");

      // Mount home directory
      if (home == "")
      {
	 // FIXME: where is the home dir if $HOME is not set ?
	 home = "/";
	 System::Report(E_WARNING, HOME_NOT_FOUND);
      }

#ifdef WIN32
      char system[256];
      GetSystemDirectory (system, 256);
#else
      // FIXME what about relocation ?
      const char *system = SYSCONFDIR"/ark";
#endif

      FS()->Mount ("home", home);
      FS()->Mount ("system", system);

      // Load global config
      Cfg()->LoadSystem("arkglobal.cfg");
      Cfg()->LoadSystem("arkrenderer.cfg");

      // Parse arguments
      m_Verbose = Cfg()->GetInt("system::Verbose", 0) != 0;
      ParseArgs (argc, argv);

      // Load game-specific config
      if (FS()->IsFile("{game}/config/game.cfg"))
	 Cfg()->Load("{game}/config/game.cfg");
      else
	 Report(E_WARNING, ERR_CONFIG_FILE("{game}/config/game.cfg"));

      // "Parse" environment..
      String lang = GetEnv("LANG");
      String::size_type pos = lang.find("_", 0);

      lang = Cfg()->GetStr("system::Lang", lang.substr(0,pos));
      SetEnv("ARK_LANG", lang);
      if (Cfg()->GetStr("game::SupportedLanguages", "").find(lang, 0) ==
	  String::npos)
      {
	 SetEnv("ARK_LANG", Cfg()->GetStr("game::DefaultLanguages", "en"));
      }

      m_Factories = new FactoryList;
   }

   void System::Init (int *argc, char ***argv)
   {
      SystemImpl *si = new SystemImpl ();
      si->Init (argc, argv);
   }

/* namespace Ark */
}

