/*
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "HFWorld.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>  /* For the random stuff */

namespace Ark
{

   static inline short f4 (short a, short b, short c, short d)
   {
      int r = a + b + c + d;

      return short(r >> 2);
   }

   /* Return a pseudo-random number in [-rnd; rnd] */
   static inline short HF_Rnd (int rnd)
   {
      return (short) (rand() % (rnd + 1) - (rnd / 2));
   }

   /** \par Creation of a random HeighField.
    * This is done in three steps :
    *  \arg - Assign random values to each points.
    *  \arg - Add high points (for mounts, by example).
    *  \arg - Remove any roughness (smooth the HF, averaging values)
    */
   void HeightField::Random ()
   {
#define HF_Loop()                       \
  for (x = 0; x < m_SizeX; x += bsize)   \
    for (z = 0; z < m_SizeZ; z += bsize)

      size_t x, z;
      int i, j;
      int bsize, csize;
      int rnd, sz2 = m_SizeX * m_SizeZ;

      memset (m_Y, 0, sz2);

      bsize = m_SizeX;
      csize = m_SizeX/2;
      rnd = 64;

      while(csize > 0)
      {
	 HF_Loop ()
	    {
	       Y (x+csize, z+csize) =
                  f4(Y (x,         z),
                     Y (x + bsize, z),
                     Y (x + bsize, z + bsize),
                     Y (x,         z + bsize))
		  + HF_Rnd (rnd);
	    }

	 HF_Loop ()
	    {
	       Y (x+csize, z) =
                  f4(Y (x,         z),
                     Y (x + bsize, z),
                     Y (x + csize, z + csize),
                     Y (x + csize, z - csize))
		  + HF_Rnd (rnd);
	    }
		
	 HF_Loop ()
	    {
	       Y (x, z+csize) =
                  f4(Y (x,         z),
                     Y (x,         z + bsize),
                     Y (x - csize, z + csize),
                     Y (x + csize, z + csize))
                  + HF_Rnd (rnd);
	    }
		
	 bsize >>= 1;
	 csize >>= 1;

	 if(csize < 16)
	    rnd /= 2;
      }

      // Add 400+ very high point to create mounts when we'll remove
      // roughness.	
      for(j = 0; j < 4; j++)
      {
	 x = rand() % m_SizeX;
	 z = rand() % m_SizeZ;

	 for(i = 0; i < 100; i++)
	 {
	    x += rand() % 17 - 8;
	    z += rand() % 17 - 8;

	    if (x < 0)  x = m_SizeX-1;
	    if (z < 0)  z = m_SizeZ-1;
	    if (x > m_SizeX) x = 0;
	    if (z > m_SizeZ) z = 0;
			
	    Y (x, z) = 100;
	 }
      }
	
      bsize = 1;

      // Remove any roughness from the heightfield by setting the value at
      // each point to the average of those of its neighbours.
      for(j = 0; j < 8; j++)
      {
	 HF_Loop ()
	    {
	       if (Y (x, z) > 255) Y (x, z) = 255;

	       // Compute the average of the height of the points near (x,y)
	       i = Y (x-1, z-1) + Y (x,   z-1) + Y (x+1, z-1)
		 + Y (x-1, z)   + Y (x,   z)   + Y (x+1, z) 
		 + Y (x-1, z+1) + Y (x,   z+1) + Y (x+1, z+1);

	       Y (x, z) = i / 9;
	    }
      }
   }

}
