# -*- coding: utf8 -*-
# Copyright: 2013-2014, Maximiliano Curia <maxy@debian.org>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import argparse
import collections
import deb822
import git
import logging
import os
import re
import shlex
import string
import subprocess

# Own imports
import moo
import util
import uscan


class ActionError(Exception):
    pass


class Action(object):

    '''Abstract class that all actions should subclass.'''

    def __init__(self, arriero, argv):
        '''Basic constructor for the class that calls the necessary functions.
           It follows the Template Pattern.
        '''
        self._arriero = arriero
        self._parser = argparse.ArgumentParser()
        self._set_parse_arguments()
        self._options = self._parser.parse_args(argv)
        self._process_options()

    def _get_packages(self, names):
        '''Expands the list of names and turns it into packages.'''
        packages = set()
        modules = self._arriero.list_modules()
        for name in names:
            if name in modules:
                module = self._arriero.get_module(name)
                packages.update(module.packages)
            else:
                packages.add(name)
        return packages

    def _set_parse_arguments(self):
        '''Add parsing arguments to self._parse.'''

    def _process_options(self):
        '''Process the options after they were parsed.'''

    def run(self):
        '''Execute this action's main goal. Returns True if successful.'''

    def print_status(self):
        '''Print a status report of the run.'''

    # Methods related to parsing arguments

    # --all
    def _add_argument_all(self):
        self._parser.add_argument('-a', '--all', action='store_true')

    def _process_argument_all(self):
        # If --all, we ignore whatever was passed as names
        if self._options.all:
            self._options.names = self._arriero.list_packages()

    # names
    def _add_argument_names(self, strict=True):
        if strict:
            names_choices = set(['']) | self._arriero.list_all()
        else:
            names_choices = None
        self._parser.add_argument('names', nargs='*', default='',
                                  choices=names_choices)

    def _process_argument_names(self):
        if not self._options.names:
            raise argparse.ArgumentError(None, 'No package names received.')


class ActionPackages(Action):

    def _get_name(self):
        return self.__class__.__name__

    def _set_parse_arguments(self):
        self._add_argument_all()
        self._add_argument_names(strict=True)

    def _process_options(self):
        self._process_argument_all()
        self._process_argument_names()

    def run(self):
        '''Method for iterating packages and applying a function.'''
        packages = self._get_packages(self._options.names)
        self._results = collections.defaultdict(set)

        for package_name in packages:
            logging.info('%s: executing %s action.',
                         package_name, self._get_name())
            package = self._arriero.get_package(package_name)
            status = self._package_action(package)
            self._results[status].add(package.name)

        if self._results[moo.ERROR]:
            return False
        return True

    def print_status(self):
        for package in self._results[moo.IGNORE]:
            logging.info('%s: ignored', package)
        for package in self._results[moo.ERROR]:
            logging.error('%s: failed', package)

    def _package_action(self, package):
        '''To override.'''


class ActionFields(ActionPackages):

    '''Common class for List and Exec'''

    available_fields = set([
        'basedir',
        'branch',
        'build_file',
        'changes_file',
        'debian_branch',
        'depends',
        'distribution',
        'export_dir',
        'i',
        'is_dfsg',
        'is_native',
        'is_merged',
        'name',
        'path',
        'pristine_tar_branch',
        'tarball_dir',
        'upstream_branch',
        'upstream_version',
        'vcs_git',
        'version',
    ])
    available_sorts = set(('raw', 'alpha', 'build'))

    def _set_parse_arguments(self):
        super(ActionFields, self)._set_parse_arguments()
        self._parser.add_argument('-f', '--fields', default='name')
        self._parser.add_argument('-s', '--sort',
                                  choices=self.available_sorts,
                                  default='raw')

    def _process_options(self, *formats):
        super(ActionFields, self)._process_options()
        self.fields = util.split(self._options.fields)
        self.field_names = {}
        for field_name in self.fields:
            if field_name not in self.available_fields:
                raise ActionError(
                    'action %s: Field %s not known' %
                        (self._get_name(), field_name))
            self.field_names[field_name] = None

        formatter = string.Formatter()
        for format_string in formats:
            if not format_string:
                continue
            for (_, field_name, _, _) in formatter.parse(format_string):
                if not field_name:
                    continue
                if field_name not in self.available_fields:
                    raise ActionError(
                        'action %s: Field %s not known' %
                            (self.__class__.__name__, field_name))
                self.field_names[field_name] = None

    def _get_field(self, package, field):
        obj = getattr(package, field)
        if hasattr(obj, '__call__'):
            result = obj()
        else:
            result = obj
        return result if isinstance(result, (str, unicode)) else str(result)

    def _get_field_values(self, package, known=None):
        values = []
        by_name = {}
        for field in self.field_names:
            if known and field in known:
                field_value = known[field]
            else:
                field_value = self._get_field(package, field)
            by_name[field] = field_value

        for field in self.fields:
            values.append(by_name[field])

        return values, by_name

    def _get_packages(self, names):
        return self._sorted(super(ActionFields, self)._get_packages(names))

    def run(self):
        '''Method for iterating packages and applying a function.'''
        packages = self._get_packages(self._options.names)
        self._results = collections.defaultdict(set)

        for i, package_name in enumerate(packages):
            logging.info('%s: executing %s action.',
                         package_name, self._get_name())
            package = self._arriero.get_package(package_name)
            status = self._package_action_i(package, i)
            self._results[status].add(package.name)

        if self._results[moo.ERROR]:
            return False
        return True

    # Sort methods
    def raw_sort(self, packages):
        return packages

    def alpha_sort(self, packages):
        return sorted(packages)

    def build_sort(self, packages):
        return self._arriero.sort_buildable(packages)

    def _sorted(self, packages):
        method_name = self._options.sort + '_sort'
        method = getattr(self, method_name)
        return method(packages)


class List(ActionFields):

    '''Query the available packages with formatting.'''

    def _set_parse_arguments(self):
        super(List, self)._set_parse_arguments()
        self._parser.add_argument('-F', '--format', default=None)
        self._parser.add_argument('-e', '--include-empty-results',
                                  action='store_true', dest='empty')

    def _process_options(self):
        super(List, self)._process_options(self._options.format)

    def _is_empty(self, iterable):
        '''Returns True if the iterables has all empty elements.'''
        if not iterable:
            return True
        for value in iterable:
            if value:
                return False
        return True

    def _package_action_i(self, package, i):
        values, by_name = self._get_field_values(package, {'i': i})
        by_name['i'] = i

        if self._options.empty or not self._is_empty(values):
            if self._options.format:
                print self._options.format.format(*values, **by_name)
            else:
                print '\t'.join(values)
        return moo.OK


class Exec(ActionFields):

    '''Run a command on each package.'''

    def _set_parse_arguments(self):
        super(Exec, self)._set_parse_arguments()
        self._parser.add_argument('-x', '--script', action='append')
        self._parser.add_argument('--no-env', action='store_false',
                dest='env')
        self._parser.add_argument('--no-chdir', action='store_false',
                dest='chdir')

    def _process_options(self):
        super(Exec, self)._process_options(*self._options.script)

    def _package_action_i(self, package, i):
        status = moo.OK
        kwargs = {'interactive': True, 'shell': True}

        if self._options.chdir:
            kwargs['cwd'] = package.path

        values, by_name = self._get_field_values(package, {'i': i})

        if self._options.env:
            kwargs['env'] = dict(os.environ, **by_name)

        for script in self._options.script:
            script_formatted = script.format(*values, **by_name)

            try:
                util.log_check_call(script_formatted, **kwargs)
            except subprocess.CalledProcessError as e:
                logging.error('%s: %s', package.name, e)
                status = moo.ERROR
                break
        return status


class Clone(Action):

    '''Clone upstream repositories.'''

    def _set_parse_arguments(self):
        self._add_argument_all()
        self._add_argument_names(strict=False)
        self._parser.add_argument('--basedir', default=None)
        self._parser.add_argument('--upstream-branch', default=None)
        self._parser.add_argument('--debian-branch', default=None)

    def _process_options(self):
        self._process_argument_all()
        self._process_argument_names()

        # Split the names into URLs and packages.
        self._packages = set()
        self._urls = set()
        for name in self._options.names:
            if ':' in name:
                self._urls.add(name)
            else:
                self._packages.add(self._arriero.get_package(name))

    def run(self):
        self._not_ok = set()
        for url in self._urls:
            if not self.url_clone(url):
                self._not_ok.add(url)

        for package in self._packages:
            if not self.package_clone(package):
                self._not_ok.add(package.name)

        # Run status
        if self._not_ok:
            return False
        return True

    def get_remote_heads(self, url):
        heads = set()
        cmd = ['git', 'ls-remote', '--heads', url]
        p = subprocess.Popen(cmd, stdout=subprocess.PIPE)
        for line in p.stdout:
            m = re.search('\srefs/heads/(.*)$', line)
            if m:
                heads.add(m.group(1))
        return heads

    def guess_branches(self, url):
        '''Use git ls-remote to check which branches are there.'''
        heads = self.get_remote_heads(url)
        upstream_branch = 'upstream'
        debian_branch = 'master'
        # Review this: Lucky guess?
        if 'debian' in heads:
            debian_branch = 'debian'
            if 'upstream' not in heads:
                if 'master' in heads:
                    upstream_branch = 'master'
        elif 'unstable' in heads:
            debian_branch = 'unstable'
        pristine_tar = False
        if 'pristine-tar' in heads:
            pristine_tar = True

        return debian_branch, upstream_branch, pristine_tar

    def url_clone(self, url):
        '''Clone a package from the provided url.'''

        # Check if this URL is already configured
        for package_name in self._arriero.list_packages():
            package = self._arriero.get_package(package_name)
            if package.vcs_git == url:
                logging.warning(
                    'The URL %s is already configured by package %s.',
                    package.vcs_git, package.name)
                logging.warning('Switching to cloning from configuration file.')
                self._packages.add(package)
                return True

        # Get basedir for this package
        if self._options.basedir:
            basedir = self._options.basedir
        else:
            basedir = self._arriero.get_config_option('DEFAULT', 'basedir')

        # Guess destdir for this package
        name = os.path.basename(url)
        if name.endswith('.git'):
            name = name[:-4]
        destdir = os.path.join(basedir, name)

        # Guess the branches
        debian_branch, upstream_branch, pristine_tar = self.guess_branches(url)

        self.clone(basedir, destdir, url, debian_branch, upstream_branch,
                   pristine_tar)

        # Obtain package name from control file
        destdir = os.path.expanduser(destdir)
        control_filepath = os.path.join(destdir, 'debian', 'control')
        if not os.path.exists(control_filepath):
            logging.error('Unable to find debian/control while cloning %s', url)
            return False
        control_file = open(control_filepath)
        # Deb822 will parse just the first paragraph, which is ok.
        control = deb822.Deb822(control_file)
        package_name = control['Source']

        if not self._arriero.add_new_package(package_name, url, destdir,
                                             debian_branch, upstream_branch,
                                             pristine_tar):
            logging.error('Clone successful for package not in configuration. '
                          'You will not be able to use arriero with it.')
            return False

        package = self._arriero.get_package(package_name)
        return package.fetch_upstream()

    # TODO: this method should probably be in the package and not here
    def clone(self, basedir, destdir, url, debian_branch, upstream_branch,
              pristine_tar):
        '''Verify the directories for the clone, and clone.'''

        basedir = os.path.expanduser(basedir)
        destdir = os.path.expanduser(destdir)
        util.ensure_path(basedir)
        logging.debug('basedir: %s', basedir)
        logging.debug('destdir: %s', destdir)

        if os.path.exists(destdir):
            logging.error('Cloning %s, directory already exists: %s',
                          url, destdir)
            return False

        # The command line parameters override internal branches
        if self._options.debian_branch:
            debian_branch = self._options.debian_branch
        if self._options.upstream_branch:
            upstream_branch = self._options.upstream_branch

        cmd = ['gbp-clone']
        if debian_branch:
            cmd.append('--debian-branch=%s' % debian_branch)
        if upstream_branch:
            cmd.append('--upstream-branch=%s' % upstream_branch)
        if pristine_tar:
            cmd.append('--pristine-tar')
        else:
            cmd.append('--no-pristine-tar')
        cmd.append(url)

        util.log_check_call(cmd, cwd=basedir)
        logging.info('Successfully cloned %s', url)

    def package_clone(self, package):
        """Clone a package that is already in the config file."""

        #TODO: shouldn't we check if this returned true or false?
        self.clone(package.basedir, package.path, package.vcs_git,
                   package.debian_branch, package.upstream_branch,
                   package.pristine_tar)

        return package.fetch_upstream()


class Build(Action):

    '''Merge and compile the received packages.'''

    def _set_parse_arguments(self):
        self._add_argument_all()
        self._add_argument_names(strict=True)
        self._parser.add_argument('-D', '--distribution', '--dist',
                                  default=None)
        self._parser.add_argument('-A', '--architecture', '--arch',
                                  default=None)
        self._parser.add_argument('-U', '--local-upload', action='store_true',
                                  dest='local_upload')

    def _process_options(self):
        self._process_argument_all()
        self._process_argument_names()

    def _build_package(self, name):
        package = self._arriero.get_package(name)

        # TODO: why is build catching the exception?
        try:
            package.build(
                distribution=self._options.distribution,
                architecture=self._options.architecture)
        except (moo.GitBranchError, moo.GitDiverge,
                subprocess.CalledProcessError) as e:
            logging.error('Build failed for %s: %s', package.name, str(e))
            return False

        if self._options.local_upload:
            try:
                package.local_upload()
            except subprocess.CalledProcessError as e:
                logging.error('Upload failed for %s: %s', package.name, str(e))
                return False

        return True

    def run(self):
        packages = self._get_packages(self._options.names)
        self._error = set()
        self._ignored = set()
        for package in self._arriero.sort_buildable(
                packages, error=self._error, ignored=self._ignored):
            if not self._build_package(package):
                self._error.add(package)

        if self._error:
            return False
        return True

    def print_status(self):
        for package in self._ignored:
            logging.info('%s: ignored', package)
        for package in self._error:
            logging.error('%s: failed', package)


class Upload(ActionPackages):

    def _set_parse_arguments(self):
        self._add_argument_all()
        self._add_argument_names(strict=True)
        self._parser.add_argument('--host', default='local')
        self._parser.add_argument('-f', '--force', action='store_true')

    def _package_action(self, package):
        return package.upload(host=self._options.host,
                              force=self._options.force)


class Pull(ActionPackages):

    def _package_action(self, package):
        status = moo.OK
        if not package.switch_branches(package.debian_branch):
            status = moo.ERROR
        else:
            try:
                package.safe_pull()
                logging.info('Successfully pulled %s.', package.name)
            except (moo.GitDiverge, git.exc.GitCommandError) as e:
                logging.error('Failed to pull %s.\n%s', package.name, e)
                status = moo.ERROR
        return status


class Push(ActionPackages):

    def _package_action(self, package):
        status = moo.OK
        if not package.switch_branches(package.debian_branch):
            status = moo.ERROR
        else:
            if not package.push():
                status = moo.ERROR
        return status


class Release(ActionPackages):

    def _set_parse_arguments(self):
        self._add_argument_all()
        self._add_argument_names(strict=True)
        self._parser.add_argument('-D', '--distribution', default=None)
        self._parser.add_argument('-P', '--pre-release', action='store_true',
                                  dest='pre_release')

    def _package_action(self, package):

        if not package.switch_branches(package.debian_branch):
            logging.error('%s: can\'t switch to branch %s.',
                          package.name, package.debian_branch)
            return moo.ERROR

        package.release(distribution=self._options.distribution,
                        pre_release=self._options.pre_release)
        return moo.OK


class Update(ActionPackages):

    def _package_action(self, package):
        status = moo.OK
        try:
            package.new_upstream_release()
        except (moo.GitDirty, moo.GitDiverge, uscan.UscanError,
                subprocess.CalledProcessError) as e:
            logging.error('Failed to get new upstream release %s.',
                          package.name)
            logging.error('Reason: %s', e)
            status = moo.ERROR
        return status


class Status(ActionPackages):

    def _package_action(self, package):
        print '\n'.join(package.get_status())


class PrepareOverlay(ActionPackages):

    def _set_parse_arguments(self):
        self._add_argument_all()
        self._add_argument_names(strict=True)
        self._parser.add_argument('-b', '--branch', default=None)

    def _package_action(self, package):
        status = moo.OK
        try:
            package.prepare_overlay(overlay_branch=self._options.branch)
        except moo.GitBranchError:
            logging.error('%s: failure while preparing overlay',
                          package.name)
            status = moo.ERROR
        return status


class FetchUpstream(ActionPackages):

    def _package_action(self, package):
        status = moo.OK
        try:
            package.fetch_upstream()
        except moo.GitDirty:
            logging.error('%s: unable to fetch upstream release',
                          package.name)
            status = moo.ERROR
        return status


class CheckIfChanged(ActionPackages):

    def _package_action(self, package):
        status = moo.OK
        if package.is_native():
            return status
        sections = package.changelog.sections
        if len(sections) < 2:
            return status
        current = package.upstream_version
        previous = moo.Version(sections[1].version).upstream
        changes = False

        try:
            package.git.diff(
                package.tag_template('upstream') % {
                    'version': moo.version_to_tag(current)},
                package.tag_template('upstream') % {
                    'version': moo.version_to_tag(previous)},
                quiet=True)
        except git.exc.GitCommandError:
            changes = True

        print "%s: %s" % (package.name,
                          "changed since %s" % previous if changes else
                          "%s = %s" % (current, previous))
        return status


# Dictionary containing the available actions in this module and some help
# about what they are for.

AVAILABLE_ACTIONS = {
    'status': (Status, 'Show the status of the package/s'),
    'update': (Update, 'Get the new upstream release of the package/s'),
    'build': (Build, 'Build the package/s in a pbuilder'),
    'list': (List, 'List package/s with a specific format'),
    'exec': (Exec, 'Run commands for each package/s'),
    'clone': (Clone, 'Obtain the repository for the package/s'),
    'upload': (Upload, 'Upload the package/s'),
    'pull': (Pull, 'Pull new changes to the package/s repositories'),
    'push': (Push, 'Push local changes to the package/s repositories'),
    'release': (Release, 'Change the distribution in the changelog'),
    'overlay': (PrepareOverlay, 'Combine upstream and debian branches'),
    'fetch-upstream': (FetchUpstream, 'Fetch upstream tarball'),
    'check-if-changed': (CheckIfChanged, 'Check changes in the upstream tarball'),
}

# vi:expandtab:softtabstop=4:shiftwidth=4:smarttab
