# -*- coding: utf8 -*-
# Copyright: 2013-2014, Maximiliano Curia <maxy@debian.org>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import collections
import git
import logging
import os
import re
import shlex
import subprocess

# Git buildpackage imports
import gbp.deb.changelog as changelog
try:
    from gbp.deb.source import DebianSource
except ImportError:
    DebianSource = None

# Own imports
import uscan
import util


##
# Constants
##
OK = 'Ok'
IGNORE = 'Ignored'
ERROR = 'Error'
MISS_DEP = 'Missing dependencies'


##
# Data structures
##
Module = collections.namedtuple(
    'Module', ['name', 'packages', 'basedir', 'depends'])


class Version(object):

    '''Represents the different parts of the version string.

    Attributes:
        version : Full version string
        epoch   : Epoch if any, None otherwise
        upstream: Upstream version
        debian  : Debian revision if any, None otherwise
    '''

    def __init__(self, version_string):
        ''' Split version in epoch, upstream_version, debian_revision, returns
            a Version namedtuple.
        '''
        self.version = version_string
        self.epoch = None
        self.debian = None

        upstream = version_string
        if ':' in upstream:
            self.epoch, upstream = upstream.split(':', 1)
        if '-' in upstream:
            upstream, self.debian = upstream.rsplit('-', 1)
        self.upstream = upstream

    @staticmethod
    def from_parts(epoch=None, upstream='0', debian=None):
        version_string = ''
        if epoch:
            version_string += '%s:' % epoch
        version_string += upstream
        if debian:
            version_string += '-%s' % debian
        return Version(version_string)

    def is_native(self):
        return not self.debian

    def new_upstream_version(self, upstream):
        '''Returns a Version for the new upstream release.'''
        # Can be ''
        epoch = '%s:' % self.epoch if self.epoch else ''
        # Let's trust upstream
        # if dfsg_re.match(self.upstream) and dfsg_re.match(upstream):
        #    upstream += '+dfsg'
        debian = '-1' if self.debian else ''

        new_version = epoch + upstream + debian
        return Version(new_version)

    def _bump(self, revision):
        '''Bump a revision so dpkg --compare-version considers it greater.'''
        match = re.match('(.*?)([0-9]*)$', revision)

        numeric_part = 0
        if match.group(2):
            numeric_part = int(match.group(2))
        numeric_part += 1

        return match.group(1) + str(numeric_part)

    def is_pre_release(self):
        '''Check if its a pre release.'''
        return (self.debian and '~' in self.debian) or \
            (not self.debian and '~' in self.upstream)

    def pre_release(self):
        '''Returns a pre release Version from the current one.'''

        def bump_pre(revision):
            if not '~' in revision:
                return revision + '~'
            base, pre = revision.split('~', 1)
            return base + '~' + self._bump(pre)

        if self.is_native():
            new_upstream = bump_pre(self.upstream)
            new_debian = self.debian
        else:
            new_upstream = self.upstream
            new_debian = bump_pre(self.debian)
        return Version.from_parts(self.epoch, new_upstream, new_debian)

    def release(self):
        '''Returns a release Version from a pre release one.'''

        def remove_pre(revision):
            pos = revision.find('~')
            if pos == -1:
                return revision
            return revision[:pos]

        if self.is_native():
            new_upstream = remove_pre(self.upstream)
            new_debian = self.debian
        else:
            new_upstream = self.upstream
            new_debian = remove_pre(self.debian)

        return Version.from_parts(self.epoch, new_upstream, new_debian)

    def __str__(self):
        return self.version

    def __repr__(self):
        return self.version


# Graph node
Node = collections.namedtuple('Node', ['input', 'output'])


class GraphError(Exception):
    pass

# Precompiled regexps
dfsg_re = re.compile(r'(.*)[+.]dfsg(?:.\d+)?$')


##
# Exceptions
##
class GitBranchError(Exception):
    pass


class GitDirty(Exception):
    pass


class GitDiverge(Exception):
    pass


class GitRemoteNotFound(Exception):
    pass


def version_to_tag(version):
    return version.replace('~', '_').replace(':', '%')


def tag_to_version(tag):
    return tag.replace('_', '~').replace('%', ':')


class Package(object):

    def __init__(self, arriero, base):
        self._arriero = arriero
        self._base = base
        self._path = None
        self._vcs_git = None
        self._repo = None
        self._changelog = None
        self._distribution = None
        self._architecture = self._arriero.architecture

    @property
    def name(self):
        return self._base.name

    @property
    def basedir(self):
        return self._base.basedir

    @property
    def depends(self):
        return self._base.depends

    # TODO: read gbp config
    @property
    def upstream_branch(self):
        return self._arriero.get_config_option(self.name, 'upstream-branch')

    # TODO: read gbp config
    @property
    def debian_branch(self):
        return self._arriero.get_config_option(self.name, 'debian-branch')

    @property
    def filter_orig(self):
        value = self._arriero.get_config_option(self.name, 'filter-orig')
        return value.lower() in ('true', 'on', 'yes')

    @property
    def pristine_tar(self):
        value = self._arriero.get_config_option(self.name, 'pristine-tar')
        return value.lower() in ('true', 'on', 'yes')

    # TODO: read gbp config
    @property
    def pristine_tar_branch(self):
        return self._arriero.get_config_option(self.name, 'pristine-tar-branch')

    @property
    def is_merged(self):
        value = self._arriero.get_config_option(self.name, 'is_merged')
        return value.lower() in ('true', 'on', 'yes')

    @property
    def path(self):
        if not self._path:
            value = self._arriero.get_config_option(
                self.name, 'path', raw=True)
            basedir = self.basedir
            package = self.name
            path = value % locals()
            self._path = os.path.expanduser(path)
        return self._path

    @property
    def vcs_git(self):
        if not self._vcs_git:
            value = self._arriero.get_config_option(
                self.name, 'vcs-git', raw=True)
            package = self.name
            self._vcs_git = value % locals()
        return self._vcs_git

    def check_path(self):
        if os.path.isdir(self.path):
            return self.path

    # TODO: read gbp config
    @property
    def tarball_dir(self):
        return os.path.normpath(os.path.join(self.path, '..', 'tarballs'))

    # TODO: read gbp config
    @property
    def export_dir(self):
        return os.path.normpath(os.path.join(self.path, '..', 'build-area'))

    @property
    def repo(self):
        if not self._repo:
            self._repo = git.Repo(self.path)
        return self._repo

    @property
    def git(self):
        return self.repo.git

    @property
    def branch(self):
        return self.repo.active_branch.name

    @property
    def changelog(self, force=False):
        if not self._changelog:
            self.update_changelog()
        return self._changelog

    def update_changelog(self):
        self._changelog = changelog.ChangeLog(
            filename=os.path.join(self.path, 'debian', 'changelog'))

    @property
    def build_file(self, check=True):
        build_file = os.path.join(
            self.export_dir,
            '%s_%s_%s.build' % (self.name,
                                self.noepoch,
                                self._architecture))
        if check and not os.path.exists(build_file):
            return ''
        return build_file

    @property
    def changes_file(self, check=True):
        changes_file = os.path.join(
            self.export_dir,
            '%s_%s_%s.changes' % (self.name,
                                  self.noepoch,
                                  self._architecture))
        if check and not os.path.exists(changes_file):
            return ''
        return changes_file

    @property
    def version(self):
        return self.changelog.version

    @property
    def noepoch(self):
        return self.changelog.noepoch

    @property
    def epoch(self):
        return self.changelog.epoch

    @property
    def upstream_version(self):
        return self.changelog.upstream_version

    @property
    def debian_version(self):
        return self.changelog.debian_version

    @property
    def distribution(self):
        if self._distribution:
            return self._distribution
        return self.changelog['distribution']

    def tag_template(self, name, version):
        '''Returns a tag version correctly formatted according to the name.'''
        value = {'version': version_to_tag(version)}
        # TODO: read gbp config to get the templates
        if name == 'upstream':
            return 'upstream/%(version)s' % value
        if name == 'debian':
            return 'debian/%(version)s' % value

    def uscan(self):
        return uscan.Uscan(self.path, destdir=self.tarball_dir)

    def is_native(self):
        if DebianSource:
            return DebianSource(self.path).is_native()
        return self.changelog.is_native()

    def is_dfsg(self):
        if self.is_native():
            return False
        if dfsg_re.match(self.upstream_version):
            return True
        return False

    def switch_branches(self, branch):
        if self.repo.is_dirty(untracked_files=True):
            logging.warn('%s: branch %s has uncommitted changes.',
                         self.name, branch)
            return False
        try:
            self.git.checkout(branch)
        except git.exc.GitCommandError:
            # Does the branch even exists?
            return False
        # TODO: Check if really needed
        self._changelog = None
        return True

    def branch_to_ref(self, branch):
        return 'refs/heads/%s' % (branch,)

    def ahead_behind_ref(self, ref1, ref2):
        out = self.git.rev_list('--left-right', '--count',
                                '%s...%s' % (ref1, ref2))
        return map(int, out.split())

    def ahead_behind(self, local=None, remote=None):
        # Hacky check between two refs
        # returns two numbers, commits ahead and commits behind
        if not local:
            local = self.branch
        ref1 = self.branch_to_ref(local)
        if not remote:
            try:
                remote_name = self.git.config('branch.%s.remote' % local)
            except git.exc.GitCommandError:
                remote_name = 'origin'
            try:
                remote_branch = self.git.config('branch.%s.merge' % local)
            except git.exc.GitCommandError:
                remote_branch = local
            prefix = 'refs/heads/'
            if remote_branch.startswith(prefix):
                remote_branch = remote_branch[len(prefix):]

            remote = '%s/%s' % (remote_name, remote_branch)
            # Check if the remote actually exists
            git_remote = self.repo.remote(remote_name)
            if not git_remote in self.repo.remotes:
                raise(GitRemoteNotFound('Remote (%s) not found' %
                                        remote_name))
            for ref in git_remote.refs:
                if ref.name == remote:
                    break
            else:
                raise(
                    GitRemoteNotFound(
                        'Could not find branch (%s) in remote (%s)' %
                        (remote_branch, remote_name)))
        ref2 = 'refs/remotes/%s' % (remote,)
        return self.ahead_behind_ref(ref1, ref2)

    # TODO: DEADCODE
    def merge_if_not_diverged(self, branch):
        ahead, behind = self.ahead_behind(
            ref1=self.branch_to_ref(self.branch),
            ref2=self.branch_to_ref(branch))
        if ahead and behind:
            raise(
                GitDiverge(
                    'Current branch (%s) has diverged with (%s)' %
                    (self.branch, branch)))
        elif not ahead and behind:
            self.git.merge('-m',
                           'Merge with %s', (branch,))
            return True
        return False

    def prepare_overlay(self, overlay_branch=None):
        '''Combines the upstream and debian branches into one working tree.

        If the debian_branch of the package already contains the upstream source
        (stored in self.merge attribute), then that branch is checked out,
        nothing else happens.

        If the package is a native package, the debian branch is checked out,
        nothing else happens.

        Args:
            overlay_branch: if received, the result is checked into a new branch
                instead of a dettached HEAD.
        '''

        # Try to switch to debian branch. If this fails, there are probably
        # uncommitted changes.
        logging.debug('%s: switching to branch %s for overlay creation.',
                      self.name, self.debian_branch)
        if not self.switch_branches(self.debian_branch):
            raise GitBranchError('Could not change to branch %s' %
                                 self.debian_branch)
        # Create working branch, if the user so requested
        if overlay_branch:
            logging.debug('%s: creating new overlay branch %s.',
                          self.name, overlay_branch)
            try:
                self.git.checkout('-b', overlay_branch)
            except git.exc.GitCommandError as e:
                logging.error('%s: %s', self.name, str(e))
                raise GitBranchError(str(e))

        if self.is_merged:
            logging.info('%s: skipping overlay creation, already merged.',
                         self.name)
            return

        if self.is_native():
            logging.info('%s: native package, no overlay needed.',
                         self.name)
            return

        # Checkout the upstream tree into the current branch, then reset the
        # index so that the files are there, but not scheduled to be committed.
        logging.info('%s: copying tree from latest upstream tag.', self.name)
        self.git.checkout(self.tag_template('upstream', self.upstream_version),
                          '.', ours=True)
        #TODO: add :!.gitignore, only if the git version supports it.
        self.git.reset()
        logging.info('%s: overlay created, branch needs to be manually cleaned',
                     self.name)

    def build(self, ignore_branch=False, distribution=None, architecture=None):

        if not ignore_branch and self.debian_branch != self.branch:
            if not self.switch_branches(self.debian_branch):
                raise(GitBranchError('Could not change to branch %s' % (
                    self.debian_branch,)))

        if distribution:
            self._distribution = distribution
        if architecture:
            self._architecture = architecture

        self._arriero.ensure_builder_image(self.distribution,
                                           self._architecture)

        # TODO: add this as options.
        cmd = [
            'git-buildpackage',
            '--git-pbuilder',
            '--git-export-dir=%s' % self.export_dir,
            '--git-arch=%s' % self._architecture,
            '--git-dist=%s' % self.distribution.lower(),
            '--git-verbose']
        if self.pristine_tar:
            cmd.append('--git-pristine-tar')
            # If already generated, use that
            cmd.append('--git-tarball-dir=%s' % self.export_dir)
        else:
            cmd.append('--git-tarball-dir=%s' % self.tarball_dir)
        if not ignore_branch:
            cmd.append('--git-debian-branch=%s' % self.debian_branch)
        if not self.is_merged and not self.is_native():
            cmd.append('--git-overlay')
        util.log_check_call(cmd, cwd=self.path, interactive=True)

        lintian_cmd = [
            'lintian', '-I', '--pedantic', '--show-overrides',
            self.changes_file]
        util.log_check_call(lintian_cmd, cwd=self.path)

        # Switch to work branch
        if not ignore_branch:
            self.switch_branches(self.debian_branch)

    def release(self, distribution=None, pre_release=False, ignore_branch=False):
        if not ignore_branch and self.debian_branch != self.branch:
            if not self.switch_branches(self.debian_branch):
                raise(GitBranchError('Could not change to branch %s' % (
                    self.debian_branch,)))

        cmd = ['dch']
        if self.distribution.lower() == 'unreleased':
            version = Version(self.version)
            if pre_release:
                if not version.is_pre_release():
                    cmd.append('-b')
                new_version = version.pre_release()
                cmd.extend(['-v', str(new_version)])
            elif version.is_pre_release():
                new_version = version.release()
                # First we need to remove ~ part
                util.log_check_call(
                    ['dch', '--release-heuristic', 'changelog',
                        '-v', str(new_version), ''],
                    cwd=self.path)
                cmd.append('-r')
            else:
                cmd.append('-r')
        else:
            # TODO(marga): this shouldn't do anything if there are no changes
            cmd.append('-r')

        if distribution:
            cmd.append('-D')
            cmd.append(distribution)
        cmd.append('')
        util.log_check_call(cmd, cwd=self.path)
        self.update_changelog()

        # If there were any changes to the changelog, commit them
        if self.repo.index.diff(None, 'debian/changelog'):
            if pre_release:
                msg = 'Pre-release %s' % self.version
            else:
                msg = 'Release to %s' % self.distribution
            self.git.commit('debian/changelog', '-m', msg)

    def local_upload(self):
        self.upload('local')

    def upload(self, host, force=False):
        upload_command = self._arriero.get_config_option(
            self.name, 'upload-command', raw=True)

        if not upload_command:
            return ERROR
        if not self.changes_file:
            return IGNORE

        cmd_variables = {
            'changes_file': self.changes_file,
            'package': self.name,
            'version': self.version,
            'distribution': self.distribution,
            'host': host,
        }

        try:
            full_command = upload_command % cmd_variables
        except (ValueError, KeyError) as e:
            logging.error('%s: unable to format upload-command: %s',
                self.name, upload_command)
            logging.error('%s: %s' % (e.__class__.__name__, e.message))
            return ERROR

        util.log_check_call(full_command, shell=True)

    def push(self):
        logging.debug('Pushing %s', self.name)
        try:
            self.git.push()
            return True
        except git.exc.GitCommandError as e:
            logging.error('Failed to push %s.\n%s', self.name, e)
            return False

    def safe_pull(self):
        # update to check ahead/behind
        # TODO: fix ugly hack
        try:
            remote_name = self.git.config('branch.%s.remote' % (self.branch,))
        except git.exc.GitCommandError:
            remote_name = 'origin'

        self.git.fetch(remote_name)
        try:
            ahead, behind = self.ahead_behind()
            if ahead > 0 and behind > 0:
                raise(GitDiverge('Needs to merge with head.'))
        except GitRemoteNotFound as e:
            logging.info('Branch not associated with a remote: %s' %
                         e.message)
            return False

        # TODO: return True if there were changes
        self.git.pull()

    def get_new_version(self, upstream):
        ''' Obtains a new version.
        '''
        old = Version(self.version)
        new = old.new_upstream_version(upstream)
        return new.version

    def new_dfsg_version(self, upstream_version):
        branch = self.branch
        rules = os.path.join(self.path, 'debian', 'rules')
        dfsg_version = upstream_version + '+dfsg'
        self.git.checkout(self.tag_template('upstream'), upstream_version)
        self.git.checkout('heads/%s' % branch, '--', 'debian')
        self.git.reset()
        util.log_check_call(
            ['fakeroot', rules, 'prune-nonfree'],
            cwd=self.path)
        util.log_check_call(['rm', '-rf', 'debian'], cwd=self.path)
        if (not self.repo.is_dirty(untracked_files=True)):
            # No changes made by the prune-nonfree call, we are free \o/
            self.switch_branches(branch)
            return upstream_version
        self.git.commit('-a', '-m', 'DFSG version %s' % (dfsg_version,))
        self.repo.create_tag(self.tag_template('upstream'), dfsg_version)
        self.switch_branches(branch)
        return dfsg_version

    def fetch_upstream(self):
        '''Fetch upstream tarball when there is no upstream_branch.'''
        if not self.upstream_branch:
            return True
        if self.is_native():
            return True

        if (self.upstream_branch not in self.repo.branches):
            logging.debug('Creating upstream branch for %s.', self.name)
            original_branch = self.branch

            # Create upstream branch
            self.git.checkout('--orphan', self.upstream_branch)
            self.git.rm('-rf', '.')
            self.git.commit('--allow-empty', '-m', 'Upstream branch')
            self.switch_branches(original_branch)

        result = self.get_upstream_release(current=True)
        return result

    def get_upstream_release(self, ignore_branch=False, local_only=False,
                             current=False):
        if self.is_native():
            return False

        logging.debug('Fetching upstream tarball for %s.', self.name)

        if self.repo.is_dirty(untracked_files=True):
            raise(GitDirty('Uncommited changes'))

        if not ignore_branch and self.debian_branch != self.branch:
            if not self.switch_branches(self.debian_branch):
                raise(GitDirty('Could not switch branches'))

        if not local_only:
            self.safe_pull()
            # Need to reread changelog
            self.update_changelog()

        try:
            pkg_scan = self.uscan()
            pkg_scan.scan(download=True, force_download=current)
        except uscan.UscanError as e:
            logging.error(
                'Could not download upstream tarball for %s: %s',
                self.name, str(e))
            return False

        # ignore uptodate status if we forced the download
        if not current and pkg_scan.uptodate:
            return False
        upstream_version = pkg_scan.uversion

        tag = self.tag_template('upstream', upstream_version)
        if tag in self.repo.tags:
            return False

        cmd = ['git-import-orig', '--upstream-version=%s' % upstream_version]

        # TODO: this should not be duplicated
        if self.debian_branch:
            cmd.append('--debian-branch=%s' % self.debian_branch)
        if self.upstream_branch:
            cmd.append('--upstream-branch=%s' % self.upstream_branch)
        if self.pristine_tar:
            cmd.append('--pristine-tar')
            if self.filter_orig:
                cmd.append('--filter-pristine-tar')
        else:
            cmd.append('--no-pristine-tar')

        cmd.append(pkg_scan.tarball)
        util.log_check_call(cmd, cwd=self.path)

        if self.is_dfsg():
            return self.new_dfsg_version(upstream_version)
        return upstream_version

    def new_upstream_release(self, ignore_branch=False, local_only=False):

        logging.debug('Trying to make new upstream release for %s', self.name)

        if not self.check_path():
            logging.error('%s doesn\'t exist', self.path)
            return False

        upstream_version = self.get_upstream_release(ignore_branch, local_only,
                                                     current=False)
        if not upstream_version:
            return False

        version = self.get_new_version(upstream_version)

        cmd = ['dch', '-p', '-v', version, 'New upstream release.']
        util.log_check_call(cmd, cwd=self.path)
        # Just changed the changelog, but its probably not going to be used
        # anymore.
        self._changelog = None

        self.git.add(os.path.join('debian', 'changelog'))
        self.git.commit('-m', 'New upstream release.')

        return True

    def get_status(self):
        status = []
        status.append('Package: %s' % self.name)
        status.append('Directory: %s' % self.path)

        if not self.check_path():
            status.append('Status: Error, %s doesn\'t exist' % self.path)
            return status

        status.extend(self.get_branch_status())
        return status

    def get_branch_status(self):
        status = []
        status.append('Branch: %s' % self.branch)

        error = False

        branches = {'debian': self.debian_branch}
        if not self.is_native():
            branches['upstream'] = self.upstream_branch
        if self.pristine_tar:
            branches['pristine-tar'] = self.pristine_tar_branch

        for k, v in branches.iteritems():
            if v not in self.repo.heads:
                status.append('Status: Error, Missing %s branch: %s' % (k, v))
                error = True
        if error:
            return status

        # Check if there are non commited changes
        dirty = self.repo.is_dirty(untracked_files=True)
        if dirty:
            status.append('Status: Uncommited changes')
            status.append(self.git.status())

        # Check current version (only in debian)
        if self.branch not in self.debian_branch:
            if dirty:
                status.append('Status: Can\'t check version, dirty branch')
                return status
            # If it's not dirty we can change branches, right?
            if not self.switch_branches(self.debian_branch):
                status.append(
                    'Status: Error, change to branch %s failed' %
                    self.debian_branch)
                return status
            else:
                status.append('Switched to branch: %s' % self.branch)

        # Check upstream tag with current version
        # TODO:
        #status.append( self.tag_template('upstream') % {
        #    'version': self.upstream_version}
        # Check if released
        # released = (self.distribution.lower() != 'unreleased')
        status.append('Distribution: %s' % self.distribution)
        # if released, check if tagged in debian repo
        # TODO:
        #status.append( self.tag_template('debian') % {
        #    'version': self.noepoch}

        # TODO: it should be possible to obtain the state without actually
        # changing branches
        status.extend(self.get_uscan_status())
        status.extend(self.get_repo_status())
        status.extend(self.get_build_status())

        return status

    def get_uscan_status(self):
        '''Check if the package is up to date with uscan.'''

        # Native packages don't have uscan status
        if self.is_native():
            return []

        try:
            uscan_status = self.uscan()
            uscan_status.scan(download=False)
        except uscan.UscanError as e:
            return ['Status: Error while running uscan: %s' % str(e)]

        status = []
        if not uscan_status.uptodate:
            status.append(
                'Status: New upstream release available. Local version: %s. '
                'Upstream version: %s.\nStatus: Source URL: %s' %
                (uscan_status.version, uscan_status.uversion,
                 uscan_status.url))
            if uscan_status.tarball:
                status.append('Source: Already downloaded in %s' %
                              uscan_status.tarball)
        return status

    def get_repo_status(self):
        # Check if up to date with git repo
        status = []
        remote_name = self.git.config('branch.%s.remote' % self.branch)
        self.git.fetch(remote_name)
        try:
            ahead, behind = self.ahead_behind()
            if behind > 0:
                status.append(
                    'Status: Remote changes commited (%d), pull them' %
                    behind)
            if ahead > 0:
                status.append(
                    'Status: Local changes commited (%d)' % ahead)
        except GitRemoteNotFound:
            status.append('Status: Branch not associated with a remote')
        return status

    def get_build_status(self):
        status = []
        if not self.changes_file:
            status.append('Status: The package has not been built')

        signed = False
        if self.changes_file:
            ret = util.quiet('gpg', '--verify', self.changes_file)
            signed = (ret == 0)

        released = (self.distribution.lower() != 'unreleased')
        if signed and not released:
            status.append('Status: Package signed for %s' % self.distribution)
        # Check lintian
        # Check for reported errors
        # Check errors reported upstream

        # tarball_dir
        # export_dir
        # self.changelog.version
        # self.changelog.noepoch
        return status


# vi:expandtab:softtabstop=4:shiftwidth=4:smarttab
