#!/usr/bin/env python
# -*- coding: utf8 -*-
# Copyright: 2013-2014, Maximiliano Curia <maxy@debian.org>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import logging
import os
import re
import subprocess
import select


def quiet(*argv, **kwargs):
    '''Make an OS call without generating any output.'''
    devnull = open(os.devnull, 'r+')
    return subprocess.call(argv, stdin=devnull, stdout=devnull,
                           stderr=devnull, **kwargs)


def split(values):
    '''Split a comma separated string of values into a list.'''
    return re.split(r'[\s,]+', values)


def ensure_path(path):
    '''Create path if it doesn't exist.'''
    if not os.path.exists(path):
        logging.info('Creating path: %s', path)
        os.makedirs(path)


def log_check_call(cmd, interactive=False, **kwargs):
    '''Equivalent to check_call, but logging before and after.'''
    str_cmd = cmd if isinstance(cmd, (str, unicode)) else ' '.join(cmd)
    logging.debug('Executing: %s. %s', str_cmd, kwargs)
    subprocess.check_call(cmd, **kwargs)
    return

    # TODO: make this work
    # Current problems:
    # * Pressing Ctrl-C interrupts arriero, but not the running process
    # * Some processes are interactive but their output should still be logged
    #   (git-build-package). And for some we only want errors.
    if not interactive:
        call = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                                stderr=subprocess.PIPE, **kwargs)
        input_fds = [call.stdout, call.stderr]
        levels = {call.stdout: logging.info, call.stderr: logging.error}
    else:
        call = subprocess.Popen(cmd, stderr=subprocess.PIPE, **kwargs)
        input_fds = [call.stderr]
        levels = {call.stderr: logging.error}

    def check_io():
        ready_to_read = select.select(input_fds, [], [], 1000)[0]
        for io in ready_to_read:
            line = io.readline()
            if line: levels[io](line[:1])

    # keep checking stdout/stderr until the child exits
    while call.poll() is None:
        check_io()
    check_io() # check again to catch anything after the process exits

    if call.returncode:
        raise subprocess.CalledProcessError(call.returncode, cmd)


# vi:expandtab:softtabstop=4:shiftwidth=4:smarttab
