/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	CD information class
 *
 *	by Tony Sideris	(05:01PM Feb 11, 2002)
 *================================================*/
#include "arson.h"

#include <stdio.h>

#include <qfile.h>

#include "cdinfo.h"
#include "konfig.h"

#ifndef ARSONDBG
#	define dump()
#endif

/*========================================================*/
/*	Format a track filename (basename) with this track
 *	information using the given format.
 *
 *	Example, "%a - %t" becomes:
 *	"Black Sabbath - Snowblind"
 *
 *	%a - Artist
 *	%c - CD Title
 *	%t - Track Title
 *	%n - Track Number
 *	%N - Track Number (2 characters wide, '0' filled)
 *	%z - Zero-based Track Number
 *	%N - Zero-based Track Number (2 characters wide, '0' filled)
 *========================================================*/

bool ArsonCdInfoBase::Track::formatTrackName (const ArsonCdInfoBase *parent,
	const QString &fmt, QString &result) const
{
	result = fmt;

	replaceFmt("%a", parent->variousArtistDisk()
		? m_artist : parent->artist(), result);
	replaceFmt("%c", parent->title(), result);
	replaceFmt("%t", m_title, result);

	//	Coming soon: roman numerals, hex, octal, & binary...
	replaceFmt("%n", QString::number(trackNo()), result);
	replaceFmt("%z", QString::number(trackNo() - 1), result);
	replaceFmt("%N",
		QString().sprintf("%02d", trackNo()), result);
	replaceFmt("%Z",
		QString().sprintf("%02d", trackNo() - 1), result);

	return true;
}
void ArsonCdInfoBase::Track::replaceFmt (const char *fmt,
	const QString &with, QString &target) const
{
	int index = target.find(fmt);

	while (index != -1)
	{
		target.replace(index, 2, with);
		index = target.find(fmt, index + 1);
	}
}

/*========================================================*/

ArsonCdInfoBase::ArsonCdInfoBase (void)
	: m_leadout(0)
{
	//	Nothing...
}

#if 0
ArsonCdInfoBase::ArsonCdInfoBase (const ArsonCdInfoBase &info)
	: m_leadout(info.m_leadout)
{
	(*this) = info;
}

ArsonCdInfoBase &ArsonCdInfoBase::operator= (const ArsonCdInfoBase &info)
{
#define CPS(val)		if (val.isEmpty())val=info.val
#define CPN(val)		if (!val)val=info.val

	CPS(m_title);
	CPS(m_artist);
	CPS(m_year);
	CPS(m_genre);
	CPS(m_categ);
	CPS(m_cdIndexID);
	CPS(m_cddbID);
	CPS(m_ext);
	CPS(m_playorder);

	CPN(m_leadout);

	if (m_tracks.count() == info.m_tracks.count())
	{
		for (int index = 0, size = m_tracks.count(); index < size; ++index)
			track(index) = info.track(index);

//		Trace("manual copy\n");
	}
	else
	{
/*
		Trace("operator= (%d %d)\n",
			m_tracks.count(),
			info.m_tracks.count());
*/
		m_tracks = info.m_tracks;
	}

//	dump();
	
	return *this;

#undef CPS
#undef CPN
}
#endif	//	0
/*========================================================*/

ArsonCdInfoBase::Track::Track (void)
	: m_offset(0),
	m_trackNo(0)
{
	//	Nothing...
}

/*
ArsonCdInfoBase::Track::Track (const Track &track)
	: m_trackNo(track.m_trackNo),
	m_offset(track.m_offset)
{
	(*this) = track;
}

ArsonCdInfoBase::Track &ArsonCdInfoBase::Track::operator= (const Track &track)
{
#define CPS(s)		if(s.isEmpty()) s=track.s
#define CPN(n)		if(!n) n=track.n

	CPS(m_artist);
	CPS(m_title);
	CPS(m_ext);
	CPN(m_offset);
	CPN(m_trackNo);

	if (!m_time.totalSeconds())
		m_time = track.m_time;

	return *this;
}
*/
/*========================================================*/

bool ArsonCdInfoBase::formatTrackName (int track,
	const QString &fmt, QString &result) const
{
	if (track == -1)
		return Track().formatTrackName(
			this, fmt, result);

	/*	Do not assume that m_tracks[1] == TRACK_1 ...
	 *	it PROBABLY does, but do not assume it, the
	 *	track number is given in the cdindex data,
	 *	use that.
	 */
	for (int index = 0; index < m_tracks.count(); index++)
		if (track == m_tracks[index].trackNo())
			return m_tracks[index].formatTrackName(this, fmt, result);

	return false;
}

/*========================================================*/

ArsonCdInfoBase::Track &ArsonCdInfoBase::addTrack (const ArsonCdInfoBase::Track &track)
{
	return *(m_tracks.append(track));
}

/*========================================================*/

void ArsonCdInfoBase::clear (void)
{
	m_tracks.clear();

	m_title = m_artist = m_cddbID =
		m_cdIndexID = m_year =
		m_genre = m_categ =
		QString::null;

	m_leadout = 0;
}

/*========================================================*/

uint ArsonCdInfoBase::totalSeconds (void) const
{
	uint total, index;

	for (total = index = 0; index < trackCount(); ++index)
		total += track(index).time().totalSeconds();

	return total;
}

/*========================================================*/
#ifdef ARSONDBG
void ArsonCdInfoBase::dump (void) const
{
#define STR(s)		(s).latin1()
	
	Trace(
		"ArsonCdInfoBase: %s / %s\n"
		"IDs: %s / %s\n"
		"Year/Genre/Category: %s/%s/%s\n"
		"Extra: %s\n"
		"Playorder: %s\n"
		"Leadout: %u\n\nTracks:\n",
		STR(m_title), STR(m_artist),
		STR(m_cddbID), STR(m_cdIndexID),
		STR(m_year), STR(m_genre), STR(m_categ),
		STR(m_ext),
		STR(m_playorder),
		m_leadout);

	for (int index = 0; index < m_tracks.count(); ++index)
	{
		const Track &track = m_tracks[index];

		Trace(
			" %d: %s / %s\n"
			"   Extra: %s\n"
			"   Offset: %u\n"
			"   Time: %s\n\n",
			track.trackNo(), STR(track.title()), STR(track.artist(this)),
			STR(track.comment()),
			track.offset(),
			STR(track.time().display()));
	}
}
#endif	//	ARSONDBG
/*========================================================*/

void ArsonCdInfo::merge (const ArsonCdInfo &info)
{
	const QString cddb (info.cddbID());
	const QString cdi (info.cdIndexID());
	ArsonCdInfoBase *that = this;

	(*that) = info;

	setCdIndexID(cdi);
	setCddbID(cddb);
}

/*========================================================*/
