/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Cd info editor dialog
 *
 *	by Tony Sideris	(10:24PM Mar 19, 2002)
 *================================================*/
#include "arson.h"

#include <qmultilineedit.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>

#include <klineeditdlg.h>
#include <klocale.h>

#ifdef ARSONDBG
#include <stdlib.h>
#endif	//	ARSONDBG

#include "cdinfoeditor.h"
#include "listwnd.h"
#include "konfig.h"
#include "usercfg.h"
#include "lookup.h"

#define DEFAULT_TRACK_TITLE(index)	\
	(QString().sprintf("track-%02d", (index) + 1))

#define GENRERC		"genres.xml"

enum {
	dbNone = 0,
	dbFreedb,
	dbCdindex,
	_db_max
};

/*========================================================*/

ArsonCdInfoEditor::ArsonCdInfoEditor (QWidget *parent, const ArsonCdInfo &info)
	: ArsonCdInfoDlgBase(parent, NULL, TRUE)
{
	int index;
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Track"), 10),
		ArsonListHeader(i18n("Title"), 35),
		ArsonListHeader(i18n("Artist"), 25),
		ArsonListHeader(i18n("Comment"), 30),
	};

	tracks->setListHeaders(hdrs, ARRSIZE(hdrs));

	tracks->setItemsRenameable(true);
	tracks->setRenameable(0, false);
	tracks->setRenameable(1, true);
	tracks->setRenameable(2, false);
	tracks->setRenameable(3, true);

	tracks->setSelectionMode(QListView::Single);

	comment->setFixedHeight(artist->height() * 3);

	QObject::connect(ok, SIGNAL(clicked()),
		this, SLOT(accept()));

	m_info.merge(info);
	
	fillGenres();
	applyInfo(info);
}

/*========================================================*/

class genreParser : public ArsonUserCfg
{
	typedef QMap<QString,int> INDEXMAP;

public:
	genreParser (const char **categs, int count, ArsonCdInfoEditor *pd)
		: m_curIndex(-1), m_pDlg(pd)
	{
		for (int index = 0; index < count; ++index)
			m_indices.insert(categs[index], index);
	}

	virtual bool startElement (const QString &ns, const QString &local,
		const QString &name, const QXmlAttributes &attr)
	{
		if (name == "genre")
		{
			int link = -1, index;
			const QString genre = attr.value("name");
			const QString categ = attr.value("cddb");
			INDEXMAP::ConstIterator it = m_indices.find(categ);

			if (it != m_indices.end())
				link = it.data();

			if ((index = m_pDlg->addGenre(genre, link)) != -1)
				m_curIndex = index;
		}

		return true;
	}

	int currentIndex (void) const { return m_curIndex; }
	
private:
	ArsonCdInfoEditor *m_pDlg;
	INDEXMAP m_indices;
	int m_curIndex;
};

/*========================================================*/

int ArsonCdInfoEditor::addGenre (const QString &gen, int link)
{
	const int index = genre->count();

	genre->insertItem(gen);

	if (link != -1)
		m_links.insert(index, link);

	return (m_info.genre() == gen)
		? index : -1;
}

/*========================================================*/

void ArsonCdInfoEditor::fillGenres (void)
{
	const char *cats[] = {
		"misc",
		"blues",
		"classical",
		"country",
		"data",
		"jazz",
		"newage",
		"reggae",
		"rock",
		"soundtrack",
	};
	genreParser parser (cats, ARRSIZE(cats), this);
	parser.readConfig(GENRERC);

	for (int index = 0; index < ARRSIZE(cats); ++index)
		categ->insertItem(cats[index]);

	if (parser.currentIndex() != -1)
	{
		genre->setCurrentItem(parser.currentIndex());
		genre_changed(parser.currentIndex());
	}
}
	
/*========================================================*/
/*	Updates the dialog controls to reflect
 *	the given cd info object.
 *========================================================*/

void ArsonCdInfoEditor::applyInfo (const ArsonCdInfo &info)
{
	uint index;
	QListViewItem *pAfter = NULL;
	const bool various = info.variousArtistDisk();

	multi->setChecked(various);
	multi_artist_toggled(various);

	if (!various)
		artist->setText(info.artist());
	
	genre->setEditText(info.genre());
	year->setValue(info.year().toInt());
	title->setText(info.title());

	//	Reset the track list box
	tracks->setRenameable(2, various);
	tracks->clear();

	//	Fill the track list
	for (index = 0; index < info.trackCount(); ++index)
	{
		const ArsonCdInfo::Track &track = info.track(index);
		const QString title = track.title();

		pAfter = new QListViewItem(tracks, pAfter,
			QString::number(index + 1),
			title.isEmpty() ? DEFAULT_TRACK_TITLE(index) : title,
			track.artist(&info),
			track.comment());
	}

	comment->setText(info.comment());

	if (QListBoxItem *pi = categ->listBox()->findItem(info.categ()))
		categ->setCurrentItem(categ->listBox()->index(pi));
}

/*========================================================*/

void ArsonCdInfoEditor::multi_artist_toggled (bool multi)
{
	tracks->setRenameable(2, multi);
}

/*========================================================*/

void ArsonCdInfoEditor::genre_changed (int index)
{
	LINKMAP::ConstIterator it = m_links.find(index);

	if (it != m_links.end())
		categ->setCurrentItem(it.data());
}

/*========================================================*/

void ArsonCdInfoEditor::reset_clicked (void)
{
	applyInfo(m_info);
}

/*========================================================*/

bool ArsonCdInfoEditor::verify (void)
{
	const bool various = multi->isChecked();
	const QString strArtist = artist->text();
	const QString strTitle = title->text();

	if (!various && strArtist.isEmpty())
		arsonErrorMsg(i18n("Please specify an artist."));
	else if (strTitle.isEmpty())
		arsonErrorMsg(i18n("Please specify a title."));
	else
		return true;

	return false;
}

/*========================================================*/

void ArsonCdInfoEditor::submit_clicked (void)
{
	if (verify())
	{
		accept();

		submitInfo(m_info);
	}
}

/*========================================================*/

void ArsonCdInfoEditor::accept (void)
{
	if (verify())
	{
		ArsonCdInfo temp;
		const bool various = multi->isChecked();
		const QString strArtist = artist->text();
		const QString strTitle = title->text();

		temp.merge(m_info);
		
		temp.setArtist(various ? QString::null : strArtist);
		temp.setYear(QString::number(year->value()));
		temp.setGenre(genre->currentText());
		temp.setTitle(strTitle);
		temp.setComment(comment->text());

		for (uint index = 0; index < temp.trackCount(); ++index)
		{
			ArsonCdInfo::Track &track = temp.track(index);

			if (QListViewItem *pi = tracks->itemAtIndex(index))
			{
				const QString title = pi->text(1);
				track.setTitle(title.isEmpty()
					? DEFAULT_TRACK_TITLE(index) : title);

				if (various)
					track.setArtist(pi->text(2));

				track.setComment(pi->text(3));
			}
		}

		m_info.merge(temp);
		ArsonCdInfoDlgBase::accept();
	}
}

/*========================================================*/
/*	Submit CD info to online databases
 *========================================================*/
/*
class freedbSocket : public ArsonHttpPostSocket
{
public:
	freedbSocket (const QString &host, short port)
		: ArsonHttpPostSocket(host, port)
	{
		//	Nothing...
	}

private:
	virtual void response (int code, const QString &msg)
	{
		if (code != 200)
			arsonErrorMsg(
				i18n("Freedb server returned unknown or unwanted code %1: %2")
				.arg(code).arg(msg));
	}

	virtual QString path (void) const
	{
		return QString("/~cddb/submit.cgi");
	}
};
*/
bool ArsonCdInfoEditor::submitInfo (const ArsonCdInfo &info)
{
	ArsonSubmitDlg dlg (info, ACONFIG.ripper().lookups(), this);

	return (dlg.exec() == QDialog::Accepted);
/*
	ArsonSendSocket *pSocket = NULL;
	ArsonCdInfoSubmit *pWhat = NULL;

	switch (where)
	{
	case dbFreedb:
		{
			QString email = ACONFIG.ripper().email();
//			const ArsonConfig::CdLookup lookup = ACONFIG.ripper().cddb();

			if (email != QString::null &&
				ACONFIG.ripper().is(ArsonConfig::RipperCfg::ripperEmailPrompt))
				email = QString::null;

			while (email == QString::null)
			{
				bool ok;
				email = KLineEditDlg::getText(
					i18n("Please enter your email address. This is needed by freedb so\nthey notify you if there is a problem with your submission."),
#ifdef ARSONDBG
					(QString(getenv("USER")) == "tonys111")	//	me == lazy
					? QString(BUGEMAIL) : QString::null
#else
					QString::null
#endif	//	ARSONDBG
					, &ok, this);

				if (!ok)
					return false;

				if (email.find('@') < 1 || email.length() < 3)
				{
					arsonErrorMsg(
						i18n("That doesn't appear to be a valid email address, please enter one (or freedb will get mad at *me*)."));
					email = QString::null;
				}
			}

			ACONFIG.ripper().setEmail(email);
			
//			pSocket = new freedbSocket(lookup.addr(), lookup.port());
			pWhat = new ArsonFreedbSubmit(info, email);
		}
		break;

	case dbCdindex:
		arsonErrorMsg(
			i18n("CdIndex submittal is not currently implemented."));
		break;
		
	case dbNone:
		break;
	}

	if (pSocket && pWhat)
	{
		ArsonCdInfoSubmitDlg dlg (pWhat, pSocket);
		const int res = dlg.exec();

		//	The dialog takes care of deleting the socket
		delete pWhat;
		return (res == Accepted);
	}

	return true;
*/
}

/*========================================================*/
/*	The submit selection dialog
 *========================================================*/

class arsonSubmitItem : public QCheckListItem
{
public:
	arsonSubmitItem (QListView *pl, const ArsonLookup *ptr)
		: QCheckListItem(pl, ptr->typeDesc(), CheckBox), m_ptr(ptr)
	{
		setText(1, ptr->display());
	}

	const ArsonLookup *lookup (void) const { return m_ptr; }

private:
	const ArsonLookup *m_ptr;
};

/*========================================================*/

ArsonSubmitDlg::ArsonSubmitDlg (const ArsonCdInfo &info,
	const ArsonLookupOrder &lp, QWidget *parent)
	: ArsonListviewDlgBase(parent, NULL, true),
	m_info(info), m_order(lp)
{
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Type"), 35),
		ArsonListHeader(i18n("Source"), 65),
	};

	teh_list->setListHeaders(hdrs, ARRSIZE(hdrs));
	teh_list->setSelectionMode(QListView::Single);

	for (int index = 0, size = m_order.count(); index < size; ++index)
	{
		arsonSubmitItem *ptr =
			new arsonSubmitItem(teh_list, m_order.lookup(index));

		teh_list->moveItem(ptr, NULL, teh_list->lastChild());
	}

	setCaption(
		i18n("CD Info Submit"));
}

/*========================================================*/

void ArsonSubmitDlg::accept (void)
{
	typedef QValueList<const ArsonLookup*> LOOKUPS;
	LOOKUPS vals;

	for (arsonSubmitItem *ptr = (arsonSubmitItem *) teh_list->firstChild();
		 ptr; ptr = (arsonSubmitItem *) ptr->nextSibling())
		if (ptr->isOn()) vals.append(ptr->lookup());

	ArsonListviewDlgBase::accept();

	for (int index = 0; index < vals.count(); ++index)
		vals[index]->submit(m_info);
}

/*========================================================*/
