/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/* CD Ripper doc/view class implementations
 *
 *	by Tony Sideris	(06:59PM Sep 01, 2001)
 *================================================*/
#include "arson.h"

#include <qtoolbutton.h>
#include <qtextstream.h>
#include <qlabel.h>
#include <qhbox.h>
#include <qheader.h>
#include <qregexp.h>
#include <qpopupmenu.h>
#include <qdatetime.h>

#include <kstddirs.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kregexp.h>
#include <kapp.h>

#include "cdinfowaitdlg.h"
#include "cdinfoeditor.h"
#include "cdripper.h"
#include "konfig.h"
#include "progressdlg.h"
#include "listwnd.h"
#include "mainwnd.h"
#include "process.h"
#include "lookup.h"
#include "encoderopts.h"

/*========================================================*/
/*	This will tag the specified file using id3v2
 *========================================================*/

class ArsonId3v2Process : public ArsonUtilityProcess
{
public:
	ArsonId3v2Process (const ArsonCdInfo &info, int track, const char *fname);
};

/*========================================================*/
/*	This singleton is accessed solely through the static
 *	genre() method. The first time this is called
 *	'id3v2 --list-genres' will be called, its output will
 *	be parsed into a static lookup table.
 *========================================================*/

class ArsonGenreProcess : public ArsonUtilityProcess
{
	typedef QMap<QString,int> GENREMAP;
	
public:
	static int genre (const QString &name);

private:
	ArsonGenreProcess (void);
	virtual void onStdout (int event, KRegExp *pre);

	enum {
		EVENT_GENRE = 0,
		_EVENT_MAX,
	};
	
	static KRegExp events[_EVENT_MAX];
	static GENREMAP m_genres;
};

/*==================================*/
/*	GLOBALS
 *==================================*/

RIPPERFORMAT ArsonRipperDoc::formats[_RIPPERFMT_MAX] = {
	RIPPERFORMAT(I18N_NOOP("Wave"), ".wav", RIPPERFMT_WAV),
	RIPPERFORMAT(I18N_NOOP("MP3"), ".mp3", RIPPERFORMAT::encoderMp3),
	RIPPERFORMAT(I18N_NOOP("Ogg Vorbis"), ".ogg", RIPPERFORMAT::encoderOgg),
#ifdef FLAC
	RIPPERFORMAT(I18N_NOOP("FLAC"), ".flac", RIPPERFORMAT::encoderFlac),
#endif
	RIPPERFORMAT(I18N_NOOP("Sun AU"), ".au", RIPPERFMT_AU),
	RIPPERFORMAT(I18N_NOOP("CDR"), ".cdr", RIPPERFMT_CDR),
	RIPPERFORMAT(I18N_NOOP("AIFF"), ".aiff", RIPPERFMT_AIFF),
	RIPPERFORMAT(I18N_NOOP("AIFC"), ".aifc", RIPPERFMT_AIFC),
};

MP3ENCODER ArsonRipperDoc::mp3encoders[_MP3ENCODER_MAX] = {
	MP3ENCODER(I18N_NOOP("BlandeEnc")),
	MP3ENCODER(I18N_NOOP("LAME")),
};

/*========================================================*/

namespace arson {
	void replFmt (const char *fmt, const QString &with, QString &target)
	{
		int at = target.find(fmt);

		while (at != -1)
		{
			target.replace(at, 2, with);
			at = target.find(fmt, at + 1);
		}
	}

	QString trackComment (const QString &comment)
	{
		QString result (comment);

		if (result.isEmpty())
		{
			result = ACONFIG.ripper().defaultComment();

			replFmt("%d", QDate::currentDate().toString(), result);
			replFmt("%t", QTime::currentTime().toString(), result);
			replFmt("%v", VERSION, result);
		}

		return result;
	}
};

/*========================================================*/

class ArsonRipperListWnd : public ArsonListWnd
{
public:
	ArsonRipperListWnd (ArsonRipperDoc *parent);
};

/*========================================================*/
/*	Ripper format description class (and encoder factory)
 *========================================================*/

ArsonEncoderProcess *RIPPERFORMAT::createEncoder(
	ArsonRipperMgr *pMgr, const char *infile, const char *outfile) const
{
	/*	Create the proper audio encoder process
	 *	based on the properties of 'this'.
	 */
	switch (encoder)
	{
	case encoderNone:
		break;

	case encoderMp3:
		return MP3ENCODER::createMp3Encoder(pMgr,
			infile, outfile);

	case encoderOgg:
		return new ArsonOggencProcess(pMgr,
			infile, outfile);

#ifdef FLAC
	case encoderFlac:
		return new ArsonFlacEncoderProcess(pMgr,
			infile, outfile);
#endif	//	FLAC
	}

	return NULL;
}

/*========================================================*/

void RIPPERFORMAT::fillComboBox (QComboBox *pWnd)
{
	for (int index = 0; index < _RIPPERFMT_MAX; index++)
		pWnd->insertItem(i18n(ArsonRipperDoc::format(index)->description));

	pWnd->setCurrentItem(ACONFIG.ripper().format());
}

/*========================================================*/
/*	Class describes an encoder
 *========================================================*/

ArsonEncoderProcess *MP3ENCODER::createMp3Encoder (ArsonRipperMgr *pMgr,
	const char *infile, const char *outfile)
{
	switch (ACONFIG.seq().seq(PROGGRP_MP3ENC))
	{
	case MP3ENCSEQ_BLADEENC:
		return new ArsonBladeProcess(pMgr,
			infile, outfile);

	case MP3ENCSEQ_LAME:
		return new ArsonLameProcess(pMgr,
			infile, outfile);
	}

	Assert(false);
	return NULL;
}

/*========================================================*/
/*	Ripper document class implementation
 *========================================================*/

ArsonRipperDoc::ArsonRipperDoc (QWidget *parent, const char *name)
	: ArsonListDoc(parent, name)
{
	clearf(stdDocActions);

	setEditableList(false);
	setRenameColumn(1);
}

/*========================================================*/

void ArsonRipperDoc::connectTo (ArsonActionConnector &ac)
{
	ac.connect("ripper_rescan", SLOT(slotRescan()));
	ac.connect("ripper_rip", SLOT(slotRip()));
	ac.connect("ripper_edit", SLOT(slotEditDiskInfo()));
	ac.connect("ripper_toggle_checked", SLOT(toggleChecked()));

	ArsonListDoc::connectTo(ac);
}

/*========================================================*/

void ArsonRipperDoc::deleteContents (void)
{
	if (m_info.valid())
	{
		const ArsonLookupOrder &lp = ACONFIG.ripper().lookups();

		lp.autoSubmit(m_info);
	}

	ArsonListDoc::deleteContents();
	m_info.clear();
}

/*========================================================*/
/*	Execute cdda2wav to query the information about the
 *	CD that's currently in the drive, parse this info
 *	into list items.
 *========================================================*/

class trackReader : public ArsonUtilityProcess
{
public:
	trackReader (ArsonRipperDoc *pDoc)
		: ArsonUtilityProcess(ACONFIG, Stderr, QString::null, ArsonConfig::PROGRAM_CDDA2WAV),
		m_lines(0),
		m_pDoc(pDoc)
	{
		if (const ArsonDevice *pd = config().devices().device(config().ripper().srcdev()))
		{
//			appendExtra();

			THIS << "-H"
				 << "-J"
				 << "-g"
				 << "-v" << "1"; // "toc,sectors,titles";

			if (pd->scsiDevice())
				THIS << "-I" << "generic_scsi"
					 << "-D" << QString(pd->id());
			else
				THIS << "-I" << "cooked_ioctl"
					 << "-D" << QString(pd->dev());

			/*	If buffering is enabled we wont get
			 *	the "please insert a CD error" we'll
			 *	reenable buffering later.
			 */
			setBuffering(false);
		}
		else
			invalidate(
				i18n("Failed to find device: %1")
				.arg(config().ripper().srcdev()));
	}

	const ArsonCdInfo &info (void) const { return m_info; }

private:
	bool getDiskInfo (const QString &out)
	{
		static KRegExp res[] = {
			"^CDINDEX discid: ([0-9a-zA-Z.-]+)",
			"^CDDB discid: 0x([0-9a-f]+)",
			"^Album title: '(.*)' from '(.*)'",
			"^Leadout:[ \t]*([0-9]+)",
		};
		enum {
			CDINDEX, CDDB, DISKINFO, LEADOUT,
			_DISCINFO_MAX,
		};

		for (int index = 0; index < _DISCINFO_MAX; ++index)
			if (res[index].match(out))
			{
				switch (index)
				{
				case CDINDEX:
					m_info.setCdIndexID(res[index].group(1));
					break;

				case CDDB:
					m_info.setCddbID(res[index].group(1));
					break;

				case DISKINFO:
					m_info.setArtist(res[index].group(2));
					m_info.setTitle(res[index].group(1));
					break;

				case LEADOUT:
					m_info.setLeadout(
						QString(res[index].group(1)).toUInt());
					break;
				}

				return true;
			}

		return false;
	}

	bool getTrackInfo (const QString &out)
	{
		static KRegExp re("^T([0-9]+):[ ]+([0-9]+)[ ]+([0-9]+)[:]([0-9]+)\\.([0-9]+)");
		static KRegExp text("'(.*)' from '(.*)'$");

//		Trace("line: %s\n", out.latin1());

		if (re.match(out))
		{
			static const QRegExp reQuote("\\\\'");
			ArsonCdInfo::Track::Time time;
			ArsonCdInfo::Track track;
			QString trackName, artist;

#define TOINT(c)	QString(re.group((c))).toInt()
#define TOUINT(c)	QString(re.group((c))).toUInt()

			track.setTrackNo(TOINT(1));
			track.setOffset(TOUINT(2));

			time.min = TOINT(3);
			time.sec = TOINT(4);
			time.fract = TOINT(5);
			track.setTime(time);

			if (text.match(out))
			{
				if (!(trackName = text.group(1)).isEmpty())
					trackName.replace(reQuote, "'");

				track.setTitle(trackName);

				if (!(artist = text.group(2)).isEmpty())
					artist.replace(reQuote, "'");

				track.setArtist(artist);
			}

			m_info.addTrack(track);
			return true;
		}

		return false;
	}

	virtual void output (const QString &out, bool error)
	{
/*
		fwrite(out.ascii(), 1, out.length(), stdout);
		puts("");puts("");fflush(stdout);
*/
		/*	Everything should be all good now, reenable
		 *	buffering, 'cause unbuffered input is scary.
		 */
		if (++m_lines > 2)
			setBuffering(true);

		if (out.find("load cdrom please and press enter") != -1)
		{
			kill();

			arsonErrorMsg(
				i18n("Please insert a CD, and try again."));

			return;
		}

		/*	Attempt to read general disc info
		 *	first, then look for track info.
		 */
		else if (!getDiskInfo(out))
			getTrackInfo(out);
	}

	ArsonRipperDoc *m_pDoc;
	ArsonCdInfo m_info;
	int m_lines;
};

/*========================================================*/

void ArsonRipperDoc::readTrackInfo (void)
{
	trackReader rdr (this);
	deleteContents();	//	Start the list over from scratch

	if (rdr.execute())
	{
		m_info.merge(rdr.info());

		for (int index = 0; index < m_info.trackCount(); ++index)
			addItem(new ArsonRipperListItem(m_info.track(index).trackNo()));
		
		applyDiskInfo();

		if (ACONFIG.ripper().is(ArsonConfig::RipperCfg::ripperCdLookup) &&
			(!m_info.valid() || ACONFIG.ripper().is(ArsonConfig::RipperCfg::ripperOverCdtext)))
			cdIndex();
	}
}

/*========================================================*/
/*	Popup the wait dialog and get cdindex information
 *	about the current disk from the cdindex server.
 *
 *	This function is misnamed... it should be something
 *	like cdLookup or something...
 *========================================================*/

void ArsonRipperDoc::cdIndex (void)
{
	const ArsonLookupOrder &order = ACONFIG.ripper().lookups();

	for (int index = 0, size = order.count(); index < size; ++index)
	{
		const ArsonLookup *pl = order.lookup(index);

		try {
			if (pl->cdInfo(m_info))
			{
				applyDiskInfo();
				break;
			}
		}
		catch (ArsonError &err) {
			err.report();
			break;
		}
	}
}

/*========================================================*/

void ArsonRipperDoc::slotRescan (void)
{
	readTrackInfo();
}

/*========================================================*/

void ArsonRipperDoc::slotEditDiskInfo (void)
{
	ArsonCdInfoEditor dlg (ArsonFrame::getFrame(), m_info);

	if (dlg.exec() == QDialog::Accepted)
	{
		m_info.merge(dlg.info());
		applyDiskInfo();
	}
}

/*========================================================*/
/*	Use CDINDEX information
 *========================================================*/

class diskInfo
{
public:
	diskInfo (const ArsonCdInfo &info)
		: m_info(info)
	{
		if (info.variousArtistDisk())
			m_text = i18n("Various Artists - ") + info.title();
		else
			m_text = QString("%1 - %2")
				.arg(m_info.artist())
				.arg(m_info.title());
	}

	diskInfo &append (const QString &key, const QString &val)
	{
		if (val != QString::null)
			m_text
				.append(i18n(", "))
				.append(key)
				.append(i18n(": "))
				.append(val);

		return *this;
	}
	
	const QString &text (void) const { return m_text; }

protected:
	const ArsonCdInfo &m_info;
	QString m_text;
};

void ArsonRipperDoc::applyDiskInfo (void)
{
	diskInfo di (m_info);

	di.append(i18n("Genre"), m_info.genre());
	di.append(i18n("Year"), m_info.year());

	//	Set status bar info
	setPaneText("discinfo", di.text());

	//	Update track items
	refresh();
}

/*========================================================*/
/*	Rip the selected tracks
 *========================================================*/

class ripperProgressDlg : public ArsonSimpleProgressDlg
{
public:
	ripperProgressDlg (ArsonRipperDoc *pDoc)
		: ArsonSimpleProgressDlg(
			i18n("Press Start to begin ripping the selected tracks..."),
			i18n("Ripper Progress"), NULL, "ripper"), m_pDoc(pDoc)
	{
		QLabel *pl = new QLabel(
			i18n("Audio Quality: "), ctrlParent());
		pl->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum));

		m_pTag = addCheckbox(
			i18n("&Add id3v2 tags to MP3 files"), "tag");

		m_pQuality = new ArsonProgressCombobox(ctrlParent(), "encqual");
		pl->setBuddy(m_pQuality);

		layoutRow() << pl << m_pQuality;
	}

private:
	virtual void processOpts (ArsonProcessOpts &opts)
	{
		opts.addBool("tag", m_pTag->isChecked() ? true : false);
		opts.addString("quality", m_pQuality->currentText());
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		ArsonRipperMgr *ptr = new ArsonRipperMgr(ui(),
			m_pDoc, m_pDoc->outputFormat());
		const QString path = m_pDoc->outputPath();
		ArsonRipperDoc::ITEMLIST il = m_pDoc->checkedItems();

		//	Add each selected track to the ripper's tracklist.
		for (int index = 0; index < il.count(); ++index)
		{
			const QString fname = m_pDoc->filename(il[index].pListItem);

			try {
				ptr->addFile(
					il[index].pRipperItem->trackNo(),
					path + fname
					);
			}
			catch (ArsonError &err) {
				delete ptr;
				throw;
			}
		}

		return ptr;
	}

	virtual void reconfigure (void)
	{
		ArsonSimpleProgressDlg::reconfigure();

		fillQuality();

		m_pTag->setEnabled(
			ACONFIG.program(ArsonConfig::PROGRAM_ID3V2)->valid());
	}

	void fillQuality (void)
	{
		ArsonEncoderPresets presets;

		m_pQuality->clear();
		m_pQuality->insertStringList(presets.names());

		if (QListBoxItem *pi = m_pQuality->listBox()->findItem(ACONFIG.ripper().quality()))
			m_pQuality->setCurrentItem(m_pQuality->listBox()->index(pi));
	}

	ArsonRipperDoc *m_pDoc;
	QComboBox *m_pQuality;
	QCheckBox *m_pTag;
};

/*========================================================*/

void ArsonRipperDoc::slotRip (void)
{
	if (checkedItems().isEmpty())
	{
		arsonErrorMsg(
			i18n("No tracks have been selected for ripping, select some."));
	}
	else
		ripperProgressDlg(this).exec();
}

/*========================================================*/
/*	A list item type describing a cd audio track
 *========================================================*/

ArsonRipperListItem::ArsonRipperListItem (int trackNo)
	: m_trackNo(trackNo)
{
	//	Nothing...
}

/*========================================================*/

namespace arson {
	class checkListItem : public QCheckListItem
	{
	public:
		checkListItem (QListView *pw, QListViewItem *after)
			: QCheckListItem(pw, QString(), QCheckListItem::CheckBox)
		{
			((KListView *) listView())->moveItem(this, NULL, after);
		}

		checkListItem (QListViewItem *parent, QListViewItem *after)
			: QCheckListItem(parent, QString(), QCheckListItem::CheckBox)
		{
			((KListView *) listView())->moveItem(this, parent, after);
		}
	};
};

QListViewItem *ArsonRipperListItem::createItem (ArsonListWnd *parentWnd,
	QListViewItem *parentItem, QListViewItem *pAfter)
{
	return arson::newListViewItem<arson::checkListItem>(parentWnd, parentItem, pAfter);
}

/*========================================================*/

void ArsonRipperListItem::refresh (QListViewItem *pi, ArsonDocWidget *pDoc)
{
	QString name;
	ArsonRipperDoc *pd = (ArsonRipperDoc *) pDoc;
	const RIPPERFORMAT *pf = pd->outputFormat();
	const ArsonCdInfo &info = pd->info();
	const ArsonCdInfo::Track &track = info.track(m_trackNo - 1);
	
	if (!track.title().isEmpty())
	{
		track.formatTrackName(&info,
			ACONFIG.ripper().cdiFormat(), name);

		if (!name.isEmpty())
			name.append(pf->extension);
	}

	if (name.isEmpty())
		name = defaultFileName(pf);

	//	Fill in the other cells
	pi->setText(0, QString::number(m_trackNo));
	pi->setText(1, name);
	pi->setText(2, track.time().display());
}

/*========================================================*/

QString ArsonRipperListItem::defaultFileName (const RIPPERFORMAT *pFmt) const
{
	return QString().sprintf("track-%02d%s",
		m_trackNo, pFmt->extension);
}

/*========================================================*/
/*	List window class for the cd ripper
 *========================================================*/

ArsonRipperListWnd::ArsonRipperListWnd (ArsonRipperDoc *parent)
	: ArsonListWnd(parent)
{
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Track"), 10),
		ArsonListHeader(i18n("Filename"), 80),
		ArsonListHeader(i18n("Length"), 10),
	};

	setListHeaders(hdrs, ARRSIZE(hdrs));

	setSorting(-1);
	setSelectionMode(QListView::Single);

	setItemsRenameable(true);
	setRenameable(0, false);
	setRenameable(1, true);
	setRenameable(2, false);

	header()->setClickEnabled(false);
}

/*========================================================*/

ArsonListWnd *ArsonRipperDoc::createListWnd (void)
{
	ArsonRipperListWnd *ptr = new ArsonRipperListWnd(this);

	connect(ptr, SIGNAL(clicked(QListViewItem*)),
		this, SLOT(slotItemStateChange(QListViewItem*)));

	return ptr;
}

/*========================================================*/

void ArsonRipperDoc::create (void)
{
	QHBox *phb;
	QButton *pb;

	ArsonListDoc::create();
	phb = new QHBox(this);

	new QLabel(i18n(" Output Format: "), phb);
	m_pFormat = new QComboBox(phb);

	RIPPERFORMAT::fillComboBox(m_pFormat);
	
	connect(m_pFormat, SIGNAL(activated(int)),
		this, SLOT(slotFormatChanged(int)));
	
	new QLabel(i18n(" Output Directory: "), phb);
	m_pPathField = new QLineEdit(phb);
	m_pPathField->setText(ACONFIG.ripper().outdir());

	pb = new QToolButton(phb);
	pb->setText(i18n("..."));

/*
	pushStatusPane("discinfo", i18n("unknown/unknown"));
	pushStatusPane("selected", totalSelected());
*/
	
	connect(pb, SIGNAL(clicked()),
		this, SLOT(slotBrowse()));

	phb->show();
}

/*========================================================*/

ArsonDocWidget::Status *ArsonRipperDoc::initStatus (QStatusBar *psb)
{
	Status *ps = ArsonListDoc::initStatus(psb);

	ps->addPane("discinfo", i18n("unknown/unknown"));
	ps->addPane("selected", totalSelected());

	return ps;
}

/*========================================================*/
/*	Display a dialog to change the output directory.
 *========================================================*/

void ArsonRipperDoc::slotBrowse (void)
{
	const QString dir = KFileDialog::getExistingDirectory();

	if (dir != QString::null)
		m_pPathField->setText(dir);
}

/*========================================================*/
/*	When the user checks/unchecks a track, update the
 *	UI (statusbar & menu/toolbar states).
 *========================================================*/

void ArsonRipperDoc::slotItemStateChange (QListViewItem *ptr)
{
	setPaneText("selected",
		totalSelected());
}

/*========================================================*/
/*	When the user changes the output format,
 *	auto-change all file extensions.
 *========================================================*/

void ArsonRipperDoc::slotFormatChanged (int index)
{
	refresh();
}

/*========================================================*/
/*	Helper to iterate through checked list items
 *========================================================*/

ArsonRipperDoc::ITEMLIST ArsonRipperDoc::checkedItems (void) const
{
	ITEMLIST result;

	for (ItemIterator it (this); it; ++it)
	{
		QCheckListItem *pc = (QCheckListItem *) it.lvi();
		ITEMPAIR ip = { pc, (ArsonRipperListItem *) it.item() };

		if (pc->isOn())
			result.append(ip);
	}

	return result;
}

/*========================================================*/
/*	Retrieve information about the checked items.
 *========================================================*/

QString ArsonRipperDoc::totalSelected (int *_total, int *_secs) const
{
	int total, totalSecs, index;
	ITEMLIST il = checkedItems();

	//	Tally up the total checked, and total time
	for (total = il.count(), totalSecs = index = 0; index < total; ++index)
		totalSecs += info().track(index).time().totalSeconds();

	//	Return requestesd info
	if (_secs) *_secs = totalSecs;
	if (_total) *_total = total;

	if (total > 0)
		return i18n("%1 items selected, %2 minutes")
			.arg(total)
			.arg(arsonDisplayTime(totalSecs));

	return i18n("Nothing selected");
}

/*========================================================*/
/*	Toggle the state of all check items
 *========================================================*/

void ArsonRipperDoc::toggleChecked (void)
{
	QCheckListItem *pItem;

	for (pItem = (QCheckListItem *) listWnd()->firstChild();
		 pItem; pItem = (QCheckListItem *) pItem->nextSibling())
		pItem->setOn(!pItem->isOn());

	slotItemStateChange(listWnd()->firstChild());
}

/*========================================================*/

QString ArsonRipperDoc::outputPath (void) const
{
	QString res = m_pPathField->text();

	if (res[res.length() - 1] != QChar('/'))
		res += "/";

	if (!QFileInfo(QFile::encodeName(res)).isDir())
		throw ArsonError(i18n("%1 is not a valid directory!").arg(res));

	return res;
}

void ArsonRipperDoc::setOutputPath (const QString &path)
{
	m_pPathField->setText(path);
}

/*========================================================*/

const RIPPERFORMAT *ArsonRipperDoc::outputFormat (void) const
{
	return ArsonRipperDoc::format(m_pFormat->currentItem());
}

/*========================================================*/

QString ArsonRipperDoc::filename (QListViewItem *pi) const
{
	return pi->text(1);
}

/*========================================================*/

bool ArsonRipperDoc::buildContextMenu (QPopupMenu *pup)
{
	KActionCollection *pa = ArsonFrame::getFrame()->actionCollection();
	const char *items[] = {
		"ripper_rescan",
		"ripper_edit",
		"ripper_toggle_checked",
		"ripper_rip",
	};

	for (int index = 0; index < ARRSIZE(items); ++index)
		pa->action(items[index])->plug(pup);

	return ArsonListDoc::buildContextMenu(pup);
}

/*========================================================*/
/*	Process manager for audio ripper
 *========================================================*/

ArsonRipperMgr::ArsonRipperMgr (ArsonProcessUI *pUI,
	ArsonRipperDoc *pDoc, const RIPPERFORMAT *pFmt)
	: ArsonProcessMgr(pUI),
	m_currentTrack(-1),
	m_pFifo(NULL),
	m_pDoc(pDoc),
	m_pFmt(pFmt)
{
	if (!m_pFmt)
		throw ArsonError(i18n("No output format specified!"));

	if (opts().seq().seq(PROGGRP_RIPPER) == RIPPERSEQ_CDPARANOIA)
	{
		switch (m_pFmt->encoder)
		{
		case RIPPERFMT_CDR:
		case RIPPERFMT_AU:
			throw ArsonError(
				i18n("This output format is not supported by cdparanoia!\n") +
				i18n("Please use cdda2wav, or choose a different format."));
		}
	}
}

ArsonRipperMgr::~ArsonRipperMgr (void)
{
	delete m_pFifo;
}

/*========================================================*/

void ArsonRipperMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonProcessMgr::begin(opts);
	next();
}

/*========================================================*/

ArsonRipperProcess *ArsonRipperMgr::createRipperProcess(
	ArsonRipperMgr *pThis, int trackNo, const char *outfile)
{
	if (opts().seq().seq(PROGGRP_RIPPER) == RIPPERSEQ_CDDA2WAV)
		return new ArsonCdda2WavProcess(pThis, trackNo, outfile);

	return new ArsonCdparanoiaProcess(pThis, trackNo, outfile);
}

/*========================================================*/

inline const char *SF (const QString &str)
{
	return str.isEmpty() ? NULL : str.latin1();
}

void ArsonRipperMgr::ripTrack (int trackNo, const char *outfile)
{
	ArsonProcess *ptr = NULL;
	ArsonProcess *pRipper = NULL;

	if (trackNo != -1)
	{
		delete m_pFifo;
		m_pFifo = new ArsonFifo("encoder-", "fifo.wav");

		try {
			const ArsonCdInfo &info = m_pDoc->info();
			const ArsonCdInfo::Track track = info.track(trackNo - 1);

			/*	Create the encoder (one will only be created if the selected
			 *	output format is an encoded format, therefore this returning
			 *	NULL, is NOT a failure), and set the tag information.
			 */
			if (ArsonEncoderProcess *pe = m_pFmt->createEncoder(
					this, m_pFifo->filename(), outfile))
			{
				pe->setTag(
					track.title(),
					trackNo,
					track.artist(&info),
					info.title(),
					arson::trackComment(track.comment()),
					info.year(),
					info.genre()
					);

				ptr = pe;
			}
		}
		catch (ArsonError &err)
		{
			if (ArsonProcessUI *pUI = ui())
				pUI->setText(i18n("Failed to create encoder."));
			
			err.report();	//	An exception IS an error.
			return;
		}

		/*	If a encoder process is actually created then we need to
		 *	setup a FIFO and a have the CD track ripped into the FIFO
		 */
		if (ptr)
		{
			try
			{
				if (!m_pFifo->valid())
					throw ArsonError(i18n("Failed to create FIFO"));

				/*	Create the process to rip the track,
				 *	and give it to the encoder to control
				 */
				pRipper = createRipperProcess(this, trackNo, m_pFifo->filename());
				((ArsonEncoderProcess *) ptr)->setRipper(pRipper);

				if (!pRipper->execute())
					throw ArsonError(QString::null);
			}
			catch (ArsonError &err)
			{
				if (ArsonProcessUI *pUI = ui())
					pUI->setText(
						i18n("Failed to rip track %1 to %2")
						.arg(trackNo)
						.arg(outfile));

				err.report();
				delete ptr;
				return;
			}
		}
		else
		{
			try {
				ArsonRipperProcess *pRip;

				//	Just rip the track straight to the file...
				delete m_pFifo;
				m_pFifo = NULL;

				pRip = createRipperProcess(this, trackNo, outfile);
				pRip->setOutputFormat(m_pFmt->encoder);
				ptr = pRip;
			}
			catch (ArsonError &err) {
				err.report();
				return;
			}
		}
	}

	setProcess(ptr);

	//	setProcess may call setText too, we need to overwrite...
	if (m_pUI && ptr && (!pRipper || pRipper->isRunning()))
		m_pUI->setText(i18n("Ripping track %1 into %2 ...")
			.arg(trackNo)
			.arg(outfile));
}

/*========================================================*/

void ArsonRipperMgr::next (void)
{
	TRACKINFO info (NULL, -1);

	if (m_currentTrack >= 0)
		m_pUI->setProgress(100);

	if (++m_currentTrack < m_tracks.count())
		info = m_tracks[m_currentTrack];

	ripTrack(info.trackNo, info.filename);
}

/*========================================================*/
/*	This allows naming formats such as "%a/%c/%N - %t" to
 *	work. It creates any missing directories.
 *========================================================*/

bool ArsonRipperMgr::verifyDirectory (const char *outfile) const
{
	const QDir srcdir (QFileInfo(outfile).dir());

	if (!srcdir.exists())
	{
		QDir dir = QDir::root();
		QStringList::ConstIterator it, end;
		const QStringList sl(
			QStringList::split(QChar('/'),
				srcdir.absPath()));

		for (it = sl.begin(), end = sl.end(); it != end; ++it)
		{
			const bool create = dir.mkdir(*it);

			if (!dir.cd(*it))
			{
				if (!create)
					arsonErrorMsg(
						i18n("Failed to create output directory: %1 in %2")
						.arg(*it)
						.arg(dir.absPath()));
				else
					arsonErrorMsg(
						i18n("Failed to access %1 in %2")
						.arg(*it)
						.arg(dir.absPath()));

				return false;
			}
		}
	}

	return true;
}

/*========================================================*/

void ArsonRipperMgr::addFile (int trackNo, const char *outfile)
{
	const TRACKINFO info (outfile, trackNo);

	Trace("adding %d %s\n", trackNo, outfile);

	if (m_tracks.isEmpty() || m_tracks.find(info) == m_tracks.end())
	{
		if (verifyDirectory(outfile))
			m_tracks.append(info);
	}
	else
		throw ArsonError(
			i18n("A track with that filename (%1) was already specified!")
			.arg(outfile));
}

/*========================================================*/

void ArsonRipperMgr::tagFile (void)
{
	if (m_pFmt->encoder == RIPPERFORMAT::encoderMp3 && opts().getBool("tag")
		&& m_currentTrack >= 0 && m_currentTrack < m_tracks.count())
	{
		const TRACKINFO &info = m_tracks[m_currentTrack];
		ArsonId3v2Process proc (m_pDoc->info(),
			info.trackNo - 1, info.filename);

		proc.execute();
	}
}

/*========================================================*/

void ArsonRipperMgr::taskComplete (ArsonProcess *ptr)
{
	ArsonProcessMgr::taskComplete(ptr);

	if (ptr == process())
	{
		tagFile();
		next();
	}
}

/*========================================================*/
/*	A container cdinfo queries
 *========================================================*/

ArsonCdInfoWaitDlg::ArsonCdInfoWaitDlg (const ArsonCdInfo &info)
	: ArsonSocketWaitDlg()
{
	m_info.merge(info);
}

/*========================================================*/

void ArsonCdInfoWaitDlg::slotCompleted (ArsonSocket *pThis)
{
	FILE *file;
	ArsonRetrSocket *ps = (ArsonRetrSocket *) m_pSocket;
	Assert(pThis == m_pSocket);

	if ((file = fopen(QFile::encodeName(ps->tempName()), "r")))
	{
		ArsonHttpStream str (file);
		handleStream(&str);
	}
}

/*========================================================*/
/*	An id3v2 process
 *========================================================*/

ArsonId3v2Process::ArsonId3v2Process (const ArsonCdInfo &info, int trackNo, const char *fname)
	: ArsonUtilityProcess(ACONFIG, NoCommunication, QString::null, ArsonConfig::PROGRAM_ID3V2)
{
	const ArsonCdInfo::Track &track = info.track(trackNo);
	const int genre = ArsonGenreProcess::genre(info.genre());
	const QString comment = arson::trackComment(track.comment());

	deferArg(QFile::encodeName(fname));

	THIS << "--artist" << track.artist(&info)
		 << "--album" << info.title()
		 << "--song" << track.title()
		 << "--year" << info.year()
		 << "--track" << QString::number(track.trackNo());

	if (!comment.isEmpty())
		THIS << "--comment" << comment;
	
	if (genre != -1)
		THIS << "--genre" << QString::number(genre);
}

/*========================================================*/
/*	Maps textual genres to mp3 header indeices
 *========================================================*/

ArsonGenreProcess::GENREMAP ArsonGenreProcess::m_genres;

KRegExp ArsonGenreProcess::events[ArsonGenreProcess::_EVENT_MAX] = {
	" *([0-9]+):[ \t]+(.+)",
};

int ArsonGenreProcess::genre (const QString &name)
{
	GENREMAP::ConstIterator it;

	if (m_genres.isEmpty())
		ArsonGenreProcess().execute();

	if ((it = m_genres.find(name)) != m_genres.end())
		return it.data();

	return -1;
}

/*========================================================*/

ArsonGenreProcess::ArsonGenreProcess (void)
	: ArsonUtilityProcess(ACONFIG, Stdout, QString::null, ArsonConfig::PROGRAM_ID3V2)
{
	setOutput(Stdout, events, ARRSIZE(events));
	
	THIS << "--list-genres";
}

/*========================================================*/

void ArsonGenreProcess::onStdout (int event, KRegExp *pre)
{
	if (event == EVENT_GENRE)	//	And why wouldn't it...
	{
		QString genre = pre->group(2);
		const int index = QString(pre->group(1)).toInt();

		m_genres.insert(genre, index);
	}
}

/*========================================================*/
