/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __DOCWIDGET_H__
#define __DOCWIDGET_H__

#include <qvbox.h>
#include <qmap.h>

#include <kstdaction.h>
#include <kurl.h>

#include "arson.h"

/*========================================================*/

class KActionCollection;

class ArsonActionEnabler
{
public:
	ArsonActionEnabler (KActionCollection &pa) : m_actions(pa) { }

	void enable (KStdAction::StdAction act, bool enable);
	void enable (const char *name, bool enable);
	void enable (KAction *pa, bool enable);

protected:
	KActionCollection &m_actions;
};

/*========================================================*/

class ArsonDocWidget;

class ArsonActionConnector : public ArsonActionEnabler
{
public:
	ArsonActionConnector (KActionCollection &pa, ArsonDocWidget *pw, bool connect);

	void connect (KStdAction::StdAction act, const char *slot, bool enable = true);
	void connect (const char *name, const char *slot, bool enable = true);
	void connect (KAction *pa, const char *slot, bool enable);

	void disconnect (KStdAction::StdAction act);
	void disconnect (const char *name);
	void disconnect (KAction *pa);

private:
	ArsonDocWidget *m_pWidget;
	bool m_connect;
};

/*========================================================*/

class QListViewItem;

class ArsonListInserter
{
public:
	ArsonListInserter (QListViewItem *p = NULL, QListViewItem *a = NULL)
		: parent(p), after(a) { }
	
	void setParent (QListViewItem *pi);
	void clearParent (void);
	void setAfter (QListViewItem *pi);

	QListViewItem *getParent (void) const { return parent; }
	QListViewItem *getAfter (void) const { return after; }

private:
	QListViewItem *last (QListViewItem *pi);

	QListViewItem *parent;
	QListViewItem *after;
};

/*========================================================*/
/*	Exception Class - Abort current action
 *========================================================*/

class ArsonAbortAction { };

/*========================================================*/

class QXmlAttributes;
class QStatusBar;
class ArsonLvPos;
class ArsonXmlWriter;
class ArsonBaseWriter;

class ArsonDocWidget : public QVBox
{
	Q_OBJECT

protected:
	class Status
	{
		typedef QMap<QString,QWidget*> STATUSPANES;
		
	public:
		Status (QStatusBar *psb) : m_pStatus(psb) { }

		void addPane (const char *name, const QString &text);
		void addPane (const char *name, QWidget *pw);
		void message (const QString &text);
		QWidget *pane (const char *name);
		void show (bool sh);

	private:
		QStatusBar *m_pStatus;
		STATUSPANES m_panes;
	};

	enum {
		sizeCanChange = 0x01,
		modified = 0x02,
		stdDocActions = 0x04,
	};

private:
	Status *m_pStatus;
	KURL m_url;
	uint m_flags;
	
public:
	ArsonDocWidget (QWidget *parent, const char *name = NULL);
	virtual ~ArsonDocWidget (void);

	virtual void newDocument (void);
	virtual bool saveDocument (const KURL &url);
	virtual void setDocumentName (const KURL &url);
	virtual void deleteContents (void);
	virtual void setActive (bool active);

	bool canSizeChange (void) const { return is(sizeCanChange); }
	void setSizeCanChange (bool ch);

	bool isModified (void) const { return is(modified); }
	void setModified (bool mod = true) { boolf(modified, mod); }

	void setStatusBar (QStatusBar *psb);

	bool is (uint mask) const { return (m_flags & mask) != 0; }
	void clearf (uint mask) { m_flags &= ~mask; }
	void setf (uint mask) { m_flags |= mask; }
	void boolf (uint mask, bool b) { if (b) setf(mask); else clearf(mask); }
	void togglef (uint mask) { m_flags ^= mask; }

	virtual QString propDocType (void) const { return QString::null; }
	virtual QString propFolder (void) const { return QString::null; }
	virtual QString propItem (void) const { return QString::null; }

	virtual bool onStartElement (const QString &name,
		const QXmlAttributes &attr, ArsonListInserter &pos) { return true; }
	virtual bool onEndElement (const QString &name,
		ArsonListInserter &pos) { return true; }
	virtual void defaultLvPos (ArsonLvPos &pos) { }
	virtual void editRootElement (ArsonXmlWriter &xml) { }

	virtual void refresh (void) { }
	virtual void create (void) { }

	static const QString ARSON_FILTER;
	
protected:
	void emitSizeChanged (void);

	virtual void connectTo (ArsonActionConnector &ac);

	virtual ArsonBaseWriter *createFileWriter (const KURL &url) { return NULL; }
	virtual bool writeDocument (ArsonBaseWriter &writer) = 0;
	bool writeDocumentFile (ArsonBaseWriter &writer);

	virtual Status *initStatus (QStatusBar *psb);
	
	void setPaneText (const char *name, const QString &text);
	QWidget *pane (const char *name);
	void status (const QString &str);

	ArsonActionConnector connector (bool active);
	ArsonActionEnabler enabler (void);

protected slots:
	void slotModified (void);

public slots:
	void slotNew (void);
	void slotSave (void);
	void slotSaveAs (void);
	void slotExportText (void);

signals:
	void sizeChanged (void);
};

/*========================================================*/

class ArsonDocSizeLock
{
public:
	ArsonDocSizeLock (ArsonDocWidget *pd)
		: m_pd(pd)
	{
		m_pd->setSizeCanChange(false);
	}

	~ArsonDocSizeLock (void)
	{
		m_pd->setSizeCanChange(true);
	}

private:
	ArsonDocWidget *m_pd;
};

/*========================================================*/

class ArsonListWnd;
class ArsonListDoc;

struct ArsonLvPos {
	ArsonLvPos (QListViewItem *p = NULL, QListViewItem *a = NULL)
		: parent(p), after(a), result(NULL) { }
	ArsonLvPos (ArsonListDoc *pd);

	void swapCurrent (void) { after = result; result = NULL; }

	QListViewItem *parent;
	QListViewItem *after;
	QListViewItem *result;
};

namespace arson {
	template<class T> QListViewItem *newListViewItem (ArsonListWnd *pw,
		QListViewItem *pItem, QListViewItem *pAfter)
	{
		if (pItem)
			return new T(pItem, pAfter);

		return new T(pw, pAfter);
	}
};

class ArsonListItem
{
public:
	ArsonListItem (void);
	virtual ~ArsonListItem (void);

	virtual QString display (void) const = 0;	//	Human
	virtual QString persist (void) const = 0;	//	Machine

	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);

	virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd);
};

/*========================================================*/

class ArsonListVisiter
{
public:
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl) = 0;
};

class ArsonTreeVisiter : public ArsonListVisiter
{
public:
	virtual bool beginBranch (QListViewItem *pi, ArsonListItem *pl) { return true; }
	virtual void endBranch (void) { }
};

/*========================================================*/

class QDropEvent;
class KListView;

class ArsonListDoc : public ArsonDocWidget
{
	Q_OBJECT

	typedef QMap<QListViewItem*,ArsonListItem*> ITEMMAP;

public:
	ArsonListDoc (QWidget *parent, const char *name = NULL);
	virtual ~ArsonListDoc (void);

	virtual bool writeDocument (ArsonBaseWriter &writer);

	virtual void addItem (ArsonListItem *ptr, ArsonLvPos *pos = NULL);
	virtual void delItem (QListViewItem *ptr);
	virtual void deleteContents (void);

	virtual void defaultLvPos (ArsonLvPos &pos);
	virtual QListViewItem *rootItem (void) const;
	ArsonListItem *item (QListViewItem *pi) const;

	int count (void) const { return m_items.count(); }
	ArsonListWnd *listWnd (void) const { return m_pList; }
	int renameColumn (void) const { return m_renameCol; }
	void setRenameColumn (int n) { m_renameCol = n; }

	bool visitAll (ArsonTreeVisiter &obj);
	bool visitAll (ArsonListVisiter &obj);
	
	class ItemIterator {
	public:
		ItemIterator (const ArsonListDoc *pd, QListViewItem *pi = NULL);

		ItemIterator childIterator (void) const;
		ItemIterator &operator++ (void);
		ItemIterator &operator++ (int);

		operator void * (void) const { return (void *) (m_pListItem != NULL); }
		ArsonListItem *item (void) { return m_pd->item(m_pListItem); }
		QListViewItem *lvi (void) const { return m_pListItem; }
		ArsonListItem *operator-> (void) { return item(); }

	private:
		QListViewItem *m_pListItem;
		const ArsonListDoc *m_pd;
	};

	virtual void refresh (void);
	virtual void create (void);

protected:
	virtual void connectTo (ArsonActionConnector &ac);
	virtual bool buildContextMenu (QPopupMenu *pm);
	virtual ArsonListWnd *createListWnd (void);

	void delTreeListItem (QListViewItem *pi);

	bool editableList (void) const { return m_editableList ? true : false; }
	void setEditableList (bool b) { m_editableList = int(b); }
	
	friend class ItemIterator;

private:
	bool visit (ArsonListVisiter &obj, QListViewItem *pi);
	bool visit (ArsonTreeVisiter &obj, QListViewItem *pi);
	
	ArsonListWnd *m_pList;
	ITEMMAP m_items;
	int m_renameCol : 8;
	int m_editableList : 1;

protected slots:
	void slotContextMenu (KListView *ptr, QListViewItem *item, const QPoint &pnt);
	
public slots:
	void slotListDel (void);
	void slotListUp (void);
	void slotListDn (void);
	void slotListSelAll (void);
	void slotListInvertSel (void);
	void slotRename (void);
	void slotShuffle (void);
};

/*========================================================*/

class QPixmap;

class ArsonFileListItem : public ArsonListItem
{
public:
	ArsonFileListItem (void) : ArsonListItem() { }

	virtual QString display (void) const = 0;
	virtual QString persist (void) const = 0;

	virtual uint length (void) const = 0;
	virtual bool isDir (void) const = 0;

	static QPixmap loadIcon (const char *name);
};

class ArsonFileListFileItem : public ArsonFileListItem
{
public:
	ArsonFileListFileItem (void) : ArsonFileListItem() { }
	ArsonFileListFileItem (const KURL &url)
		: ArsonFileListItem(), m_url(url), m_file(url) { }

	virtual QString display (void) const { return m_url.filename(); }
	virtual QString persist (void) const { return m_url.url(); }

	virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd);
	virtual uint length (void) const;

	virtual bool isDir (void) const { return false; }

	const char *local (void) const { return m_file.path(); }
	const KURL &getURL (void) const { return m_url; }
	void setURL (const KURL &url) { m_url = url; }

	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);

private:
	ArsonNetFile m_file;
	KURL m_url;
};

class ArsonFileListDirItem : public ArsonFileListItem
{
public:
	ArsonFileListDirItem (void) : ArsonFileListItem() { }
	ArsonFileListDirItem (const QString &dn);

	virtual QString display (void) const { return m_name; }
	virtual QString persist (void) const { return m_name; }

	virtual uint length (void) const;

	virtual bool isDir (void) const { return true; }

	void setName (const QString &dn) { m_name = dn; }

	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);

private:
	QString m_name;
};

/*========================================================*/

class ArsonFileListVisiter : public ArsonListVisiter
{
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
	
	virtual bool visitFile (QListViewItem *pi, ArsonFileListFileItem *pf) { return true; }
	virtual bool visitDir (QListViewItem *pi, ArsonFileListDirItem *pf) { return true; }
};

class ArsonFileTreeVisiter : public ArsonTreeVisiter
{
	virtual bool visitFile (QListViewItem *pi, ArsonFileListFileItem *pf) { return true; }
	virtual bool visitDir (QListViewItem *pi, ArsonFileListDirItem *pf) { return true; }

protected:
	virtual bool beginBranch (QListViewItem *pi, ArsonListItem *pl);
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
};

/*========================================================*/

class ArsonProgressDlg;
class ArsonProcessMgr;
class ArsonProcessUI;
class QProgressBar;

class ArsonFileListDoc : public ArsonListDoc
{
	Q_OBJECT

public:
	ArsonFileListDoc (QWidget *parent, const char *name = NULL);
	virtual ~ArsonFileListDoc (void);

	virtual void create (void);

	virtual void addItem (ArsonListItem *ptr, ArsonLvPos *pos = NULL);
	virtual void delItem (QListViewItem *ptr);

	virtual QString propItem (void) const { return "track"; }

	virtual void buildFileFilter (ArsonFileFilter &filter) { }
	
	bool addFileItem (const KURL &url, ArsonLvPos *pos = NULL);
	bool addDirItem (const QString &dirname, ArsonLvPos *pp = NULL);

	void openMd5Dir (const QString &dir);
	void openMd5File (const KURL &url);
	void openM3uFile (const KURL &url);

	void setMaxProgress (int maxl);
	void setProgress (int length);

	ArsonFileListFileItem *currentFileItem (void) const;

protected:
	virtual ArsonFileListItem *createFileItem (const KURL &url) const;
	virtual ArsonFileListItem *createDirItem (const QString &dn) const;
	
	virtual ArsonProcessMgr *createCdWriter (ArsonProcessUI *pUI) = 0;
	virtual ArsonProgressDlg *createProgressDlg (void) = 0;
	virtual void connectTo (ArsonActionConnector &ac);
	virtual ArsonBaseWriter *createFileWriter (const KURL &url);
	virtual void addDirectory (const QString &dir, ArsonListInserter &pos);
	virtual QString progressString (int length) const;
	virtual int getMaxProgress (void) const;
	virtual Status *initStatus (QStatusBar *psb);
	virtual bool buildContextMenu (QPopupMenu *pm);
	virtual ArsonListWnd *createListWnd (void);

	virtual bool onStartElement (const QString &name,
		const QXmlAttributes &attr, ArsonListInserter &pos);
	virtual bool onEndElement (const QString &name,
		ArsonListInserter &pos);

private:
	QProgressBar *m_pProgress;	//	The progress bar (minutes)
	int m_maxProgress;			//	The REAL maximum progress value
	uint m_totalLength;			//	Total length of all items

protected slots:
	void slotDropped (QDropEvent *pd, QListViewItem *parent, QListViewItem *after);
	void slotSizeChanged (void);

public slots:
	void slotListAddDir (void);
	void slotListAdd (void);
	void slotWriteCD (void);
	void slotExportM3u (void);
	void slotFileOpen (void);		//	Bad name, should be slotFileRun or something
	void slotFileOpenWith (void);
	void slotFileView (void);
};

/*========================================================*/
#endif	/*	__DOCWIDGET_H__	*/
