/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Arson configuration code
 *
 *	by Tony Sideris	(06:18AM Aug 08, 2001)
 *================================================*/
#include "arson.h"

#include <qmultilineedit.h>
#include <qcombobox.h>
#include <qdir.h>

#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <klocale.h>

#include "_textviewer.h"

#include "process.h"
#include "konfig.h"

/*==================================*/
/*	GLOBALS
 *==================================*/

//	The default size/position of the main frame window
const QRect DEFAULT_GEOMETRY(0,0,640,480);

ArsonConfig g_CONFIG;	//	Global configuration instance

/*	This array must match the PROGRAM_ enum in the
 *	ArsonConfig class. The string specified here is
 *	the string displayed in the 1st column of the
 *	programs tab, and is the string searched for by
 *	the "auto-detect" "feature". Any non-real programs
 *	should be placed at the end, and at least the 1st
 *	non-real program name should have at least one ' '
 *	in it.
 *
 *	For the 'help' member use ' ' to seperate multiple
 *	switches (as in 'cdrdao write -h'), use '|' to specify
 *	a string of failure cases (for version guessing) like
 *	"--longhelp|--help" so that --help is used if --longhelp
 *	fails.
 */
struct {
	QString program;
	const char *help;
	QString desc;

} g_programs[ArsonConfig::_PROGRAMS_MAX] = {
	{ "cdrecord", "-help", I18N_NOOP("Burns data to CDR.") },
	{ "cdrdao", "write -h", I18N_NOOP("Burns audio, and data to CDR as well as performs CD copy.") },
	{ "mkisofs", "-help", I18N_NOOP("Builds ISO9660 filesystems for burning to CDR.") },
	{ "mpg123", "--longhelp", I18N_NOOP("Decodes mp3 files so they can be burned to CDR.") },
#ifdef OGG
	{ "ogg123", "--help", I18N_NOOP("Decodes Ogg Vorbis files so they can be burned to CDR.") },
#endif
#ifdef FLAC
	{ "flac", "--help", I18N_NOOP("Decodes, and encodes .flac audio files.") },
#endif
	{ "cdda2wav", "-help", I18N_NOOP("Reads track information from CDs, as well as rips tracks from CD (required for CD ripper).") },
	{ "cdparanoia", "-help", I18N_NOOP("Rips tracks from audio CDs (optional for CD ripper).") },
	{ "bladeenc", NULL, I18N_NOOP("MP3 encoder.") },
	{ "lame", "--longhelp|--help", I18N_NOOP("Another MP3 encoder.") },
	{ "oggenc", "--help", I18N_NOOP("Ogg Vorbis encoder.") },
	{ "readcd", "-help", I18N_NOOP("Reads data CDs into an image file.") },
	{ "shorten", "-help", I18N_NOOP("Decodes SHN files so they can be burned to CDR.") },
	{ "shnlen", "-h", I18N_NOOP("Determines track length of SHN files.") },
	{ "md5sum", "-help", I18N_NOOP("Checks the MD5 checksum of files specified in MD5 file.") },
	{ "sox", "-help", I18N_NOOP("Fixes invalid audio tracks (that are not in 44100Khz/16bps/stereo).") },
	{ "normalize", "-help", I18N_NOOP("Evens out the volumes of different audio tracks.") },
	{ "vcdxbuild", "--help", I18N_NOOP("Builds [S]VCD image suitable for burning to CDR.") },
	{ "vcdxgen", "--help", I18N_NOOP("Generates XML files required by vcdxbuild.") },
	{ "id3v2", "--help", I18N_NOOP("Adds id3v2 tag information to MP3 files.") },
	{ I18N_NOOP("CDR Initialization"), NULL, I18N_NOOP("Run when arson is started to initialize CDR devices (load kernel modules, etc), this won't be necessary for most people.") },
	{ I18N_NOOP("CDR Cleanup"), NULL, I18N_NOOP("Run when arson is shutting down to deinitialize CDR devices.") },
};

#define WRITES(var)		if ((var) != QString::null) pk->writeEntry(#var, var)
#define WRITEI(var) 	pk->writeEntry(#var, var)

#define READS(var)		var = pk->readEntry(#var, var)
#define READI(var)		var = pk->readNumEntry(#var, var)
#define READU(var)		var = pk->readUnsignedNumEntry(#var, var)

/*========================================================*/
/*	Controls program preferences
 *========================================================*/

ArsonProgramSeqs::ArsonProgramSeqs (void)
{
	memset(&m_seq, sizeof(m_seq), 0);
}

/*========================================================*/

#define PROGKEY(i)		QString("progseq-") + QString::number((i) + 1)

void ArsonProgramSeqs::save (KConfig *pk)
{
	pk->setGroup("progseq");
	
	//	Save the program sequences
	for (int index = 0; index < _PROGGRP_MAX; ++index)
		pk->writeEntry(PROGKEY(index), m_seq[index]);
}

void ArsonProgramSeqs::load (KConfig *pk)
{
	pk->setGroup("progseq");
	
	//	Load the program sequences
	for (int index = 0; index < _PROGGRP_MAX; ++index)
		m_seq[index] = pk->readNumEntry(PROGKEY(index), 0);
}

/*========================================================*/
/*	Base config class implemenatation
 *========================================================*/

bool ArsonConfigBase::load (KConfig *pk)
{
	pk->setGroup(m_name);
	return true;
}

void ArsonConfigBase::save (KConfig *pk, bool finalize)
{
	pk->setGroup(m_name);
}

/*========================================================*/

bool ArsonFeatureConfig::load (KConfig *pk)
{
	ArsonConfigBase::load(pk);
	READI(m_flags);
	return true;
}


void ArsonFeatureConfig::save (KConfig *pk, bool finalize)
{
	ArsonConfigBase::save(pk, finalize);

	WRITEI(m_flags);
}

/*========================================================*/
/*	ArsonConfig class implementation
 *========================================================*/

ArsonConfig::ArsonConfig (void)
	: m_nCdLenMin(74),
	m_nCdLenMB(650),
	m_nSpeed(2),
	m_flags(flagMd5Verify | flagMd5Reset),
	m_startDoc(0),
	m_niceLevel(0),
	m_ripper(this)
{
	//	Nothing...
}

/*========================================================*/

const char *ArsonConfig::programHelpSwitch (int index)
{
	return g_programs[index].help;
}

QString ArsonConfig::programName (int index)
{
	if (index < PROGRAM_CDRINIT)
		return g_programs[index].program;
	
	return i18n(g_programs[index].program);
}

QString ArsonConfig::programDesc (int index)
{
	return i18n(g_programs[index].desc);
}

/*========================================================*/
/*	Load and save configuration
 *========================================================*/

#define PROGRAM_SEP		","

bool ArsonConfig::load (void)
{
	int sb = 0, index;
	QString temp;
	KConfig *pk = kapp->config();

	pk->setGroup("cfg");
	READS(m_strDevice);
	READS(m_strDriver);
	READS(m_strSrcDriver);
	READU(m_flags);
	READI(m_nCdLenMin);
	READI(m_nCdLenMB);
	READI(m_nSpeed);
	READI(m_startDoc);
	READI(m_niceLevel);
	READS(m_tempDirectory);

	m_geometry = pk->readRectEntry(ACFGKEY(m_geometry), &DEFAULT_GEOMETRY);

	m_seq.load(pk);
	
	//	Read in the programs
	pk->setGroup("programs");
	for (index = 0; index < _PROGRAMS_MAX; index++)
	{
		temp = pk->readEntry(programName(index));

		QStringList sl = QStringList::split(PROGRAM_SEP, temp, TRUE);

		if (!sl.isEmpty())
		{
			Program &prog = m_programs[index];

			prog.m_program = sl[0];
			sl.remove(sl.begin());

			prog.m_params = sl.join(PROGRAM_SEP);
		}
	}

	/*
	 *	Autodetect the necessary programs if none exist
	 */
	if (!validDeviceScannersExist())
	{
		autoDetectPrograms();
		
		if (!validDeviceScannersExist())
		{
			throw ArsonStartupError(
				i18n("cdrecord and cdrdao were not found, either of these programs\n") +
				i18n("are used to scan the SCSI bus for CD[RW] devices, so at least one\n") +
				i18n("of them must be installed. If one of them is installed, goto the\n") +
				i18n("Programs tab of the configuration dialog and specify the path.\n") +
				i18n("Otherwise install one or both of these programs and configure your\n") +
				i18n("system to use them properly (see the CD-Writing HOWTO).\n\n") +
				i18n("If you are only using arson as a CD ripper, then this should not\n") +
				i18n("concern you. Check off 'Do not ask again.', and use the\n") +
				i18n("Cooked IOCTL interface (ripper configuration tab)."),
				"noscanners");
		}
	}

	/*	If scanbus fails, initialize the
	 *	CDR and try again.
	 */
	try {
		m_devs.load(pk);
		sb = m_devs.scsiCount();
	}
	catch (ArsonStartupError &err)
	{
		/*
		 *	DO NOT REPORT ERROR - CDR init and try again
		 *	if THAT fails the caller should be reporting
		 *	the error.
		 */
		if (cdrInit(true))
		{
			m_private |= privateInitCdr;
			sb = m_devs.scanbus();
		}
	}

	m_ripper.load(pk);
	return true;
}

/*========================================================*/

void ArsonConfig::save (bool finalize)
{
	int index;
	KConfig *pk = kapp->config();

	pk->setGroup("cfg");
	WRITES(m_strDevice);
	WRITES(m_strDriver);
	WRITES(m_strSrcDriver);
	WRITEI(m_flags);
	WRITEI(m_nSpeed);
	WRITEI(m_nCdLenMin);
	WRITEI(m_nCdLenMB);
	WRITEI(m_startDoc);
	WRITEI(m_niceLevel);
	WRITES(m_tempDirectory);
	pk->writeEntry(ACFGKEY(m_geometry), m_geometry);

	m_seq.save(pk);

	//	Write the program configuration
	pk->setGroup("programs");
	for (index = 0; index < _PROGRAMS_MAX; ++index)
	{
		QStringList sl;
		sl << m_programs[index].m_program
		   << m_programs[index].m_params;

		QString qs = sl.join(PROGRAM_SEP);

		pk->writeEntry(programName(index), qs);
	}

	m_devs.save(pk);
	m_ripper.save(pk, finalize);

	if (finalize && (m_private & privateInitCdr))
		cdrInit(false);
}

/*========================================================*/
/*	Simple iterators
 *========================================================*/

const ArsonConfig::Program *ArsonConfig::program (int index) const
{
	return (index < _PROGRAMS_MAX) ? &(m_programs[index]) : NULL;
}

ArsonConfig::Program *ArsonConfig::program (int index)
{
	return (index < _PROGRAMS_MAX) ? &(m_programs[index]) : NULL;
}

/*========================================================*/
/*	Some users who have CDR-related kernel software
 *	compiled as modules, and load these modules as needed,
 *	have scripts to load the neseccary modules... these
 *	functions execute these scripts if no devices are
 *	available.
 *========================================================*/

bool ArsonConfig::cdrInit (bool init)
{
	ArsonUtilityProcess proc (ACONFIG,
		KProcess::NoCommunication,
		QString::null,
		init ? PROGRAM_CDRINIT : PROGRAM_CDRFIN);

#ifdef ARSON_KDE3
	if (!proc.args().isEmpty())
#else
	if (!proc.args()->isEmpty())
#endif	//	ARSON_KDE_3
	{
		return proc.execute();
	}

	return true;
}

/*========================================================*/
/*	Search the filesystem for neseccary programs, so the
 *	user doesn't need to manually configure the paths...
 *========================================================*/

void ArsonConfig::autoDetectPrograms (const QString &xtra)
{
	for (int index = 0; index < PROGRAM_CDRINIT; index++)
	{
		const QString name = programName(index);

		if (name.find(' ') != -1)
			continue;

		m_programs[index].m_program = detectProgram(name, xtra);
	}
}

QString ArsonConfig::detectProgram (const char *name, const QString &xtra)
{
	QStringList::Iterator it, end;
	QString xpath = xtra;

	if (xpath == QString::null)
		xpath = extraProgramPaths();

	//	Path list built from $PATH and extra paths option
	QStringList paths = QStringList::split(":", xpath);
	paths += QStringList::split(":", getenv("PATH"));

	//	Check each path in the path list
	for (it = paths.begin(), end = paths.end(); it != end; ++it)
	{
		const QString path = QDir(*it).absFilePath(name);

		if (arsonIsExecutable(path))
			return path;
	}

	return QString::null;
}

/*========================================================*/

bool ArsonConfig::validDeviceScannersExist (void) const
{
	return (program(PROGRAM_CDRECORD)->valid() ||
		program(PROGRAM_CDRDAO)->valid());
}

/*========================================================*/

void ArsonConfig::showMissingPrograms (void)
{
	QStringList missing;

	for (int index = 0; index < _PROGRAMS_MAX; ++index)
		if (!program(index)->valid())
			missing.append(
				programName(index) +
				i18n(" - ") +
				programDesc(index));

	if (!missing.isEmpty())
		arsonErrorMsg(
			i18n("The following worker programs were not detected. Whether or not they are required or not depends on what you plan on using arson for (burning CDs requires somethings, while ripping audio requires others). The following was not found:\n\n") +
			missing.join("\n"));
}

/*========================================================*/
/*	Ripper settings implementation
 *========================================================*/

ArsonConfig::RipperCfg::RipperCfg (const ArsonConfig *cfg)
	: ArsonFeatureConfig("ripper"),
	m_outputFormat(1),
	m_cdiFormat("%n %a - %t"),
	m_quality(i18n("CD Quality")),
	m_defaultComment(i18n("Ripped by arson v%v")),
	m_strOutDir(QDir::home().canonicalPath()),
	m_cfg(cfg)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonConfig::RipperCfg::load (KConfig *pk)
{
	const bool res = ArsonFeatureConfig::load(pk);

	READI(m_outputFormat);
	READS(m_strOutDir);
	READS(m_cdiFormat);
	READS(m_email);
	READS(m_srcdev);
	READS(m_quality);
	READS(m_defaultComment);

	m_lookups.load(pk);
	return res;
}

/*========================================================*/

void ArsonConfig::RipperCfg::save (KConfig *pk, bool finalize)
{
	ArsonFeatureConfig::save(pk, finalize);

	WRITEI(m_outputFormat);
	WRITES(m_strOutDir);
	WRITES(m_cdiFormat);
	WRITES(m_email);
	WRITES(m_srcdev);
	WRITES(m_quality);
	WRITES(m_defaultComment);

	m_lookups.save(pk);
}

/*========================================================*/

#define ARSON_CDIFORMATS_MRU		"fmtmru"
#define ARSON_MRU_SEP				'\t'
#define ARSON_MRU_MAX				16

void ArsonConfig::RipperCfg::setCdiFormat (const QString &fmt)
{
	QStringList sl = recentCdiFormats();

	if (sl.find(fmt) == sl.end())
	{
		KConfig *pk = kapp->config();

		sl.append(fmt);

		while (sl.count() > ARSON_MRU_MAX)
			sl.remove(sl.begin());
		
		pk->setGroup("ripper");
		pk->writeEntry(ARSON_CDIFORMATS_MRU, sl, ARSON_MRU_SEP);
	}

	m_cdiFormat = fmt;
}

QStringList ArsonConfig::RipperCfg::recentCdiFormats (void) const
{
	KConfig *pk = kapp->config();
	pk->setGroup("ripper");
	return pk->readListEntry(ARSON_CDIFORMATS_MRU, ARSON_MRU_SEP);
}

/*========================================================*/
/*	A class for gathering, and displaying
 *	program --help messages.
 *========================================================*/

ArsonProgramHelp::ArsonProgramHelp (int program)
{
	if (program != ArsonConfig::PROGRAM_UNKNOWN)
		setProgram(program);
}

/*========================================================*/
/*	The help process
 *========================================================*/

class arsonProgramHelpProcess : public ArsonUtilityProcess
{
public:
	arsonProgramHelpProcess (int program)
		: ArsonUtilityProcess(ACONFIG,
			Communication(Stderr|Stdout),
			ArsonConfig::programName(program),
			program)
	{
	}

	const QString &help (void) const { return m_help; }

	virtual bool execute (void)
	{
		ArsonUtilityProcess::execute();
		return !m_help.isEmpty();
	}
	
private:
	virtual void output (const QString &str, bool error)
	{
		Trace("(%d) %s\n", error, str.latin1());
		m_help.append(str + "\n");
	}

	QString m_help;
};

/*========================================================*/
/*	The dialog
 *========================================================*/

class arsonTextViewer : public ArsonTextViewerBase
{
public:
	arsonTextViewer (const QString &str, QWidget *parent)
		: ArsonTextViewerBase(parent, NULL, true)
	{
		text->setFont(KGlobalSettings::fixedFont());
		text->setWordWrap(QMultiLineEdit::NoWrap);
		text->setText(str);
	}
};

/*========================================================*/
/*	Do it till it works.
 *========================================================*/

int ArsonProgramHelp::setProgram (int program)
{
	if (program > ArsonConfig::PROGRAM_UNKNOWN &&
		program < ArsonConfig::PROGRAM_CDRINIT)
	{
		const char *switches = ArsonConfig::programHelpSwitch(program);
		const QStringList sl = QStringList::split(QString("|"), switches);

		m_program = program;
		
		//	For program's that display help with no args
		if (!switches)
		{
			arsonProgramHelpProcess proc (program);

			if (proc.execute())
				m_help = proc.help();
		}

		//	All others
		for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
			 it != end; ++it)
		{
			arsonProgramHelpProcess proc (program);

			proc.appendArgList(QStringList::split(QString(" "), (*it)));

			if (proc.execute())
			{
				m_help = proc.help();

				if (proc.successful())
					break;
			}
		}
	}
	
	return (m_help.isEmpty() ? -1 : (int) m_help.length());
}

/*========================================================*/

void ArsonProgramHelp::show (QWidget *parent) const
{
	arsonTextViewer dlg (m_help, parent ? parent : kapp->mainWidget());

	dlg.setCaption(i18n("Help for ") + ArsonConfig::programName(m_program));
	dlg.exec();
}

/*========================================================*/
